unit PrintUtil;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  System.UITypes, System.Generics.Collections,
  Windows, Graphics, Dialogs, Printers, SysUtils, Forms, Classes, Math, Controls;


type
  TActionRect = interface
    procedure DoAction(ScreenPosition: TPoint);
    function  getRect(): TRect;
    procedure setRect(x1, y1, width, height: integer);
    function  getPaintBoxOwner(): TObject;
    procedure setPaintBoxOwner(PaintBox: TObject);
  end;



  TActionRectBase = class(TInterfacedObject, TActionRect)
    Rect: TRect;
    PaintBoxOwner: TObject;
    function  getRect(): TRect;
    procedure setRect(x1, y1, width, height: integer);
    function  getPaintBoxOwner(): TObject;
    procedure setPaintBoxOwner(PaintBox: TObject);
    procedure DoAction(ScreenPosition: TPoint); virtual; abstract;
  end;



  TZoomCanvas = class(TObject)
  private
    FCanvas: TCanvas;
    FDpiX: integer;
    FDpiY: integer;
    FZoom: Double;
    FMinYVisible: integer;
    FMaxYVisible: integer;
    Offset: Integer;
    FFontSize: Integer;
    FFontRotation: Integer;
    FFontBold: boolean;

    procedure SetBrushColor(const Value: TColor);
    procedure SetPenColor(const Value: TColor);
    procedure SetFontColor(const Value: TColor);

    function GetBrushColor: TColor;
    function GetFontColor: TColor;
    function GetPenColor: TColor;
    function ZoomYCoord(Value: integer): integer;
    procedure SetFontSize(const Value: Integer);
    function UnZoomXCoord(Value: integer): integer;
    function UnZoomYCoord(Value: integer): integer;
    procedure SetFontBold(const Value: boolean);
  protected
    function IsVisible(yPos: integer): boolean;
  public
    ActionRects: TList<TActionRect>;
  public
    constructor Create(Canvas: TCanvas; DpiX: integer; DpiY: integer; Zoom: Double); overload;
    constructor Create(Canvas: TCanvas; Zoom: Double); overload;

    destructor Destroy(); override;

    function DeConvertYCoord(Value: integer): integer;
    function DeConvertXCoord(Value: integer): integer;
    function ConvertXCoord(Value: integer): integer;
    function ConvertYCoord(Value: integer): integer;

    function TextExtent(S: String): TSize;

    procedure DrawArc(x, y, rad: integer);
    procedure DrawQuadrat(x, y, rad: integer);

    procedure FillRect(Rect: TRect);

    procedure TextOut(x, y: integer; S: String; ignoreVisibilityCheck: boolean = false);

    procedure MoveTo(x, y: integer);
    procedure LineTo(x, y: integer; ignoreVisibilityCheck: boolean = false);

    procedure AddActionRect(ActionRect: TActionRect);

    property BrushColor: TColor read GetBrushColor write SetBrushColor;
    property PenColor: TColor read GetPenColor write SetPenColor;
    property FontColor: TColor read GetFontColor write SetFontColor;
    property FontHeight: Integer read FFontSize write SetFontSize;
    property FontBold: boolean read FFontBold write SetFontBold;
    property FontRotation: Integer read FFontRotation write FFontRotation;

  end;

  TPrintFunction = function (Canvas: TZoomCanvas; ForPrint: boolean): TSize of Object;

  //: Einfaches Druck-Kapslung mit mglichkeit der Auswahl Drucker + Ausrichtung + Format
  TSimplePrinter = class(TComponent)
  private
    FFont: TFont;
    FDpiX: integer;
    FDpiY: integer;
    MaxX: integer;
    MaxY: integer;
    OffsetX: integer;
    OffsetY: integer;
    RandX, RandTop, RandBottom: Integer;
    FZoom: Double;
    Canvas: TZoomCanvas;
    procedure PrintHeader(Title: String);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy(); override;
    function BeginPrint(Titel: String; LandScape: boolean): boolean;
    procedure EndPrint();
    procedure NewPage();
    property Font: TFont read FFont write FFont;
    property Zoom: Double read FZoom write FZoom;

    procedure print(PrintFunction: TPrintFunction; NewPageAtStart: boolean; Title: String);
  end;



implementation



{ TSimplePrinter }

function TSimplePrinter.BeginPrint(Titel: String; LandScape: boolean): boolean;
var
  PrintDialog: TPrintDialog;
begin
  PrintDialog := TPrintDialog.Create(Application.MainForm);
  Result := PrintDialog.Execute;
  if Result then
  begin
    Printer.Title := Titel;
    if LandScape then
    begin
      Printer.Orientation := poLandscape;
    end
    else
    begin
      Printer.Orientation := poPortrait;
    end;
    Printer.BeginDoc;
    FDpiX := GetDeviceCaps(Printer.Canvas.Handle, LOGPIXELSX);
    FDpiY := GetDeviceCaps(Printer.Canvas.Handle, LOGPIXELSY);
    //Druckrnder kompensieren
    OffsetX := GetDeviceCaps(Printer.Canvas.Handle, PHYSICALOFFSETX);
    OffsetY := GetDeviceCaps(Printer.Canvas.Handle, PHYSICALOFFSETY);

    printer.Canvas.Font.Name := Font.Name;
    printer.Canvas.Font.Height := Round(Font.Height * 7);
    printer.Canvas.Font.Height := Font.Height;
    Canvas := TZoomCanvas.Create(printer.Canvas, FDpiX, FDpiY, Zoom);

    RandX := Canvas.ConvertXCoord(50);
    RandTop := Canvas.ConvertYCoord(100);
    RandBottom := Canvas.ConvertYCoord(50);

    MaxX := GetDeviceCaps(Printer.Canvas.Handle, PHYSICALWIDTH) - (2 * RandX);
    MaxY := GetDeviceCaps(Printer.Canvas.Handle, PHYSICALHEIGHT) - (RandTop + RandBottom);

    SetMapMode(Printer.Canvas.Handle, MM_TEXT);
    SetViewportOrgEx(Printer.Canvas.Handle, -OffsetX + RandX, -OffsetY + RandTop, nil);


  end;
  PrintDialog.Free;
end;


constructor TSimplePrinter.Create(AOwner: TComponent);
begin
  inherited;
  FFont := TFont.Create;
  FZoom := 1.0;
end;

destructor TSimplePrinter.Destroy;
begin
  FFont.Free;
  inherited;
end;

procedure TSimplePrinter.EndPrint;
begin
  FreeAndNil(Canvas);
  Printer.EndDoc;
end;

procedure TSimplePrinter.NewPage;
begin
  Printer.NewPage();
  SetMapMode(Printer.Canvas.Handle, MM_TEXT);
  SetViewportOrgEx(Printer.Canvas.Handle, -OffsetX + RandX, -OffsetY + RandTop, nil);
end;

procedure TSimplePrinter.PrintHeader(Title: String);
var
  OldFontHeight: integer;
begin
  OldFontHeight := Canvas.FontHeight;
  Canvas.FontHeight := 30;
  Canvas.TextOut(0, Canvas.Offset - 60, Title, true);
  Canvas.MoveTo(0, Canvas.Offset - 20);
  Canvas.LineTo(Canvas.DeConvertXCoord(MaxX), Canvas.Offset - 20, true);
  Canvas.FontHeight := OldFontHeight;
end;

procedure TSimplePrinter.print(PrintFunction: TPrintFunction; NewPageAtStart: boolean; Title: String);
var
  Size: TSize;
  MaxSize: integer;
begin
  Canvas.FMinYVisible := 0;
  Canvas.Offset := 0;
  MaxSize := Canvas.DeConvertYCoord(MaxY);
  Canvas.FMaxYVisible := MaxSize;

  if NewPageAtStart then
  begin
    NewPage;
  end;

  while(True) do
  begin
    PrintHeader(Title);
    Size := PrintFunction(Canvas, True);

    if Size.cy < Canvas.FMaxYVisible then
    begin
      break;
    end;

    NewPage;

    Canvas.Offset := Canvas.Offset + MaxSize;

    Canvas.FMinYVisible := Canvas.FMaxYVisible;
    Canvas.FMaxYVisible := Canvas.FMaxYVisible + MaxSize;

  end;
end;


{ TZoomCanvas }

constructor TZoomCanvas.Create(Canvas: TCanvas; DpiX, DpiY: integer; Zoom: Double);
begin
  inherited Create;
  FCanvas := Canvas;
  FDpiX := DpiX;
  FDpiY := DpiY;
  FZoom := Zoom;
  FFontSize := -12;
  FFontBold := False;
  FMinYVisible := 0;
  FMaxYVisible := MAXLONG;
  ActionRects := TList<TActionRect>.Create;
end;

constructor TZoomCanvas.Create(Canvas: TCanvas; Zoom: Double);
begin
  Create(Canvas, Screen.PixelsPerInch, Screen.PixelsPerInch, Zoom);
end;


procedure TZoomCanvas.FillRect(Rect: TRect);
begin
  if IsVisible(Rect.Top) then
  begin
    Rect.Left := ConvertXCoord(Rect.Left);
    Rect.Right := ConvertXCoord(Rect.Right);
    Rect.Top := ConvertYCoord(Rect.Top);
    Rect.Bottom := ConvertYCoord(Rect.Bottom);

    FCanvas.FillRect(Rect);
  end;
end;

function TZoomCanvas.GetBrushColor: TColor;
begin
  Result := FCanvas.Brush.Color;
end;

function TZoomCanvas.GetFontColor: TColor;
begin
  Result := FCanvas.Font.Color;
end;

function TZoomCanvas.GetPenColor: TColor;
begin
  Result := FCanvas.Pen.Color;
end;

function TZoomCanvas.IsVisible(yPos: integer): boolean;
begin
  if (yPos >= FMinYVisible) and (yPos <= FMaxYVisible) then
    Result := True
  else
    Result := False;
end;

procedure TZoomCanvas.LineTo(x, y: integer; ignoreVisibilityCheck: boolean = false);
begin
  if IsVisible(y) or ignoreVisibilityCheck then
  begin
    x := ConvertXCoord(x);
    y := ConvertYCoord(y);

    FCanvas.LineTo(x, y);
  end
  else
  begin
    x := ConvertXCoord(x);
    y := ConvertYCoord(y);

    FCanvas.MoveTo(x, y);
  end;
end;

procedure TZoomCanvas.MoveTo(x, y: integer);
begin
  x := ConvertXCoord(x);
  y := ConvertYCoord(y);

  FCanvas.MoveTo(x, y);

end;

procedure TZoomCanvas.SetBrushColor(const Value: TColor);
begin
  FCanvas.Brush.Color := Value;
end;

procedure TZoomCanvas.SetFontBold(const Value: boolean);
begin
  FFontBold := Value;
end;

procedure TZoomCanvas.SetFontColor(const Value: TColor);
begin
  FCanvas.Font.Color := Value;
end;

procedure TZoomCanvas.SetFontSize(const Value: Integer);
begin
  FFontSize := Value;
end;

procedure TZoomCanvas.SetPenColor(const Value: TColor);
begin
  FCanvas.Pen.Color := Value;
end;

function TZoomCanvas.TextExtent(S: String): TSize;
var
  OldHeight: Integer;
begin
  OldHeight := FCanvas.Font.Height;
  FCanvas.Font.Height := ZoomYCoord(FontHeight);
  FCanvas.Font.Style := [];
  if FontBold then
  begin
    FCanvas.Font.Style := [fsBold];
  end;

  Result := FCanvas.TextExtent(S);
  FCanvas.Font.Height := OldHeight;

  Result.cx := UnZoomXCoord(Result.cx);
  Result.cy := UnZoomXCoord(Result.cy);
end;


procedure TZoomCanvas.TextOut(x, y: integer; S: String; ignoreVisibilityCheck: boolean = false);
var
  OldHeight: Integer;
  OldBackMode: Integer;
begin
  if IsVisible(y) or ignoreVisibilityCheck then
  begin
    OldBackMode := GetBkMode(FCanvas.Handle);
    SetBkMode(FCanvas.Handle, TRANSPARENT);
    x := ConvertXCoord(x);
    y := ConvertYCoord(y);
    OldHeight := FCanvas.Font.Height;
    if FontRotation <> 0 then
    begin
      FCanvas.Font.Orientation := -FontRotation * 10;
    end;
    FCanvas.Font.Height := ZoomYCoord(FontHeight);
    FCanvas.Font.Style := [];
    if FontBold then
    begin
      FCanvas.Font.Style := [fsBold];
    end;

    FCanvas.TextOut(x, y, S);
    FCanvas.Font.Height := OldHeight;
    FCanvas.Font.Orientation := 0;
    SetBkMode(FCanvas.Handle, OldBackMode);
  end;
end;

procedure TZoomCanvas.AddActionRect(ActionRect: TActionRect);
begin
  ActionRects.Add(ActionRect);
end;

function TZoomCanvas.ConvertXCoord(Value: integer): integer;
begin
  Result := Round(Value * 1.0 * FDpiX / 96.0 * FZoom);
end;

function TZoomCanvas.ConvertYCoord(Value: integer): integer;
begin
  Value := Value - Offset;
  Result := ZoomYCoord(Value);
end;

function TZoomCanvas.ZoomYCoord(Value: integer): integer;
begin
  Result := Round(Value * 1.0 * FDpiY / 96.0 * FZoom);
end;

function TZoomCanvas.UnZoomYCoord(Value: integer): integer;
begin
  Result := Round(Value * 1.0 / FDpiY * 96.0 / FZoom);
end;


function TZoomCanvas.DeConvertYCoord(Value: integer): integer;
begin
  Result := UnZoomYCoord(Value);
  Result := Result + Offset;
end;

destructor TZoomCanvas.Destroy;
begin
  ActionRects.Free;
  inherited;
end;

procedure TZoomCanvas.DrawArc(x, y, rad: integer);
var
  x1, x2, y1, y2: integer;
begin
  if IsVisible(y) then
  begin
    x1 := ConvertXCoord(x - rad);
    x2 := ConvertXCoord(x + rad);
    y1 := ConvertYCoord(y - rad);
    y2 := ConvertYCoord(y + rad);

    FCanvas.Ellipse(x1, y1, x2, y2);
  end;
end;

procedure TZoomCanvas.DrawQuadrat(x, y, rad: integer);
var
  x1, x2, y1, y2: integer;
begin
  if IsVisible(y) then
  begin
    x1 := ConvertXCoord(x - rad);
    x2 := ConvertXCoord(x + rad);
    y1 := ConvertYCoord(y - rad);
    y2 := ConvertYCoord(y + rad);

    FCanvas.Rectangle(x1, y1, x2, y2);
  end;
end;


function TZoomCanvas.UnZoomXCoord(Value: integer): integer;
begin
  Result := Round(Value * 1.0 / FDpiX * 96.0 / FZoom);
end;

function TZoomCanvas.DeConvertXCoord(Value: integer): integer;
begin
  Result := UnZoomXCoord(Value);
end;


{ TActionRectBase }

function TActionRectBase.getPaintBoxOwner: TObject;
begin
  Result := PaintBoxOwner;
end;

function TActionRectBase.getRect: TRect;
begin
  Result := Rect;
end;


procedure TActionRectBase.setPaintBoxOwner(PaintBox: TObject);
begin
  Self.PaintBoxOwner := PaintBox;
end;

procedure TActionRectBase.setRect(x1, y1, width, height: integer);
begin
  Rect.Left := x1;
  Rect.Top := y1;
  Rect.Right := x1 + width;
  Rect.Bottom := y1 + height;
end;

end.
