unit PlanUtils;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Forms, Windows, Messages, SysUtils, Variants, Classes, Graphics, Contnrs,
  Math, SyncObjs, Vcl.Controls, System.DateUtils, ShlObj, ComObj, ShellAPI, Dialogs,
  Vcl.StdCtrls;

const
  cNewLine = #0013#0010;


type
  MyDouble = Double;
  PMyDouble = ^MyDouble;


function MyFormatDate(Date: TDateTime): string;
function MyFormatDateShort(Date: TDateTime): string;
function MyFormatDateTime(Date: TDateTime): string;
function MyFormatTime(Date: TDateTime): string;
function MyFormatFloat(Value: MyDouble): String;
function MyFormatFloatShort(Value: MyDouble): String;
function MyFormatInt(Value: Int64): String;
function KalenderWoche(Date: TDateTime): integer;
function InternalWeekNumber(Date: TDateTime): integer;
function MyFormatClickDiff(Tick: DWORD): String;
procedure TraceString(Value: string);

function MyStrToFloat(const Value: String): MyDouble;

function StartWaitCursor(NewCursor: TCursor = crHourGlass): IInterface;

function getAppLocalPath(): String;

function DecodeFileName(const Value: String): String;
function EncodeFileName(const Value: String): String;
function MakeValidFileName(const Value: String): String;

procedure FilesFromDir(Result: TStrings; const Mask: String);

function GetFileVersionString(const FileName: String): string;

function MondayBefore(Date: TDateTime): TDateTime;

procedure TestDateRoutines();

function FixDateTime(Value: TDateTime): TDateTime;

procedure FixControls(Win: TWinControl);

procedure FixFormSize(Win: TWinControl; Width, Height: integer);

function GetListComboValue(ComboBox: TComboBox): String;

procedure SetListComboValue(ComboBox: TComboBox; Value: String);

function DecToRom(Dec: LongInt): String;

implementation

// Die ungenauigkeit beim Addieren von Fliekommazahlen fhrt manchmal dazu, das 1.1.2001 17:00 + 4:00 nicht genau 1.1.2001 21:00 ist
// decode das ganze und encode es wieder
function FixDateTime(Value: TDateTime): TDateTime;
var
  Year, Month, Day, Hour, Minute, Second, Milli: Word;
begin
  DecodeDateTime(Value, Year, Month, Day, Hour, Minute, Second, Milli);
  Result := EncodeDateTime(Year, Month, Day, Hour, Minute, Second, Milli);
end;



function MyStrToFloat(const Value: String): MyDouble;
var
  FormatSettings: TFormatSettings;
begin
  FormatSettings.DecimalSeparator := '.';
  Result := StrToFloat(Value, FormatSettings);
end;

function KalenderWoche(Date: TDateTime): integer;
begin
  Result := WeekOf(Date);
end;

function InternalWeekNumber(Date: TDateTime): integer;
begin
  Result := (Trunc(Date)-2) div 7;
end;


function MondayBefore(Date: TDateTime): TDateTime;
begin
  Result := (((Trunc(Date)-2) div 7) * 7) + 2;
end;


function MyFormatClickDiff(Tick: DWORD): String;
begin
  Tick := Tick div 1000;
  if Tick >= 60 then
  begin
    Result := IntToStr(Tick div 60) + ' Min ';
  end;

  Result := Result + IntToStr(Tick mod 60) + ' Sek';
end;


function MyFormatFloat(Value: MyDouble): String;
begin
  if (Value < 10) and (Value <> 0.0) then
    Result := Format('%1.2n', [Value])
  else
    Result := Format('%1.0n', [Value]);
end;


function MyFormatFloatShort(Value: MyDouble): String;
begin
  if Value >= 1000000.0 then
  begin
    if Value >= 10000000.0 then
    begin
      Result := Format('%1.0n', [Value / 1000000.0]) + ' Mio';
    end
    else
    begin
      Result := Format('%1.1n', [Value / 1000000.0]) + ' Mio';
    end;
  end
  else
  if Value >= 100000.0 then
  begin
    Result := Format('%1.0n', [Value / 1000.0]) + ' T';
  end
  else
  if (Value < 10) and (Value <> 0.0) then
    Result := Format('%1.2n', [Value])
  else
    Result := Format('%1.0n', [Value]);
end;


function MyFormatInt(Value: Int64): String;
begin
  Result := Format('%3.0n', [(Value * 1.0)]);
end;


function MyFormatDate(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'ddd dd.mm.yyyy', Date);
end;

function MyFormatDateShort(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'dd.mm', Date);
end;


function MyFormatDateTime(Date: TDateTime): string;
begin
  Result := MyFormatDate(Date);

  Result := Result + ' ' + MyFormatTime(Date);
end;


function MyFormatTime(Date: TDateTime): string;
var
  TimeString: string;
begin
  DateTimeToString(TimeString, 'hh:mm', Date);

  Result := TimeString;
end;

var
  LogCriticalSection: TCriticalSection;
  LogFileName: String;

function getLogFileName(): String;
var
  Buffer: array[0..2001] of char;
  F: TextFile;
begin
  GetTempPath(2000, Buffer);
  Result := Buffer;
  if '\' <> Copy(Result, Length(Result), 1) then
  begin
    Result := Result + '\';
  end;
  Result := Result + 'Andi-Generator.log';

  AssignFile(f, Result);
  try
    if not FileExists(Result) then
    begin
      Rewrite(f);
    end
    else
    begin
      Append(f);
    end;
    Writeln(f, '');
    Writeln(f, 'Programmstart: ' + DateTimeToStr(now));
  finally
    CloseFile(f);
  end;
end;


procedure Log(Value: String);
var
  F: TextFile;
begin
  LogCriticalSection.Enter;
  try
    try
      if LogFileName = '' then
      begin
        LogFileName := getLogFileName();
      end;

      AssignFile(f, LogFileName);
      try
         Append(f);
         Writeln(f, Value);
      finally
         CloseFile(f);
      end;
    Except
    end;
  finally
    LogCriticalSection.Leave;
  end;
end;



procedure TraceString(Value: string);
begin
  OutputDebugString(PChar(Value));
  Log(Value);
end;





function DateFromString(dateString: String): TDateTime;
var
  year, month, day: Word;
begin
  day := StrToInt(Copy(dateString, 1, 2));
  month := StrToInt(Copy(dateString, 4, 2));
  year := StrToInt(Copy(dateString, 7, 4));
  Result := EncodeDate(year, month, day);
end;

function TimeFromString(dateString: String): TDateTime;
var
  hour, min: Word;
begin
  hour := StrToInt(Copy(dateString, 1, 2));
  min := StrToInt(Copy(dateString, 4, 2));
  Result := EncodeTime(hour, min, 0, 0);
end;



type
  TWaitCursorHelperObject = class( TInterfacedObject)
    OldCursor: TCursor;
    constructor Create( NewCursor: TCursor);
    destructor  Destroy; override;
  end;


{ TWaitCursorHelperObject }

constructor TWaitCursorHelperObject.Create( NewCursor: TCursor);
begin
  inherited Create;
  OldCursor := crDefault;
  try
    OldCursor := Screen.Cursor;
    Screen.Cursor := NewCursor;
  except
  end;
end;

destructor TWaitCursorHelperObject.Destroy;
begin
  try
    Screen.Cursor := OldCursor;
  except
  end;
  inherited;
end;

function StartWaitCursor( NewCursor: TCursor): IInterface;
begin
  Result := TWaitCursorHelperObject.Create( NewCursor);
end;


function getAppLocalPath(): String;
var
  ItemIDList: PItemIDList;
  FilePath: array[0..MAX_PATH] of WideChar;
begin
  { Get path of selected location }
  SHGetFolderLocation(0, CSIDL_LOCAL_APPDATA, 0, 0, ItemIDList);

  SHGetPathFromIDListW(ItemIDList, FilePath);
  result := FilePath;

end;

function EncodeFileName(const Value: String): String;
var
  i: integer;
  Digit: Char;
begin
  Result := '';
	for i:=1 to Length(Value) do
  begin
    Digit := Value[i];

		if (Digit = '#') or CharInSet(Digit, ['<','>','"','[',']','/','\','?','*',':','|']) then
    begin
      Result := Result + '#';
      Result := Result + Format('%3.3d', [ord(Digit)]);
    end
		else
    begin
      Result := Result + Digit;
    end;
  end;
end;


function MakeValidFileName(const Value: String): String;
var
  i: integer;
  Digit: Char;
begin
  Result := '';
	for i:=1 to Length(Value) do
  begin
    Digit := Value[i];

		if (Digit = '#') or CharInSet(Digit, ['<','>','"','[',']','/','\','?','*',':','|']) then
    begin
      Result := Result + '-';
    end
		else
    begin
      Result := Result + Digit;
    end;
  end;
end;


function DecodeFileName(const Value: String): String;
var
  i: integer;
  Digit: Char;
  DigitInt: Integer;
  Verarbeitet: boolean;
begin
  i := 1;
  Result := '';
  while i<=Length(Value) do
  begin
    Verarbeitet := False;
    if Value[i] = '#' then
    begin
      if i+3 <= Length(Value) then
      begin
        try
          DigitInt := StrToInt(Copy(Value, i+1, 3));
          Digit := char(DigitInt);
          Result := Result + Digit;
          Verarbeitet := True;
          Inc(i, 4);
        Except
        end;
      end;
    end;
    if not Verarbeitet then
    begin
      Result := Result + Value[i];
      Inc(i, 1);
    end;
  end;
end;


procedure FilesFromDir(Result: TStrings; const Mask: String);
var
  SearchRec: TSearchRec;
  stat: Integer;
begin
  Result.Clear;

  stat := FindFirst(Mask, faNormal, SearchRec);
  while stat = 0 do
  begin
    Result.Add(SearchRec.Name);
    stat := FindNext(SearchRec);
  end;
  FindClose(SearchRec);
end;



function GetFileVersionString(const FileName: String): string;
var
  InfoResSize, Dummy: DWORD;
  VersionInfo, Info: Pointer;
  VersionInfoSize: UINT;
  InfoPath: string;
begin
  result := '';

  InfoResSize := GetFileVersionInfoSize(PChar(FileName), Dummy);
  if InfoResSize = 0 then
    exit;

  GetMem(VersionInfo, InfoResSize);
  try
    if not GetFileVersionInfo(PChar(FileName), 0, InfoResSize, VersionInfo) then
    begin
      FreeMem(VersionInfo);
      exit;
    end;

    InfoPath := '\' + #0;
    if VerQueryValue(VersionInfo, PChar(InfoPath), Info, VersionInfoSize) then
    begin
      result := IntToStr(HIWORD(PVSFixedFileInfo(Info).dwFileVersionMS)) + '.' +
                IntToStr(LOWORD(PVSFixedFileInfo(Info).dwFileVersionMS)) + '.' +
                IntToStr(HIWORD(PVSFixedFileInfo(Info).dwFileVersionLS)) + '.' +
                IntToStr(LOWORD(PVSFixedFileInfo(Info).dwFileVersionLS));
    end;

  finally
    FreeMem(VersionInfo, InfoResSize);
  end;

end;


procedure TestDateRoutines();
var
  Date: TDateTime;
  Date2: TDateTime;
  i: integer;
  DiffWeek1, DiffWeek2: integer;
  Date3, Date4: TDateTime;
  Year, Week, Dummy: Word;
begin
  Date := EncodeDateTime(2015, 6, 15, 0, 0, 0, 0);
  for i := 0 to 100 do
  begin
    Date2 := Date + i;
    DiffWeek1 := KalenderWoche(Date2) - KalenderWoche(Date2 + 1);
    DiffWeek2 := InternalWeekNumber(Date2) - InternalWeekNumber(Date2 + 1);

    if DiffWeek1 <> DiffWeek2 then
      raise Exception.Create('InternalWeekNumber macht Fehler');

    Date3 := MondayBefore(Date2);

    DecodeDateWeek(Date2, Year, Week, Dummy);
    Date4 := EncodeDateWeek(Year, Week);

    if Date3 <> Date4 then
    begin
      raise Exception.Create('MondayBefore macht Fehler');
    end;
  end;
end;

function TabDavor(Win1, Win2: TWinControl): boolean;
Begin
  if Win2.Top + Win2.Height <= Win1.Top then
  Begin
    Result := True;
    Exit;
  End;
  if Win1.Top + Win1.Height <= Win2.Top then
  Begin
    Result := False;
    Exit;
  End;
  if (Win2.Left <> Win1.Left) then
  Begin
    Result := (Win2.Left < Win1.Left);
    Exit;
  End;
  if Win1.Top <> Win2.Top then
  Begin
    Result := (Win1.Top > Win2.Top);
    Exit;
  End;
  Result := (Win1.TabOrder > Win2.TabOrder);
End;

procedure FixFormSize(Win: TWinControl; Width, Height: integer);
begin
  Win.Width := min(muldiv(Width, Screen.PixelsPerInch, 96), MulDiv(Screen.Width, 9, 10));
  Win.Height := min(muldiv(Height, Screen.PixelsPerInch, 96), MulDiv(Screen.Height, 9, 10));
end;

var
  UsedFont: String;


(* Setzt die Tabulatorreihenfolge automatisch anhand der Position auf dem Formular *)
procedure FixControls(Win: TWinControl);
var
  Nummer,x,y: integer;
  Win1, Win2: TWinControl;
  Winds: TList;
Begin
  if UsedFont <> '' then
  begin
    if Win is TForm then
    begin
      (Win as TForm).Font.Name := UsedFont;
    end;
  end;
  (* Aufruf fr alle Unterfenster *)
  for x:=0 to Win.ControlCount-1 do
  Begin
    if Win.Controls[x] is TWinControl then
    Begin
        FixControls((Win.Controls[x] as TWinControl));
    End;
  End;

  Winds := TList.Create;

  for x:=0 to Win.ControlCount-1 do
  Begin
    Winds.Add(nil);
  End;

  for x:=0 to Win.ControlCount-1 do
  Begin
    if Win.Controls[x] is TWinControl then
    Begin
      Win1 := (Win.Controls[x] as TWinControl);
      Nummer := 0;
      for y:=0 to Win.ControlCount-1 do
      Begin
        if y = x then continue;

        if Win.Controls[y] is TWinControl then
        Begin
          Win2 := (Win.Controls[y] as TWinControl);

          if TabDavor(Win1, Win2) then
            Inc(Nummer);
        End;
      End;
      Winds[Nummer] := Win1;
    End;
  End;

  x:=0;
  while x < Winds.Count do
  Begin
    if Winds[x] = nil then
    Begin
      Winds.Delete(x);
      x := -1;
    End;
    Inc(x);
  End;

  x:=0;
  while x < Winds.Count do
  Begin
    Win1 := Winds[x];

    if Win1.TabOrder <> x then
    Begin
      Win1.TabOrder := x;
      x := 0;
    End
    Else
      Inc(x);
  End;
  Winds.Free;
End;


const
  cArial    = 'Arial';

function RunsUnderWine(): boolean;
var
  hntdll: HMODULE;
begin
  Result := False;

  hntdll := GetModuleHandle('ntdll.dll');
  if(hntdll <> 0) then
  begin
    if nil <> GetProcAddress(hntdll, 'wine_get_version') then
    begin
      Result := True;
    end;
  end;
end;


procedure DetermineUsedFont;
begin
  if RunsUnderWine then
  begin
    UsedFont := cArial;
  end
  else
  begin
    UsedFont := '';
  end;
end;

function GetListComboValue(ComboBox: TComboBox): String;
begin
  Result := '';
  if ComboBox.ItemIndex >= 0  then
  begin
    Result := ComboBox.Items[ComboBox.ItemIndex];
  end;
end;

procedure SetListComboValue(ComboBox: TComboBox; Value: String);
begin
  ComboBox.ItemIndex := ComboBox.Items.IndexOf(Value);
end;

function DecToRom(Dec: LongInt): String;
const
  Nums : Array[1..13] of Integer =
    (1, 4, 5, 9, 10, 40, 50, 90, 100,
      400, 500, 900, 1000);
  RomanNums:  Array[1..13] of string =
    ('I', 'IV', 'V', 'IX', 'X', 'XL',
      'L', 'XC', 'C', 'CD', 'D', 'CM', 'M');
var
  i: Integer;
begin
  Result := '';
  for i := 13 downto 1 do
    while (Dec >= Nums[i]) do
    begin
      Dec := Dec - Nums[i];
      Result  := Result + RomanNums[i];
    end;
end;

initialization
  DetermineUsedFont;
  LogCriticalSection := TCriticalSection.Create;
  LogFileName := '';
finalization
  LogCriticalSection.Free;
end.
