unit PlanTypes;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Contnrs,
  Math, SyncObjs, PlanUtils, System.Generics.Collections, PlanDataObjects,
  System.Generics.Defaults, TypInfo, Xml.XMLDoc, Xml.XMLIntf, DateUtils;


const

  cThreadTauschPercent: array[0..13] of String = ('R', '100', '15', '5', '2', '1', 'S1,100', 'S1,25', 'S1,10', 'S2,10', 'M1,10', 'M1,25', 'M1,100', 'M2,10');
  cMaxThreads = 4;
  NUM_DYN_THREADS = 9;

(*
  cThreadTauschPercent: array[0..0] of String = ('R');
  NUM_DYN_THREADS = 0;
  cMaxThreads = 1;
{$ifopt D-}
    {$Message Fatal 'Nix nur ein Thread'}
{$endif}
*)


var
  MinGameDistance: TDateTime;

{$ifdef CACHE_HIT_TEST}
  CacheHit, CacheMiss: integer;
{$endif}


type

  TWunschterminOption = (woKoppelPossible, woKoppelSoft, woKoppelHard, woKoppelDoubleDatePossible, woKoppelDoubleDateSoft, woKoppelDoubleDateHard, woKoppelSecondTime, woAuswaertsKoppelSecondTime, woAuswaertsKoppelHasSecondTime, woKoppelNothing);

  TWunschterminOptions = set of TWunschterminOption;


  TDateList = class(TList<TDateTime>)
  private
  public
    procedure Assign(Source: TDateList);
  end;

  TMannschaftsKostenType = (mktHalleBelegt, mktSisterGames, mktForceSameHomeGames, mktSperrTermine, mktAusweichTermine, mkt60Kilometer, mktEngeTermine, mkt2SpieleProWoche, mktSpielverteilung, mktKoppelTermine, mktAuswaertsKoppelTermine, mktZahlHeimSpielTermine, mktWechselHeimAuswaerts, mktAbstandHeimAuswaerts, mktRanking, mktMandatoryGames);

  TPlanKostenType = (pktGameDayOverlap, pktGameDayLength, pktLastGameDayOverlap, pktLastGameDayLength, pktSisterGamesAtBegin);

  TAuswaertsKoppelType = (ktNone, ktSameDay, ktDiffDay, ktAllDay);

  THomeRightValue = (hrNone, hrRound1, hrRound2);

const
  cAuswaertsKoppelTypeName: array[Low(TAuswaertsKoppelType) .. High(TAuswaertsKoppelType)] of String = ('kein Koppeltermin', 'am gleichen Tag', 'mit bernachtung', 'am gleichen Tag oder mit bernachten');

  cWunschterminOptionName: array[Low(TWunschterminOption) .. High(TWunschterminOption)] of String =
  ('Zwei Heimspiele an diesem Tag mglich', 'Zwei Heimspiele an diesem Tag gewnscht', 'Zwei Heimspiele an diesem Tag gewnscht (hohe Prio)', 'Doppelspieltag mglich', 'Doppelspieltag gewnscht', 'Doppelspieltag gewnscht (hohe Prio)'
    , 'Zusatztermin fr Koppeltermine', 'Zusatztermin fr Auswrtskoppeltermine', 'Alternativzeit fr Auswrtskoppeltermine vorhanden', 'kein Koppeltermin/Doppelspieltag');


type

  TKostenValue = record
    Anzahl: Integer;
    AllCount: integer;
    GesamtKosten: MyDouble;
  end;

  TMannschaft = class;
  TCalculateOptions = class;
  TPlan = class;
  TWunschTermin = class;

  TMannschaftsKostenValue = array[Low(TMannschaftsKostenType) .. High(TMannschaftsKostenType)] of TKostenValue;

  TKostenCache = record
    Valid: boolean;
    Values: TMannschaftsKostenValue;
  end;


  TMannschaftsKosten = class(TObject)
  public
    Values: TMannschaftsKostenValue;
    procedure ClearValues(KostenType: TMannschaftsKostenType); inline;
  private
    FMessages: TStrings;
    function GetMessages: TStrings;
  public
    constructor Create();
    destructor Destroy(); override;

    function getGesamtKosten(): MyDouble; inline;
    property Messages: TStrings read GetMessages;
  end;

  TGame = class(TObject)
  private
    FDate: TDateTime;
    FDateOptions: TWunschterminOptions;
    FDateMaxHome: Integer;
    FWunschTermin: TWunschtermin;
    FFixedDate: boolean;
    FNotNecessary: boolean;
    FDateRef: TDateTime;
    FDateRefOptions: TWunschterminOptions;
    FDateRefMaxHome: Integer;
    FWunschTerminRef: TWunschtermin;
    FNotNecessaryRef: boolean;
    procedure ClearAllValues; inline;
    function getLocation: String;

  public
    MannschaftHeim: TMannschaft;
    MannschaftGast: TMannschaft;
    OverrideLocation: String;

    constructor Create(MannschaftHeim: TMannschaft; MannschaftGast: TMannschaft);
    destructor Destroy(); override;

    procedure setDate(Date: TDateTime; IsKoppel: TWunschterminOptions; MaxHome: Integer; fixedDate: boolean; notNecessary: boolean; WunschTermin: TWunschTermin); inline;
    procedure setEmptyDate(); inline;

    function DateValid(): Boolean; inline;

    function isTheSame(game: TGame): boolean; inline;

    function AsString(): String;

    property Date: TDateTime read FDate;
    property DateOptions: TWunschterminOptions read FDateOptions;
    property DateMaxHome: Integer read FDateMaxHome;
    property FixedDate: boolean read FFixedDate;
    property NotNecessary: boolean read FNotNecessary;
    property Location: String read getLocation;
  end;

  TGameComparer = class(TComparer<TGame>)
  public
    function Compare(const game1, game2: TGame): Integer; override;
  end;

  TGameList = class(TObjectList<TGame>)
  private
    gameComparer: TGameComparer;
  public
    constructor Create(OwnsObjects: boolean);
    destructor Destroy(); override;

    function findSameGame(game: TGame): TGame;

    procedure GetRefListe(Result: TGameList);
    procedure SortByDate();
    function getFirstGameRefIndexByDate(Date: TDateTime): integer;
  end;

  TScheduleGame = class(TObject)
  public
    Date: TDateTime;
    MannschaftHeimName: String;
    MannschaftGastName: String;
    Location: String;

    function DateValid(): boolean;
  end;

  TScheduleComparer = class(TComparer<TScheduleGame>)
  public
    function Compare(const game1, game2: TScheduleGame): Integer; override;
  end;


  TSchedule = class(TObjectList<TScheduleGame>)
  private
    ScheduleComparer: TScheduleComparer;
  public
    constructor Create();
    destructor Destroy(); override;
    procedure SortByDate();
  end;


  TWunschTermin = class(TObject)
  public
    Date: TDateTime;
    ParallelGames: Integer;
    Options: TWunschterminOptions;
    KoppelSecondTime: TDateTime;
    assignedGame: TGame;
    Location: String;
    procedure Assign(Source: TWunschTermin);
  end;

  TAuswaertsKoppel = class(TObject)
  public
    TeamA: String;
    TeamB: String;
    CachedMannschaftA: TMannschaft;
    CachedMannschaftB: TMannschaft;
    Option: TAuswaertsKoppelType;
    procedure Assign(Source: TAuswaertsKoppel);
    function AsString: String;
    function getMannschaftA(Plan: TPlan): TMannschaft; inline;
    function getMannschaftB(Plan: TPlan): TMannschaft; inline;
  end;

  TAuswaertsKoppelList = class(TObjectList<TAuswaertsKoppel>)
  public
    procedure Assign(Source: TAuswaertsKoppelList);
  end;


  THomeRight = class(TObject)
  public
    Team: String;
    Option: THomeRightValue;
    procedure Assign(Source: THomeRight);
    function AsString(): String;
  end;


  THomeRightList = class(TObjectList<THomeRight>)
  public
    procedure Assign(Source: THomeRightList);
    function getValueForTeam(teamName: String): THomeRightValue;
  end;


  TWunschTerminList = class(TObjectList<TWunschTermin>)
  public
    constructor Create(OwnsObjects: boolean = True);
    procedure Assign(Source: TWunschTerminList);

    function Find(Date: TDateTime): TWunschTermin; inline;
    function Exists(Date: TDateTime): boolean; inline;
    function ExistsIgnoreTime(Date: TDateTime): boolean; inline;
  end;

  TSisterGame = class(TObject)
  public
    Date: TDateTime;
    gender: String;
    Number: Integer;
    teamName: String;
    IsHomeGame: Boolean;
    HomeMannschaftName: String;
    GuestMannschaftName: String;
    Location: String;
    PreventParallelGame: boolean;
    ForceParallelHomeGames: boolean;
    procedure Assign(Source: TSisterGame);
  end;

  TSisterGames = class(TObjectList<TSisterGame>)
  public
    function toSortedArray: TArray<TSisterGame>;
  end;

  TSisterGameComparer = class(TComparer<TSisterGame>)
  public
    function Compare(const game1, game2: TSisterGame): Integer; override;
  end;


  TSisterGamesComparer = class(TComparer<TSisterGames>)
  public
    function Compare(const game1, game2: TSisterGames): Integer; override;
  end;


  TMapSisterGames = class(TObject)
  private
    values: TObjectDictionary<Integer, TSisterGames>;
  public
    constructor Create();
    destructor Destroy(); override;

    procedure Assign(Source: TMapSisterGames);

    procedure Add(sisterGame: TSisterGame);

    function get(Date: TDateTime): TSisterGames; inline;

    function IsEmpty(): Boolean; inline;

    function toArray(): TArray<TSisterGames>;
    function toSortedArray(): TArray<TSisterGames>;
  end;

  TMannschaftList = class;

  TParallelGamesSuchRichtung = (tpgBoth, tpgTop, tpgDown);

  // Optimierter Zugriff auf die Teams zu denen Auswrtskoppelspiele gewnscht sind
  TAuswaertsKoppelCachedTeams = class(TObject)
  public
    TeamsOnDay: TList<String>;
    TeamIdToTeamId: TObjectDictionary<String, TList<String>>;
  public
    constructor Create();
    destructor Destroy(); override;

    procedure AddTeamIdToTeamId(TeamA, TeamB: String);
  end;

  TMannschaftDataCache = class(TObject)
  public
    CachedAuswaertsKoppelTeams: TAuswaertsKoppelCachedTeams;
    // Nachbarmannschaften, die am gleichen Tag Heimspiel haben sollen Gender + Tab + Manschaftsname
    CachedTeamsForForceHomeGames: TStringList;
    constructor Create();
    destructor Destroy(); override;
  end;

  TMessageType = (tmNoMessage, tmNormalMessage, tmDetailMessage);

  TMannschaft = class(TObject)
  private
    function IsKoppelTermin(game: TGame): TWunschterminOptions; inline;
    procedure loadSisterGameDays(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
    procedure loadSisterTeam(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
    procedure LoadNoWeekGame(Plan: TPlan; Data: TDataObject);

    procedure CalculateKosten60kMRegel(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenSperrTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenEngeTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenSpielverteilung(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenHallenBelegung(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenParallelGames(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenSameHomeDates(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKosten2SpieleDieWoche(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenKoppelTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenAuswaertsKoppelTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenMandatoryGames(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenZahlHeimSpiele(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenWechselHeimAuswaerts(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenAbstandHeimAuswaerts(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenRanking(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure CalculateKostenAusweichTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);

    procedure GetParallelGamesIntern(Date: TDateTime; Plan: TPlan; SearchTeamNum: integer; SearchRichtung: TParallelGamesSuchRichtung;
                                      var NumGames: integer; var MessageString: String; FillMessageString, IgnoreThisPlan: boolean);

    procedure GetMissingParallelHomeGames(Date: TDateTime; Plan: TPlan; var NumGames: integer; var MessageString: String; FillMessageString: boolean);

    procedure loadAuswaertsKoppel(Plan: TPlan; Data: TDataObject);
    function IsValidAuswaertsKoppelDate(Date1, Date2: TDateTime; teamA, teamB: String): boolean;
    function getKoppeledGameAfter(IndexInRefGames: integer): TGame;

    function  getCachedKosten(AktType: TMannschaftsKostenType): MyDouble; inline;
    procedure setCachedKosten(AktType: TMannschaftsKostenType; Kosten: TKostenValue); inline;
    function CreateListOfViolatedSperrTermine(Plan: TPlan): TList<TDateTime>;
    function CreateListOfViolated60KmRegel(Plan: TPlan): TList<TGame>;
    procedure getMessagesEngeTermine2Days(Plan: TPlan; Messages: TStrings);
    function getKoppeledGameBefore(IndexInRefGames: integer): TGame;
    procedure loadHomeRights(Plan: TPlan; Data: TDataObject);
  public
    teamName: String;
    clubID: String;
    teamId: String;
    teamNumber: Integer;
    DefaultLocation: String;

    sisterGames: TMapSisterGames;

    noWeekGame: TList<String>;

    SperrTermine: TDateList;
    AusweichTermine: TDateList;
    WunschTermine: TWunschTerminList;
    AuswaertsKoppel: TAuswaertsKoppelList;

    HomeRightCountRoundOne: integer;
    HomeRights: THomeRightList;

    KostenAnzahl: MyDouble;
    KostenAbstand: MyDouble;
    KostenWochentag: MyDouble;

    SisterTeamsInPlan: TMannschaftList;

    gamesRef: TGameList;

    Index: integer;

    DataCache: TMannschaftDataCache;

    KostenCache: TKostenCache;
    KostenCacheRef: TKostenCache;

    constructor Create(Index: integer);
    destructor Destroy(); override;
    procedure Assign(Source: TMannschaft);

    function getKoppeledGame(Game: TGame): TGame; overload;
    function getKoppeledGame(IndexInRefGames: integer): TGame; overload;
    function InternIsKoppeledGames(game1, game2: TGame): Boolean;
    function IsAuswaertsKoppeledGames(game1, game2: TGame): boolean;

    function IsTerminFree(Termin: TWunschtermin; Plan: TPlan; HeimMannschaft: TMannschaft; GastMannschaft: TMannschaft; IgnoreGame: TGame; AllSecondTimeGamesAreValid: boolean = false): Boolean;

    procedure CalculateKosten(Plan: TPlan; Kosten: TMannschaftsKosten; WithMessages: TMessageType);

    procedure getRankingDiffs(Plan: TPlan; ResultValues: TList<Integer>);

    procedure load(Data: TDataObject; Plan: TPlan; PlanData: TPlanData);
    procedure loadGameDays(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
    procedure loadHomeGame(Plan: TPlan; TeamNode: TDataObject; Data: TDataObject; PlanData: TPlanData);
    procedure loadNoGame(Plan: TPlan; Data: TDataObject);

    procedure CalculateKostenForType(Plan: TPlan; KostenType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
    procedure GetHallenBelegung(Date: TDateTime; Location: String; Plan: TPlan; var BelegungNum: integer; var BelegungString: String; FillBelegungString: boolean; IgnoreThisPlan: boolean);
    procedure GetParallelGames(Date: TDateTime; Plan: TPlan; var NumGames: integer; var MessageString: String; FillMessageString: boolean; IgnoreThisPlan: boolean);

    procedure getTermineNotPossible(Plan: TPlan; MessageList: TList<String>);

    function  IsMannschaftInAuswaertsKoppelWunschAtOneDay(Plan: TPlan; MannschaftHeim: TMannschaft): boolean;

    function  isAusWeichTermin(Date: TDateTime): boolean;

    function  CreateAuswaertsKoppeledGamesDependenciesList(Plan: TPlan; MannschaftHeim: TMannschaft): TGameList;

    function  getGameCount(FromDate, ToDate: TDateTime): integer;

    function  getCache(Plan: TPlan): TMannschaftDataCache;

    procedure ClearCache();
    procedure ClearKostenCache(); inline;
  end;

  TMannschaftList = class(TObjectList<TMannschaft>)
  public
    constructor Create(OwnsObjects: boolean = True);
    procedure Assign(Source: TMannschaftList; Plan: TPlan);
    function FindByName(const Name: String): TMannschaft; inline;
    function FindByTeamId(const TeamId: String): TMannschaft; inline;
    procedure Sort();
  end;

  TCalculateOptionsValues = (coIgnore, coSehrWenig, coWenig, coNormal, coHoch, coSehrHoch, coExtremHoch);


  TPlanMandatoryTime = class(TObject)
  public
    DateFrom: TDateTime;
    DateTo: TDateTime;
    GameCount: integer;
    procedure Assign(Source: TPlanMandatoryTime);
    function AsString: String;
    function IsTheSame(Source: TPlanMandatoryTime): boolean;
  end;

  TPlanMandatoryTimeList = class(TObjectList<TPlanMandatoryTime>)
  public
    constructor Create();
    procedure Assign(Source: TPlanMandatoryTimeList);
    function FindValueByDate(Value: TPlanMandatoryTime): integer; inline;
  end;

  TRoundPlaning = (rpBoth, rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona);

  TMannschaftsKostenArray = array[Low(TMannschaftsKostenType) .. High(TMannschaftsKostenType)] of TCalculateOptionsValues;

  TGewichtungMannschaften = class(TObject)
  private
    // TeamId zu Werten
    GewichtungMannschaften: TDictionary<String, TCalculateOptionsValues>;
    GewichtungMannschaftenDetail: TDictionary<String, TMannschaftsKostenArray>;
    function GetTeamIds: TEnumerable<String>;
    procedure AddTeam(teamId: String);
  public
    constructor Create();
    destructor  Destroy(); override;
    procedure Clear();
    procedure Assign(Source: TGewichtungMannschaften);
    procedure addMain(teamId: String; value: TCalculateOptionsValues);
    procedure addDetail(teamId: String; Art: TMannschaftsKostenType; value: TCalculateOptionsValues);

    function getMain(teamId: String): TCalculateOptionsValues;
    function getDetail(teamId: String; Art: TMannschaftsKostenType): TCalculateOptionsValues;

    property TeamIds: TEnumerable<String> read GetTeamIds;
  end;


  TCalculateOptions = class(TObject)
  private
    procedure getWeightMessagesDiffToDefault(Plan: TPlan; Messages: TStrings);
    function getOptionsValueFromAttribute(Plan: TPlan; node: IXMLNode; attName: String): TCalculateOptionsValues;
  public
    FreitagIsAllowedBy60km: boolean;
    SpieltagGewichtung: TCalculateOptionsValues;
    SpieltagOverlapp: TCalculateOptionsValues;
    SisterTeamsAmAnfangGewichtung: TCalculateOptionsValues;
    LastSpieltagGewichtung: TCalculateOptionsValues;
    LastSpieltagOverlapp: TCalculateOptionsValues;

    GewichtungMainMannschaftsKosten: TMannschaftsKostenArray;

    GewichtungMannschaften: TGewichtungMannschaften;

    RoundPlaning: TRoundPlaning;

    DoubleRound: boolean;
    AutomaticMidDate: boolean;
    MidDate1: TDateTime;
    MidDate2: TDateTime;

    constructor Create();
    destructor Destroy(); override;
    procedure Clear();
    procedure Assign(Source: TCalculateOptions);
    function ToString: string; override;

    function getMidDate1(Plan: TPlan): TDateTime;
    function getMidDate2(Plan: TPlan): TDateTime;

    function getAutomaticMidDate1(Plan: TPlan): TDateTime;
    function getAutomaticMidDate2(Plan: TPlan): TDateTime;

    function getGewichtungPlan(KostenType: TPlanKostenType): TCalculateOptionsValues;
    procedure setGewichtungPlan(KostenType: TPlanKostenType; value: TCalculateOptionsValues);

    procedure Load(Plan: TPlan; FileName: String);
    procedure Save(Plan: TPlan; FileName: String);

    procedure getDiffToDefault(Plan: TPlan; Messages: TStrings);

  end;

  TRoundInfo = class(TObject)
  public
    DateFrom: TDateTime;
    DateTo: TDateTime;
  public
    function DateInRound(Date: TDateTime): boolean; inline;
  end;


  TPlanNoGameDay = class(TObject)
  public
    DateFrom: TDateTime;
    DateTo: TDateTime;
    Comment: String;
    procedure Assign(Source: TPlanNoGameDay);
  end;

  TPlanNoGameDayList = class(TObjectList<TPlanNoGameDay>)
  public
    procedure Assign(Source: TPlanNoGameDayList);
  end;


  TTauschTyp = (ttNormal, ttSpielTag, ttMannschaft, ttRaster);

  TPlanDataCache = class(TObject)
  public
    Has60KilometerValues: boolean;
    NumWunschDays: integer;
    HasAuswaertsKoppelTermine: boolean;
    HasKoppelTermine: boolean;
    HasAusweichTermine: boolean;
    HasMandatoryGameDays: boolean;
    HasForceSameHomeGames: boolean;

    sollGamesByDate: TDictionary<string, MyDouble>;

    constructor Create();
    destructor Destroy(); override;

    function getSollGamesByDate(): TDictionary<string, MyDouble>;
  end;

  TPlan = class(TObject)
  private
    procedure loadTeams(Data: TDataObject; PlanData: TPlanData);
    procedure GenerateGames;
    function findGames(const HeimId, GastId: String): TGameList; inline;
    procedure FillTermine();
    procedure FillTermin(Game: TGame; AllSecondTimeGamesAreValid: boolean); inline;
    function GetSpielTageKostenFaktor: MyDouble; inline;
    function GetMannschaftKostenFaktor(Mannschaft: TMannschaft; AktType: TMannschaftsKostenType): MyDouble; inline;
    procedure SortMannschaftsDates();
    procedure RemoveNotValidSisterGames;
    procedure AddSisterTeamsInPlan;
    function GetSisterTeamsAmAnfangKostenFaktor: MyDouble; inline;
    function GetSpielOverlapKostenFaktor: MyDouble; inline;
    function GetLastSpielOverlapKostenFaktor: MyDouble;
    function GetLastSpielTageKostenFaktor: MyDouble;
    procedure CreateRoundInfo();
    function TerminIsValid(WunschTermin: TWunschTermin; Game: TGame; IgnoreGame: TGame = nil; AllSecondTimeGamesAreValid: boolean = false): Boolean;
    procedure GenerateMannschaftsRefGames;
    procedure ClearMannschaftGameRefs;
    procedure ClearMannschaftWunschTerminRefs();
    procedure ClearGameList(gameList: TGameList);

    procedure GenerateRaster();
    procedure NeuWuerfeln(TauschTyp: TTauschTyp; TauschAnzahl: integer; TauschPercent: MyDouble);
    procedure NeuWuerfelnForGames(Games: TGameList; Percent: MyDouble;
      WithRueckGame: boolean);
    procedure CreateRasterOneRound(Mannschaften: TObjectList<TMannschaft>;
      SpieltagIndexe: TList<Integer>; DateFrom, DateTo: TDateTime;
      RueckRunde: boolean);
    procedure ScheduleGameForRaster(HeimMannschaft, GastMannschaft: TMannschaft;
      WocheMonday: TDateTime);

    // Falls nur die Vorrunde generiert wird, dann ist das Rckspiel nicht mehr ntig
    procedure MarkSecondGameAsUnNecessary(game: TGame);
    procedure ClearAllGamesOfMannschaften(MannschaftHeim, MannschaftGast: TMannschaft);
    procedure LoadGameDates(allGames: TGameList; DataObject: TDataObject; PlanData: TPlanData);
    procedure assignPredefinedGames(allGames: TGameList);
    procedure assignExistingScheduleGames(allGames: TGameList);
    procedure FillPredefinedGames;

    function CalcHas60KilometerValues: boolean;
    function CalcNumWunschDays: integer;
    function CalcHasAuswaertsKoppelTermine: boolean;
    function CalcHasKoppelTermine: boolean;
    function CalcHasAusweichTermine: boolean;
    function CalcHasMandatoryGameDays: boolean;
    procedure ModifyTermineFromOptions;
    function isValidDateForWunschtermin(Date: TDateTime): boolean;
    procedure FillTermineIntern(AllSecondTimeGamesAreValid: boolean);
    procedure RemoveNotValidSecondTimeGames;
    procedure LoadNoGameDays(Data: TDataObject);

    procedure ClearCache();
    function  getCache(): TPlanDataCache; inline;

    procedure FillTerminIntern(WunschTerminIndexList: array of Integer;
      Game: TGame; AllSecondTimeGamesAreValid: boolean);
    procedure FillAuswaertsKoppelTermin(Game: TGame; Date: TDateTime;
      AllSecondTimeGamesAreValid: boolean);
    procedure FillPredefinedGame(gamePredefined: TGame);
    function IsRoundToGenerate(Round: TRoundInfo): boolean;
    function getFirstRealDateInRound(currRound: integer): TDateTime;
    function getLastRealDateInRound(currRound: integer): TDateTime;
    procedure MarkNotNecessaryGames;
    function getNotTerminatedGames(Messages: TStrings): integer;
    function getNotAllowedGames(Messages: TStrings): integer;
    function getTeamsNotAtBegin: integer;
    function IsFreeGameDate(Date: TDateTime): boolean;
    function CreateCache: TPlanDataCache;
    function getNumSollGamesIntern(DateFrom, DateTo: TDateTime): MyDouble;
    function findChildNode(node: IXMLNode; const searchTag: String): IXMLNode;
    function getAttributValue(node: IXMLNode; const searchName,
      default: String): String;
    procedure setAttributValue(node: IXMLNode; const name, value: String);
    procedure LoadMandatoryGameDays(Data: TDataObject);
    procedure LoadRanking(Data: TDataObject);
    function HasForceSameHomeGames: boolean;
    function CalcHasForceSameHomeGames: boolean;
  public
    { Public-Deklarationen }
    dataCache: TPlanDataCache;
    gender: String;

    Mannschaften: TMannschaftList;

    existingSchedule: TGameList;
    predefinedGames: TGameList;
    gamesAllowed: TGameList;
    gamesNotAllowed: TGameList;
    cachedGameList: TObjectDictionary<String, TGameList>;

    MandatoryGameDays: TPlanMandatoryTimeList;
    FreeGameDays: TDictionary<TDateTime, String>;
    DateBeginRueckrundeInXML: TDateTime;
    DateBegin: TDateTime;
    DateEnd: TDateTime;
    PlanName: String;
    PlanId: String;
    LastOptimizedThread: String;

    RankingIndexe: TList<Integer>;

    FRounds: TObjectList<TRoundInfo>;

    FOptions: TCalculateOptions;

    AllDatesAreValid: boolean;
  public
    constructor Create();
    destructor Destroy(); override;
    procedure Clear;
    procedure Assign(Source: TPlan);
    procedure AssignPlanData(Source: TPlan);

    // Weenn sich keine Optionen gendert haben, dann die schnelle Variante um die Termine zu setzen
    procedure AssignOptimizedDates(Source: TPlan);

    function GetFirstDate: TDateTime; inline;
    function GetLastDate: TDateTime; inline;

    function IsInRoundToGenerate(Date: TDateTime): boolean;

    procedure getSpieltagMondays(SpieltageMondays: TList<TDateTime>; DateFrom, DateTo: TDateTime; OnlyWunschTermine: boolean = true);

    function GameIsValid(Game: TGame): Boolean;

    function DateIsAllowedForWeekendGames(Date: TDateTime): boolean;

    procedure AssignOptions(Options: TCalculateOptions);

    procedure RemoveNotValidDates();

    function  getOptions(): TCalculateOptions; inline;

    procedure clearAllDates();

    procedure getCloneOfAllGames(gamesList: TGameList);

    procedure AssignGames(allGames: TGameList);

    procedure getSchedule(gamesList: TSchedule);
    procedure AssignSchedule(allGames: TSchedule);

    function NoGames(): boolean; inline;

    // 0 Vorrunde 1 Rckrunde 2 Doppelrunde Corrunde 1 etc. -1 nicht im erlaubten Bereich
    function GetRoundNumberByDate(Date: TDateTime): Integer; inline;

    function getNumSollGames(DateFrom: TDateTime; DateTo: TDateTime): MyDouble;

    procedure Load(PlanData: TPlanData);
    procedure LoadScheduleFromXml(FileName: String);

    procedure TransferRefDateToDate();
    procedure TransferDateToRefDate();

    function CalculateKosten(BreakByValue: MyDouble): MyDouble;
    procedure CalculateSpielTagKosten(var KostenBreite, KostenOverlap, LastKostenBreite, LastKostenOverlap: MyDouble);
    function CalculateFehlterminKosten: MyDouble;
    function CalculateNotAllowedKosten: MyDouble;
    function CalculateFreeGameDateKosten: MyDouble;
    function CalculateSisterTeamsAmAnfang: MyDouble;

    function CalculateMannschaftsKosten(KostenType: TMannschaftsKostenType): MyDouble;

    function GetMannschaftKostenDisplayInteger(Mannschaft: TMannschaft; AktType: TMannschaftsKostenType): integer;

    procedure SaveScheduleToCsv(FileName: String);
    procedure SaveScheduleToXml(FileName: String);

    procedure ProfileFillTermine(TauschPercent: MyDouble);

    function  getMaxGamePerMannschaft(): integer;

    function HasValuesForType(KostenType: TMannschaftsKostenType): boolean; inline;

    function HasPossibleKoppelTermine(): boolean;

    function Has60KilometerValues: boolean; inline;
    function GetNumWunschDays: integer; inline;
    function HasAuswaertsKoppelTermine: boolean; inline;
    function HasKoppelTermine: boolean; inline;
    function HasAusweichTermine: boolean; inline;
    function HasMandatoryGameDays: boolean; inline;
    function HasRanking: boolean; inline;

    procedure getHardErrorMessages(Messages: TStrings);

    property Rounds: TObjectList<TRoundInfo> read FRounds write FRounds;
  end;


  TPlanCalcThread = class (TThread)
  private
    Durchlauf: int64;
    DurchLaufAfterReInit: int64;
    OptKosten: MyDouble;
    RetKosten: MyDouble;
  private
    OptPlan: TPlan;
    Plan: TPlan;
    ReInitPlanFlag: boolean;
    ReInitPlanDataFlag: boolean;
    ReInitPlan: TPlan;
    ReInitPlanData: TPlan;
    ParentCriticalSection: TCriticalSection;
    TauschTyp: TTauschTyp;
    TauschAnzahl: integer;
    TauschPercent: MyDouble;
    ThreadName: String;
    ReinitString: String;
    FPaused: boolean;
    LowPrioThread: boolean;
    procedure SetPaused(const Value: boolean);
  public
    ThreadType: String;
    IsInit: boolean;
    procedure GetActResult(Plan: TPlan; var DurchLauf: Int64; var Kosten: MyDouble);
    procedure SetThreadType(TauschArt: String);
    function GetCompleteThreadName: String;
  public
    constructor Create();
    destructor  Destroy; override;
    procedure StartWithLists(ThreadName: String; Plan: TPlan; TauschArt: String; ParentCriticalSection: TCriticalSection);
    procedure DoReInitPlan(Plan: TPlan; const ReinitString: String);
    procedure DoReInitPlanData(Plan: TPlan; const ReinitString: String);
    procedure Execute; override;
    property Paused: boolean read FPaused write SetPaused;
  end;


  TSavedPlan = class(TObject)
  public
    FileName: String;
    Name: String;
    FileAge: TDateTime;
    Valid: Boolean;
    Plan: TPlan;
  public
    constructor Create();
    destructor Destroy(); override;
  end;


const

  cMannschaftsKostenTypeNamen: array[Low(TMannschaftsKostenType) .. High(TMannschaftsKostenType)] of String =
        ('Hallenbelegung', 'parallele Spiele', 'synchrone Heimsp.', 'Sperrtermine', 'Ausweichtermine', '60km Regel', '3-Tage-Abstand', '2-Spiele-die-Woche', 'Spielverteilung', 'Heimkoppel', 'Auswrtskoppel', 'Ungleich H/A', 'Wechsel H/A', 'Abstand H/A', 'Setzliste', 'Pflichtspieltage');

  cPlanKostenTypeNamen: array[Low(TPlanKostenType) .. High(TPlanKostenType)] of String =
        ('berlappung der Spieltage', 'Lnge der Spieltage', 'berlappung letzer Spieltag', 'Lnge letzer Spieltag', 'Vereinsinterne Spiele am Anfang');

  cCalculateOptionsNamen: array[Low(TCalculateOptionsValues) .. High(TCalculateOptionsValues)] of String =
        ('Nicht bercksichtigen', 'sehr wenig', 'wenig', 'normal', 'hoch', 'sehr hoch', 'extrem hoch');


  cCalculateOptionsValues: array[Low(TCalculateOptionsValues) .. High(TCalculateOptionsValues)] of MyDouble =
        (0, 1, 10, 100, 1000, 10000, 100000);

  cCalculateOptionsDisplayInteger: array[Low(TCalculateOptionsValues) .. High(TCalculateOptionsValues)] of integer =
        (-1000, -2, -1, 0, 1, 2, 3);


function OptionIsKoppel(Options: TWunschterminOptions): boolean;
function OptionIsKoppelSingleDate(Options: TWunschterminOptions): boolean;

function KoppelValueFromOptions(Option: TWunschterminOptions): TWunschterminOption;

function FormatDateForExport(Date: TDateTime): string;

function Multiply(Value: MyDouble; MulValue: TCalculateOptionsValues): MyDouble;

implementation


const
  cMaxKostenNoHardError = 1000000000.0;


function PercentRandom(Percent: Integer): boolean;
begin
  Result := (Random(100) < Percent);
end;

function KoppelValueFromOptions(Option: TWunschterminOptions): TWunschterminOption;
begin
  Result := woKoppelNothing;
  if woKoppelHard in Option then
  begin
    Result := woKoppelHard;
  end;
  if woKoppelPossible in Option then
  begin
    Result := woKoppelPossible;
  end;
  if woKoppelSoft in Option then
  begin
    Result := woKoppelSoft;
  end;
  if woKoppelDoubleDateSoft in Option then
  begin
    Result := woKoppelDoubleDateSoft;
  end;
  if woKoppelDoubleDateHard in Option then
  begin
    Result := woKoppelDoubleDateHard;
  end;
  if woKoppelDoubleDatePossible in Option then
  begin
    Result := woKoppelDoubleDatePossible;
  end;
end;



function Date: TObject;
begin
  Result := nil;
end;


function OptionIsKoppel(Options: TWunschterminOptions): boolean;
begin
  if (woKoppelPossible in Options) or (woKoppelHard in Options) or (woKoppelSoft in Options) or (woKoppelDoubleDateSoft in Options) or (woKoppelDoubleDateHard in Options) or (woKoppelDoubleDatePossible in Options) then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;
end;


function OptionIsKoppelHard(Options: TWunschterminOptions): boolean;
begin
  if (woKoppelHard in Options)or (woKoppelDoubleDateHard in Options) then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;
end;

function OptionIsKoppelSoft(Options: TWunschterminOptions): boolean;
begin
  if (woKoppelSoft in Options) or (woKoppelDoubleDateSoft in Options) then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;
end;



function OptionIsKoppelSingleDate(Options: TWunschterminOptions): boolean;
begin
  if (woKoppelPossible in Options) or (woKoppelHard in Options) or (woKoppelSoft in Options) then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;
end;



function TPlan.findChildNode(node: IXMLNode; const searchTag: String): IXMLNode;
var
  i: integer;
begin
  Result := nil;
  for i:=0 to node.childNodes.Count - 1 do
  begin
    if node.childNodes[i].NodeName = searchTag then
    begin
      Result := node.childNodes[i];
      Break;
    end;
  end;
end;


function TPlan.getAttributValue(node: IXMLNode; const searchName: String; const default: String): String;
begin
  Result := default;
  if node.HasAttribute(searchName) then
  begin
    Result := VarToStr(node.Attributes[searchName]);
  end
end;

procedure TPlan.setAttributValue(node: IXMLNode; const name: String; const value: String);
begin
  node.Attributes[name] := value;
end;


{ TDateList }


procedure TDateList.Assign(Source: TDateList);
begin
  Clear;
  AddRange(Source.ToArray);
end;



{ TPlanCalcThread }


destructor TPlanCalcThread.Destroy;
begin
  Plan.Free;
  OptPlan.Free;
  FreeAndNil(ReInitPlan);
  FreeAndNil(ReInitPlanData);
  inherited;
end;

function TPlanCalcThread.GetCompleteThreadName(): String;
begin
  Result := ThreadName + '|' + ThreadType + ReinitString;
end;

procedure TPlanCalcThread.Execute;
var
  Kosten: MyDouble;
begin
{$ifdef TRACE}
  TraceString('Beginn TPlanCalcThread.Execute ' + GetCompleteThreadName());
{$endif}

  OptKosten := Plan.CalculateKosten(-1);
  Plan.TransferDateToRefDate();

  Durchlauf := 0;
  DurchLaufAfterReInit := 0;
  IsInit := True;
  while not Terminated do
  begin
    if Paused then
    begin
      Sleep(100);
      Continue;
    end;

    if LowPrioThread then
    begin
      if DurchLaufAfterReInit > 10000 then
      begin
        Sleep(50);
      end;
    end;

    Inc(Durchlauf);
    Inc(DurchLaufAfterReInit);

    Plan.TransferRefDateToDate();
    Plan.FillPredefinedGames();

    Plan.NeuWuerfeln(TauschTyp, TauschAnzahl, TauschPercent);

    Plan.FillTermine();
    Kosten := Plan.CalculateKosten(OptKosten);

    if (Kosten >= 0) and ((Kosten < OptKosten) or (OptKosten < 0)) then
    begin
{$ifdef TRACE}
      //TraceString('Thread ' + GetCompleteThreadName() + ' bessere Kosten: ' + MyFormatFloat(Kosten));
{$endif}

      ParentCriticalSection.Enter;
      // Nochmal Kosten ermitteln und die Kosten der einzelnen Spiele eintragen
      Kosten := Plan.CalculateKosten(-1);
      Plan.TransferDateToRefDate();
      OptKosten := Kosten;
      RetKosten := Kosten;
      Plan.LastOptimizedThread := GetCompleteThreadName();

      OptPlan.AssignOptimizedDates(Plan);

      ParentCriticalSection.Leave;

    end;

    if ReInitPlanFlag or ReInitPlanDataFlag then
    begin
      DurchLaufAfterReInit := 0;
      ParentCriticalSection.Enter;
      if Assigned(ReInitPlan) then
      begin
        Plan.AssignOptimizedDates(ReInitPlan);
        Plan.CalculateKosten(-1);
        Plan.TransferDateToRefDate();
        OptPlan.AssignOptimizedDates(Plan);

        OptKosten := Plan.CalculateKosten(-1);
        RetKosten := OptKosten;

        FreeAndNil(ReInitPlan);
      end;

      if Assigned(ReInitPlanData) then
      begin
        Plan.AssignPlanData(ReInitPlanData);
        Plan.RemoveNotValidDates();
        Plan.TransferDateToRefDate();
        OptPlan.AssignPlanData(ReInitPlanData);
        OptPlan.RemoveNotValidDates();
        OptPlan.TransferDateToRefDate();

        OptKosten := OptPlan.CalculateKosten(-1);
        // Damit die Kosten im Cache sind
        Plan.TransferDateToRefDate();
        RetKosten := OptKosten;

        FreeAndNil(ReInitPlanData);
{$ifdef TRACE}
        TraceString('Thread ' + GetCompleteThreadName() + ' new Options' + Plan.getOptions.ToString);
{$endif}
      end;

      ReInitPlanFlag := False;
      ReInitPlanDataFlag := False;

      ParentCriticalSection.Leave;
    end;


    Sleep(0);

{$ifopt D+}
    //Sleep(100);
{$ENDIF D+}

  end;

{$ifdef TRACE}
  TraceString('Ende TPlanCalcThread.Execute ' + GetCompleteThreadName());
{$endif}

end;


procedure TPlanCalcThread.SetPaused(const Value: boolean);
begin
  FPaused := Value;
end;

procedure TPlanCalcThread.StartWithLists(ThreadName: String; Plan: TPlan; TauschArt: String; ParentCriticalSection: TCriticalSection);
begin
  RetKosten := -1;

  Self.ThreadName := ThreadName;

  SetThreadType(TauschArt);

  Self.ParentCriticalSection := ParentCriticalSection;

  Self.Plan.Assign(Plan);
  Self.Plan.RemoveNotValidDates();

  OptPlan.Assign(Plan);
  OptPlan.RemoveNotValidDates();
end;


procedure TPlanCalcThread.SetThreadType(TauschArt: String);
begin
  Self.ThreadType := TauschArt;
  LowPrioThread := False;
  TauschTyp := ttNormal;
  TauschAnzahl := 0;
  TauschPercent := 10;

  if TauschArt[1] = 'R' then
  begin
    TauschTyp := ttRaster;
    LowPrioThread := True;
  end
  else
  if TauschArt[1] = 'S' then
  begin
    TauschTyp := ttSpielTag;
    TauschAnzahl := StrToInt(TauschArt[2]);
    TauschPercent := MyStrToFloat(Copy(TauschArt, 4, Length(TauschArt)));
  end
  else
  if TauschArt[1] = 'M' then
  begin
    TauschTyp := ttMannschaft;
    TauschAnzahl := StrToInt(TauschArt[2]);
    TauschPercent := MyStrToFloat(Copy(TauschArt, 4, Length(TauschArt)));
  end
  else
  begin
    TauschTyp := ttNormal;
    TauschAnzahl := 0;
    TauschPercent := MyStrToFloat(TauschArt);
    LowPrioThread := TauschPercent >= 30;
  end;
end;



constructor TPlanCalcThread.Create;
begin
  inherited Create(true);
  IsInit := False;
  Priority := tpLower;
  Plan := TPlan.Create;
  OptPlan := TPlan.Create;

end;


procedure TPlanCalcThread.GetActResult(Plan: TPlan; var DurchLauf: Int64; var Kosten: MyDouble);
begin
  ParentCriticalSection.Enter;
  if Assigned(Plan) then
  begin
    Plan.AssignOptimizedDates(OptPlan);
  end;

  DurchLauf := Self.Durchlauf;

  if ReInitPlanFlag or ReInitPlanDataFlag then
  begin
    // Ich soll eh neu aufsetzen -> also hab ich noch keine Kosten
    Kosten := -1;
  end
  else
  begin
    Kosten := RetKosten;
  end;
  ParentCriticalSection.Leave;
end;


procedure TPlanCalcThread.DoReInitPlanData(Plan: TPlan; const ReinitString: String);
begin
  ParentCriticalSection.Enter;

  FreeAndNil(ReInitPlanData);
  ReInitPlanData := TPlan.Create;
  ReInitPlanData.Assign(Plan);
  ReInitPlanDataFlag := True;
  if ReinitString <> '' then
  begin
    Self.ReinitString := ReInitString;
  end;
  ParentCriticalSection.Leave;

end;

procedure TPlanCalcThread.DoReInitPlan(Plan: TPlan; const ReinitString: String);
begin
  ParentCriticalSection.Enter;

  FreeAndNil(ReInitPlan);
  ReInitPlan := TPlan.Create;
  ReInitPlan.Assign(Plan);
  ReInitPlanFlag := True;
  if ReinitString <> '' then
  begin
    Self.ReinitString := ReInitString;
  end;
  ParentCriticalSection.Leave;

end;

{ TPlan }


procedure TPlan.Assign(Source: TPlan);
var
  SourceGames: TSchedule;
begin
  AssignPlanData(Source);

  SourceGames := TSchedule.Create();
  Source.getSchedule(SourceGames);
  AssignSchedule(SourceGames);
  SourceGames.Free;

end;


procedure TPlan.AssignPlanData(Source: TPlan);
var
  SourceGames: TSchedule;
  FreeGameDaysPair: TPair<TDateTime, String>;
  value: integer;
begin

  // Games merken, die werden nicht geklont
  SourceGames := TSchedule.Create();
  getSchedule(SourceGames);


  gender := Source.gender;

  // Vor dem Assign der Mannschaften die Referenzen auf die Mannschaften entfernen, sind in den Games
  ClearMannschaftGameRefs();
  gamesAllowed.Clear;
  gamesNotAllowed.Clear;

  Mannschaften.Assign(Source.Mannschaften, Self);

  AddSisterTeamsInPlan();

  DateBeginRueckrundeInXML := Source.DateBeginRueckrundeInXML;
  DateBegin := Source.DateBegin;
  DateEnd := Source.DateEnd;
  PlanName := Source.PlanName;
  LastOptimizedThread := Source.LastOptimizedThread;
  PlanId := Source.PlanId;
  AssignOptions(Source.getOptions());

  assignPredefinedGames(Source.predefinedGames);
  assignExistingScheduleGames(Source.existingSchedule);

  FreeGameDays.Clear;
  for FreeGameDaysPair in Source.FreeGameDays do
  begin
    FreeGameDays.Add(FreeGameDaysPair.Key, FreeGameDaysPair.Value);
  end;

  MandatoryGameDays.Assign(Source.MandatoryGameDays);


  RankingIndexe.Clear;
  for value in Source.RankingIndexe do
  begin
    if (value >= 0) and (value < Mannschaften.Count) then
    begin
      RankingIndexe.Add(value);
    end;
  end;

  AssignSchedule(SourceGames);
  SourceGames.Free;

  FillPredefinedGames();

  AssignOptions(Source.getOptions());

  ClearCache();


end;



procedure TPlan.GenerateRaster();
var
  MannschaftenTemp: TObjectList<TMannschaft>;
  Mannschaft: TMannschaft;
  I: Integer;
  i1, i2: Integer;
  SpieltagIndexe: TList<Integer>;
  RoundInfo: TRoundInfo;
begin
  if Self.Mannschaften.Count < 2 then
  begin
    exit;
  end;

  for i:=0 to gamesAllowed.Count - 1 do
  begin
    if not gamesAllowed[i].FixedDate  then
    begin
      gamesAllowed[i].setDate(0.0, [], 0, False, False, nil);
    end;
  end;

  if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
  begin
    MarkNotNecessaryGames();
  end;


  MannschaftenTemp := TObjectList<TMannschaft>.Create(False);
  for Mannschaft in Self.Mannschaften do
  begin
    MannschaftenTemp.Add(Mannschaft);
  end;

  //Verwrfeln
  for I := 0 to 100 do
  begin
    i1 := Random(MannschaftenTemp.Count);
    i2 := Random(MannschaftenTemp.Count);
    if i2 <> i1 then
    begin
      MannschaftenTemp.Exchange(i1, i2);
    end;
  end;

  SpieltagIndexe := TList<Integer>.Create;
  for i := 0 to MannschaftenTemp.Count - 2 do
  begin
    SpieltagIndexe.Add(i);
  end;

  for I := 0 to 100 do
  begin
    i1 := Random(SpieltagIndexe.Count);
    i2 := Random(SpieltagIndexe.Count);
    if i2 <> i1 then
    begin
      SpieltagIndexe.Exchange(i1, i2);
    end;
  end;

  for i:=0 to Rounds.Count-1 do
  begin
    RoundInfo := Rounds[i];
    CreateRasterOneRound(MannschaftenTemp, SpieltagIndexe, RoundInfo.DateFrom, RoundInfo.DateTo, i mod 2 = 0);
  end;

  MannschaftenTemp.Free;
  SpieltagIndexe.Free;
end;


procedure TPlan.CreateRasterOneRound(Mannschaften: TObjectList<TMannschaft>; SpieltagIndexe: TList<Integer>; DateFrom: TDateTime; DateTo: TDateTime; RueckRunde: boolean);
var
  SpieltageMondays: TList<TDateTime>;
  Spieltag: Integer;
  Mannschaft, Mannschaft2, FillerMannschaft: TMannschaft;
  m, m2: integer;
  MannschaftCount: integer;
  RueckSpiel: boolean;
  RasterValue: integer;
begin
  SpieltageMondays := TList<TDateTime>.Create;

  getSpieltagMondays(SpieltageMondays, DateFrom, DateTo);

  // Auf die ntige Menge zufllig verkleinern
  while SpieltageMondays.Count > Mannschaften.Count-1 do
  begin
    SpieltageMondays.Delete(Random(SpieltageMondays.Count));
  end;

  // Raster siehe https://en.wikipedia.org/wiki/Round-robin_tournament#Scheduling_algorithm

  FillerMannschaft := nil;
  MannschaftCount := Mannschaften.Count;
  if Mannschaften.Count mod 2 = 0 then
  begin
    FillerMannschaft := Mannschaften[Mannschaften.Count-1];
    Dec(MannschaftCount);
  end;


  for m := 0 to MannschaftCount-1 do
  begin
    Mannschaft := Mannschaften[m];

    for Spieltag := 0 to SpieltageMondays.Count-1 do
    begin
      RasterValue := SpieltagIndexe[Spieltag];
      m2 := 0 - RasterValue - m;

      while m2 < 0 do
      begin
        Inc(m2, MannschaftCount);
      end;

      if(m2 >= m) then
      begin

        if m2 = m then
        begin
          Mannschaft2 := FillerMannschaft;
        end
        else
        begin
          Mannschaft2 := Mannschaften[m2];
        end;

        if Mannschaft2 <> nil then
        begin
          RueckSpiel := RueckRunde;

          if (Spieltag mod 2 = 0) then
          begin
            RueckSpiel := not RueckSpiel;
          end;

          if Spieltag >= (SpieltageMondays.Count / 2) then
          begin
            RueckSpiel := not RueckSpiel;
          end;

          if RueckSpiel then
          begin
            ScheduleGameForRaster(Mannschaft, Mannschaft2, SpieltageMondays[Spieltag]);
          end
          else
          begin
            ScheduleGameForRaster(Mannschaft2, Mannschaft, SpieltageMondays[Spieltag]);
          end;
        end;
      end;
    end;
  end;

  SpieltageMondays.Free;
end;


procedure TPlan.ScheduleGameForRaster(HeimMannschaft: TMannschaft; GastMannschaft: TMannschaft; WocheMonday: TDateTime);
var
  WunschTermin: TWunschTermin;
  game: TGame;
  games: TGameList;
begin
  for WunschTermin in HeimMannschaft.WunschTermine do
  begin
    if MondayBefore(WunschTermin.Date) = WocheMonday then
    begin
      games := findGames(HeimMannschaft.teamId, GastMannschaft.teamId);

      if Assigned(games) then
      begin
        for game in games do
        begin
          if not game.NotNecessary then
          begin
            if not game.DateValid then
            begin
              if IsInRoundToGenerate(WunschTermin.Date) then
              begin
                if TerminIsValid(WunschTermin, Game) then
                begin
                  game.setDate(WunschTermin.Date, WunschTermin.Options, WunschTermin.ParallelGames, False, False, WunschTermin);

                  if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
                  begin
                    MarkSecondGameAsUnNecessary(game);
                  end;
                  break;
                end;
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;








procedure TPlan.NeuWuerfeln(TauschTyp: TTauschTyp; TauschAnzahl: integer; TauschPercent: MyDouble);
var
  i, j, k: integer;
  TempGames: TGameList;
  Mannschaft: TMannschaft;
  NumSpielTage: integer;
begin
  if TauschTyp = ttRaster  then
  begin
    GenerateRaster();
  end
  else
  if TauschTyp = ttMannschaft  then
  begin
    if Mannschaften.Count > 0 then
    begin
      TempGames := TGameList.Create(False);

      for i:= 1 to TauschAnzahl do
      begin
        j := min(Mannschaften.Count-1, Random(Mannschaften.Count));
        Mannschaft := Mannschaften[j];
        TempGames.AddRange(Mannschaft.gamesRef);
      end;

      NeuWuerfelnForGames(TempGames, TauschPercent, false);
      TempGames.Free;
    end;
  end
  else
  if TauschTyp = ttSpielTag  then
  begin
    if Mannschaften.Count > 0 then
    begin
      NumSpielTage := Mannschaften[0].gamesRef.Count;
      TempGames := TGameList.Create(False);

      for i:= 1 to TauschAnzahl do
      begin
        j := min(NumSpielTage-1, Random(NumSpielTage));
        for k := 0 to Mannschaften.Count-1 do
        begin
          if j < Mannschaften[k].gamesRef.Count then
          begin
            TempGames.Add(Mannschaften[k].gamesRef[j]);
          end;
        end;
      end;

      NeuWuerfelnForGames(TempGames, TauschPercent, True);
      TempGames.Free;
    end;
  end
  else
  begin
    NeuWuerfelnForGames(gamesAllowed, TauschPercent, True);
  end;
end;


procedure TPlan.ClearGameList(gameList: TGameList);
var
  game: TGame;
begin
  if Assigned(gameList) then
  begin
    for game in gameList do
    begin
      if not game.FixedDate then
      begin
        game.setDate(0.0, [], 0, False, False, nil);
      end;
    end;
  end;
end;


procedure TPlan.ClearAllGamesOfMannschaften(MannschaftHeim, MannschaftGast: TMannschaft);
var
  gameList: TGameList;
begin
  gameList := findGames(MannschaftHeim.teamId, MannschaftGast.teamId);

  ClearGameList(gameList);

  gameList := findGames(MannschaftGast.teamId, MannschaftHeim.teamId);

  ClearGameList(gameList);

end;


procedure TPlan.ClearCache;
var
  i: integer;
begin
  FreeAndNil(dataCache);
  for i := 0 to Mannschaften.Count-1 do
  begin
    Mannschaften[i].ClearCache();
  end;
end;


procedure TPlan.NeuWuerfelnForGames(Games: TGameList; Percent: MyDouble; WithRueckGame: boolean);
var
  i: integer;
  Ziel, Entfernt: integer;
  TempGames: TGameList;
  game: TGame;
  koppelHomeGame: TGame;
  koppelGuestGame: TGame;
  AuswaertsKoppelGames: TGameList;
begin
  if Percent = 0 then
  begin
    Exit;
  end;

  if Percent = 100 then
  begin
    for i:=0 to games.Count - 1 do
    begin
      if not games[i].FixedDate then
      begin
        games[i].setDate(0.0, [], 0, False, False, nil);
      end;
    end;
  end
  else
  begin
    TempGames := TGameList.Create(False);

    for i:=0 to games.Count - 1 do
    begin
      TempGames.Add(games[i]);
    end;

    Ziel := Max(1, Min(games.Count, Trunc(games.Count * Percent / 100.0)));

    Entfernt := 0;
    while Entfernt < Ziel do
    begin
      if TempGames.Count = 0 then
      begin
        Break;
      end;

      i := min(TempGames.Count-1, Random(TempGames.Count));

      if TempGames[i].Date = 0.0 then
      begin
        TempGames.Delete(i);
        continue;
      end;

      if WithRueckGame and (HasKoppelTermine or HasAuswaertsKoppelTermine) then
      begin
        koppelHomeGame := TempGames[i].MannschaftHeim.getKoppeledGame(TempGames[i]);

        if Assigned(koppelHomeGame) then
        begin
          if not koppelHomeGame.FixedDate then
          begin
            koppelHomeGame.setDate(0.0, [], 0, False, False, nil);
          end;
        end;

        if TempGames[i].MannschaftGast.AuswaertsKoppel.Count > 0 then
        begin
          AuswaertsKoppelGames := TempGames[i].MannschaftGast.CreateAuswaertsKoppeledGamesDependenciesList(Self, TempGames[i].MannschaftHeim);

          if Assigned(AuswaertsKoppelGames) then
          begin
            for koppelGuestGame in AuswaertsKoppelGames do
            begin
              if not koppelGuestGame.FixedDate then
              begin
                koppelGuestGame.setDate(0.0, [], 0, False, False, nil);
              end;
            end;
            AuswaertsKoppelGames.Free;
          end;
        end;

      end;

      if not TempGames[i].FixedDate then
      begin
        TempGames[i].setDate(0.0, [], 0, False, False, nil);
      end;

      game := TempGames[i];
      TempGames.Delete(i);

      Inc(Entfernt);

      if WithRueckGame then
      begin

        // Rckspiel auch gleich entfernen
        ClearAllGamesOfMannschaften(game.MannschaftHeim, game.MannschaftGast);
      end;

    end;

    TempGames.Free;
  end;

end;





procedure TPlan.AssignOptimizedDates(Source: TPlan);
var
  i: Integer;
  HeimId, GastId: string;
  games: TGameList;
  game: TGame;
  WunschTermin: TWunschTermin;
begin
  clearAllDates();
  for i:=0 to Source.gamesAllowed.Count-1 do
  begin
    HeimId := Source.gamesAllowed[i].MannschaftHeim.teamId;
    GastId := Source.gamesAllowed[i].MannschaftGast.teamId;

    games := findGames(HeimId, GastId);

    if(Assigned(games)) then
    begin
      for game in games do
      begin
        if (not game.DateValid) then
        begin
          WunschTermin := game.MannschaftHeim.WunschTermine.Find(Source.gamesAllowed[i].Date);
          game.setDate(Source.gamesAllowed[i].Date, Source.gamesAllowed[i].DateOptions, Source.gamesAllowed[i].DateMaxHome, Source.gamesAllowed[i].FixedDate, Source.gamesAllowed[i].NotNecessary, WunschTermin);
          break;
        end;
      end;
    end
    else
    begin
      raise Exception.Create('Plan.Assign: game wurde nicht gefunden');
    end;
  end;
  LastOptimizedThread := Source.LastOptimizedThread;

  if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
  begin
    MarkNotNecessaryGames();
  end;


  SortMannschaftsDates();
end;

function TPlan.isValidDateForWunschtermin(Date: TDateTime): boolean;
begin
  result := true;
  if getOptions.RoundPlaning = rpHalfRound then
  begin
    if (Date < DateBegin) or (Date > DateEnd)then
    begin
      result := false;
    end;
  end;
end;


procedure TPlan.ModifyTermineFromOptions();
var
  Mannschaft: TMannschaft;
  Wunschtermin: TWunschTermin;
  i: integer;
  NewList: TList<TWunschTermin>;
  NewElem2: TWunschTermin;
begin
  // Wunschtermine, die fr die Auswaertkoppelwnsche angelegt wurden entfernen
  for Mannschaft in Mannschaften do
  begin
    for i:= Mannschaft.SperrTermine.Count-1 downto 0 do
    begin
      if not isValidDateForWunschtermin(Mannschaft.SperrTermine[i]) then
      begin
        Mannschaft.SperrTermine.Delete(i);
      end
    end;
  end;


  // Wunschtermine, die fr die Auswaertkoppelwnsche angelegt wurden entfernen
  for Mannschaft in Mannschaften do
  begin
    for i:= Mannschaft.WunschTermine.Count-1 downto 0 do
    begin
      Wunschtermin := Mannschaft.WunschTermine[i];
      if (woAuswaertsKoppelSecondTime in Wunschtermin.Options) or (woKoppelSecondTime in Wunschtermin.Options) then
      begin
        Mannschaft.WunschTermine.Delete(i);
      end
      else
      if not isValidDateForWunschtermin(Wunschtermin.Date) then
      begin
        Mannschaft.WunschTermine.Delete(i);
      end
    end;
  end;

  ClearCache();

  // Heimkoppeltermine initialisieren
  for Mannschaft in Mannschaften do
  begin
    NewList := TList<TWunschTermin>.Create;
    for Wunschtermin in Mannschaft.WunschTermine do
    begin
      if OptionIsKoppelSingleDate(Wunschtermin.Options) then
      begin
        // Koppeltermin an einem Tag -> den zweiten Termin anlegen
        NewElem2 := TWunschTermin.Create;
        NewList.Add(NewElem2);
        NewElem2.Assign(Wunschtermin);
        NewElem2.Options := NewElem2.Options + [woKoppelSecondTime];
        NewElem2.Date := Wunschtermin.KoppelSecondTime;
      end;
    end;

    if NewList.Count > 0 then
    begin
      Mannschaft.WunschTermine.AddRange(NewList);
      Mannschaft.WunschTermine.Sort;
    end;

    NewList.Free;
  end;



  // Zusatztermine fr die Auswaertskoppelwnsche
  for Mannschaft in Mannschaften do
  begin
    NewList := TList<TWunschTermin>.Create;
    for Wunschtermin in Mannschaft.WunschTermine do
    begin
      if woAuswaertsKoppelHasSecondTime in Wunschtermin.Options then
      begin
        // Und jetzt noch den zweiten Termin
        NewElem2 := TWunschTermin.Create;
        NewList.Add(NewElem2);
        NewElem2.Assign(Wunschtermin);
        NewElem2.Options := NewElem2.Options + [woAuswaertsKoppelSecondTime];
        NewElem2.Date := Wunschtermin.KoppelSecondTime;
      end;
    end;

    if NewList.Count > 0 then
    begin
      Mannschaft.WunschTermine.AddRange(NewList);
      Mannschaft.WunschTermine.Sort;
    end;

    NewList.Free;
  end;


  ClearCache();

end;



procedure TPlan.MarkNotNecessaryGames();
var
  game: TGame;
begin
  for game in gamesAllowed do
  begin
    game.setDate(game.Date, game.DateOptions, game.DateMaxHome, game.FixedDate, false, game.FWunschTermin);
  end;

  if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
  begin
    for game in gamesAllowed do
    begin
      if (getOptions.RoundPlaning = rpCorona) then
      begin
        if game.DateValid (*and (not IsInRoundToGenerate(game.Date))*) then
        begin
          MarkSecondGameAsUnNecessary(game);
        end;
      end
      else
      if IsInRoundToGenerate(game.Date) then
      begin
        MarkSecondGameAsUnNecessary(game);
      end;
    end;
  end;
end;


procedure TPlan.AssignOptions(Options: TCalculateOptions);
var
  SourceGames: TSchedule;
begin
  SourceGames := TSchedule.Create();
  getSchedule(SourceGames);

  FOptions.Assign(Options);

  ModifyTermineFromOptions();

  AssignSchedule(SourceGames);

  MarkNotNecessaryGames();

  SourceGames.Free;

  ClearCache();
end;

procedure TPlan.Clear;
begin
  Mannschaften.Clear;
  gamesAllowed.Clear;
  gamesNotAllowed.Clear;
  existingSchedule.Clear;
  predefinedGames.Clear;
end;

constructor TPlan.Create;
begin
  inherited Create();

  Mannschaften := TMannschaftList.Create;

  gamesAllowed := TGameList.Create(true);
  gamesNotAllowed := TGameList.Create(true);

  cachedGameList := TObjectDictionary<String, TGameList>.Create([doOwnsValues]);

  FOptions := TCalculateOptions.Create;

  FRounds := TObjectList<TRoundInfo>.Create();

  predefinedGames := TGameList.Create(True);

  existingSchedule := TGameList.Create(True);

  FreeGameDays := TDictionary<TDateTime, String>.Create();

  MandatoryGameDays := TPlanMandatoryTimeList.Create;

  RankingIndexe := TList<Integer>.Create;

end;


procedure TPlan.getSpieltagMondays(SpieltageMondays: TList<TDateTime>; DateFrom, DateTo: TDateTime; OnlyWunschTermine: boolean = true);
var
  Mannschaft: TMannschaft;
  WunschTermin: TWunschTermin;
  Monday: TDateTime;
  Game: TGame;
begin
  SpieltageMondays.Clear;
  for Mannschaft in Mannschaften do
  begin
    for WunschTermin in Mannschaft.WunschTermine do
    begin
      if (WunschTermin.Date >= DateFrom) and (WunschTermin.Date <= DateTo) then
      begin
        Monday := MondayBefore(WunschTermin.Date);

        if not SpieltageMondays.Contains(Monday) then
        begin
          SpieltageMondays.Add(Monday);
        end;
      end;
    end;
  end;

  if not OnlyWunschTermine then
  begin
    for game in gamesAllowed do
    begin
      if game.DateValid then
      begin
        Monday := MondayBefore(game.Date);

        if not SpieltageMondays.Contains(Monday) then
        begin
          SpieltageMondays.Add(Monday);
        end;
      end;
    end;

    for game in gamesNotAllowed do
    begin
      if game.DateValid then
      begin
        Monday := MondayBefore(game.Date);

        if not SpieltageMondays.Contains(Monday) then
        begin
          SpieltageMondays.Add(Monday);
        end;
      end;
    end;

  end;

  SpieltageMondays.Sort;
end;


function TPlan.CalcHas60KilometerValues(): boolean;
var
  Mannschaft: TMannschaft;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    if Mannschaft.noWeekGame.Count > 0 then
    begin
      Result := True;
    end;
  end;
end;


function TPlan.CalcHasKoppelTermine(): boolean;
var
  Mannschaft: TMannschaft;
  Wunschtermin: TWunschTermin;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    for Wunschtermin in Mannschaft.WunschTermine do
    begin
      if OptionIsKoppel(Wunschtermin.Options) then
      begin
        Result := True;
      end;
    end;
  end;
end;


function TPlan.CalcHasMandatoryGameDays: boolean;
begin
  Result := MandatoryGameDays.Count > 0;
end;

function TPlan.CalcNumWunschDays: integer;
var
  DateList: TList<Integer>;
  Mannschaft: TMannschaft;
  Termin: TWunschTermin;
  IntDate: integer;
  I: Integer;
  Found: boolean;
begin
  DateList := TList<Integer>.Create;

  Result := 0;

  for Mannschaft in Mannschaften do
  begin
    for Termin in Mannschaft.WunschTermine do
    begin
      IntDate := Trunc(Termin.Date);
      Found := False;
      for I in DateList do
      begin
        if I = IntDate then
        begin
          Found := True;
          break;
        end;
      end;

      if not Found then
      begin
        Result := Result + 1;
        DateList.Add(IntDate);
      end;
    end;
  end;

  DateList.Free;
end;

function TPlan.CalcHasAuswaertsKoppelTermine(): boolean;
var
  Mannschaft: TMannschaft;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    if Mannschaft.AuswaertsKoppel.Count > 0 then
    begin
      Result := True;
    end;
  end;
end;


function TPlan.CalcHasForceSameHomeGames(): boolean;
var
  Mannschaft: TMannschaft;
  sisterGames: TSisterGames;
  sisterGame: TSisterGame;
  sisterGamesMap: TArray<TSisterGames>;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    sisterGamesMap := Mannschaft.sisterGames.toArray();
    for sisterGames in sisterGamesMap do
    begin
      for sisterGame in sisterGames do
      begin
        if sisterGame.ForceParallelHomeGames then
        begin
          Result := True;
        end;
      end;
    end;
  end;
end;


function TPlan.CalcHasAusweichTermine: boolean;
var
  Mannschaft: TMannschaft;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    if Mannschaft.AusweichTermine.Count > 0 then
    begin
      Result := True;
    end;
  end;
end;

function TPlan.Has60KilometerValues: boolean;
begin
  Result := getCache().Has60KilometerValues;
end;

function TPlan.HasAuswaertsKoppelTermine: boolean;
begin
  Result := getCache().HasAuswaertsKoppelTermine;
end;

function TPlan.HasForceSameHomeGames: boolean;
begin
  Result := getCache().HasForceSameHomeGames;
end;

function TPlan.HasAusweichTermine: boolean;
begin
  Result := getCache().HasAusweichTermine;
end;

function TPlan.HasKoppelTermine: boolean;
begin
  Result := getCache().HasKoppelTermine;
end;

function TPlan.HasMandatoryGameDays: boolean;
begin
  Result := getCache().HasMandatoryGameDays;
end;

function TPlan.GetNumWunschDays: integer;
begin
  Result := getCache().NumWunschDays;
end;


function TPlan.HasPossibleKoppelTermine: boolean;
var
  Mannschaft: TMannschaft;
  WunschTermin1, WunschTermin2: TWunschTermin;
begin
  Result := False;
  for Mannschaft in Mannschaften do
  begin
    for WunschTermin1 in Mannschaft.WunschTermine do
    begin
      if OptionIsKoppelSingleDate(WunschTermin1.Options) then
      begin
        Result := True;
      end;

      for WunschTermin2 in Mannschaft.WunschTermine do
      begin
        if 1 = Abs(Trunc(WunschTermin1.Date) - Trunc(WunschTermin2.Date)) then
        begin
          Result := True;
        end;
      end;
    end;
  end;
end;

function TPlan.HasRanking: boolean;
begin
  Result := RankingIndexe.Count > 0;
end;

function TPlan.HasValuesForType(KostenType: TMannschaftsKostenType): boolean;
begin
  Result := True;

  if KostenType = mkt60Kilometer then
  begin
    Result := Has60KilometerValues;
  end;

  if KostenType = mktKoppelTermine then
  begin
    Result := HasKoppelTermine;
  end;

  if KostenType = mktAusweichTermine then
  begin
    Result := HasAusweichTermine;
  end;

  if KostenType = mktAuswaertsKoppelTermine then
  begin
    Result := HasAuswaertsKoppelTermine;
  end;

  if KostenType = mktMandatoryGames then
  begin
    Result := HasMandatoryGameDays;
  end;

  if KostenType = mktRanking then
  begin
    Result := HasRanking();
  end;

  if KostenType = mktForceSameHomeGames then
  begin
    Result := HasForceSameHomeGames();
  end;

end;

function TPlan.IsInRoundToGenerate(Date: TDateTime): boolean;
var
  RoundNo: integer;
begin
  Result := True;
  if getOptions.RoundPlaning <> rpBoth then
  begin
    RoundNo := GetRoundNumberByDate(Date);
    if RoundNo >= 0 then
    begin
      if getOptions().DoubleRound then
      begin
        if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly] then
        begin
          if RoundNo > 1 then
          begin
            Result := False;
          end;
        end
        else
        if getOptions.RoundPlaning in [rpSecondOnly, rpCorona] then
        begin
          if RoundNo <= 1 then
          begin
            Result := False;
          end;
        end;
      end
      else
      begin
        if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly] then
        begin
          if RoundNo > 0 then
          begin
            Result := False;
          end;
        end
        else
        if getOptions.RoundPlaning in [rpSecondOnly, rpCorona] then
        begin
          if RoundNo <= 0 then
          begin
            Result := False;
          end;
        end;
      end;
    end
    else
    begin
      Result := False;
    end;
  end;
end;


function TPlan.IsRoundToGenerate(Round: TRoundInfo): boolean;
var
  MidDate: TDateTime;
begin
  Result := True;
  if getOptions.RoundPlaning <> rpBoth then
  begin
    MidDate := Round.DateFrom + ((Round.DateTo - Round.DateFrom) / 2);
    Result := IsInRoundToGenerate(MidDate);
  end;
end;


procedure TPlan.CreateRoundInfo;
var
  RoundInfo: TRoundInfo;
  DateMid1: TDateTime;
  DateMid2: TDateTime;
begin
  FRounds.Clear;
  if getOptions().RoundPlaning = rpHalfRound then
  begin
    if getOptions().DoubleRound then
    begin
      DateMid1 := getOptions().getMidDate1(Self);
      DateMid2 := getOptions().getMidDate2(Self);

      if DateMid1 < DateBegin then
      begin
        DateMid1 := DateMid2;
      end;


      RoundInfo := TRoundInfo.Create;
      RoundInfo.DateFrom := DateBegin;
      RoundInfo.DateTo := DateMid1;
      FRounds.Add(RoundInfo);

      RoundInfo := TRoundInfo.Create;
      RoundInfo.DateFrom := DateMid1;
      RoundInfo.DateTo := DateEnd;
      FRounds.Add(RoundInfo);
    end
    else
    begin
      RoundInfo := TRoundInfo.Create;
      RoundInfo.DateFrom := DateBegin;
      RoundInfo.DateTo := DateEnd;
      FRounds.Add(RoundInfo);
    end;
  end
  else
  if getOptions().DoubleRound then
  begin
    DateMid1 := getOptions().getMidDate1(Self);
    DateMid2 := getOptions().getMidDate2(Self);

    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := GetFirstDate()-100;
    RoundInfo.DateTo := DateMid1;
    FRounds.Add(RoundInfo);

    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := DateMid1;
    RoundInfo.DateTo := DateBeginRueckrundeInXML;
    FRounds.Add(RoundInfo);

    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := DateBeginRueckrundeInXML;
    RoundInfo.DateTo := DateMid2;
    FRounds.Add(RoundInfo);

    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := DateMid2;
    RoundInfo.DateTo := GetLastDate + 100;
    FRounds.Add(RoundInfo);

  end
  else
  begin
    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := GetFirstDate()-100;
    RoundInfo.DateTo := DateBeginRueckrundeInXML;
    FRounds.Add(RoundInfo);

    RoundInfo := TRoundInfo.Create;
    RoundInfo.DateFrom := DateBeginRueckrundeInXML;
    RoundInfo.DateTo := GetLastDate + 100;
    FRounds.Add(RoundInfo);
  end;
end;

function TPlan.DateIsAllowedForWeekendGames(Date: TDateTime): boolean;
var
  WeekDay: Integer;
begin
  WeekDay := DayOfWeek(Date);

  if (WeekDay = 7) or (WeekDay = 1) then
  begin
    // Samsatg oder Sonntag
    Result := True;
  end
  else
  if (WeekDay = 6) then
  begin
    // Freitag
    if getOptions().FreitagIsAllowedBy60km then
    begin
      Result := True;
    end
    else
    begin
      Result := False;
    end;
  end
  else
  begin
    Result := False;
  end;
end;

destructor TPlan.Destroy;
begin
  Mannschaften.Free;

  gamesAllowed.Free;

  gamesNotAllowed.Free;

  cachedGameList.Free;

  FOptions.Free;

  FRounds.Free;

  predefinedGames.Free;

  existingSchedule.Free;

  FreeGameDays.Free;

  MandatoryGameDays.Free;

  RankingIndexe.Free;

  dataCache.Free;

  inherited;
end;


function DayOfWeekAsString(Date: TDateTime): String;
var
  Day: Integer;
begin
  Day := DayOfWeek(Date);

  case(Day) of
    1: Result := 'Sonntag';
    2: Result := 'Montag';
    3: Result := 'Dienstag';
    4: Result := 'Mittwoch';
    5: Result := 'Donnerstag';
    6: Result := 'Freitag';
    7: Result := 'Samstag';
  else
    raise Exception.Create('Ungltiger Tag');
  end;
end;

function FormatDateForExport(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'dd.mm.yyyy', Date);
end;

function FormatBoolForExport(Value: boolean): string;
begin
  if Value then
  begin
    Result := 'true';
  end
  else
  begin
    Result := 'false';
  end;
end;


procedure TPlan.SaveScheduleToCsv(FileName: String);
var
  Data: TStrings;
  i: Integer;
  S: String;
  VorRueck: String;
  Date: TDateTime;
  Location: String;
  allGames: TSchedule;
  MannschaftHeim, MannschaftGast: TMannschaft;
begin
  Data := TStringList.Create;

  // Header
  Data.Add('Nr.;Vor/Rck;Tag;;Datum;;Uhrzeit;HeimVereinNr;Heim-Mannschaft;GastVereinNr;Gast-Mannschaft;HeimMannschaftNr;GastMannschaftNr;Ergebnisse;Spiellokal');

  allGames := TSchedule.Create();

  getSchedule(allGames);
  allGames.SortByDate;

  for i := 0 to allGames.Count-1 do
  begin
    if IsInRoundToGenerate(allGames[i].Date) then
    begin
      S := IntToStr(i+1) + ';';

      //VorRueck := IntToStr(max(0, GetRoundNumberByDate(allGames[i].Date)));

      // Auch bei Doppelrunden ist das zweite Spiel in Runde 0
      VorRueck := '0';
      if getOptions.RoundPlaning <> rpHalfRound then
      begin
        if allGames[i].Date > DateBeginRueckrundeInXML then
        begin
          VorRueck := '1';
        end;
      end;

      S := S + VorRueck + ';';

      if not allGames[i].DateValid then
      begin
        Date := DateBegin;
      end
      else
      begin
        Date := allGames[i].Date;
      end;

      MannschaftHeim := Mannschaften.FindByName(allGames[i].MannschaftHeimName);
      MannschaftGast := Mannschaften.FindByName(allGames[i].MannschaftGastName);

      Location := allGames[i].Location;

      if Assigned(MannschaftHeim) and Assigned(MannschaftGast) then
      begin
        S := S + DayOfWeekAsString(Date) + ';;';
        S := S + FormatDateForExport(Date) + ';;';
        S := S + FormatTimeForExport(Date) + ';';
        S := S + MannschaftHeim.clubID + ';';
        S := S + MannschaftHeim.teamName + ';';
        S := S + MannschaftGast.clubID + ';';
        S := S + MannschaftGast.teamName + ';';
        S := S + MannschaftHeim.teamId + ';';
        S := S + MannschaftGast.teamId + ';';
        S := S + ';';
        S := S + Location + ';';

        Data.Add(S)
      end;
    end;

  end;


  Data.SaveToFile(FileName);
  Data.Free;

  allGames.free;
end;


procedure TPlan.SaveScheduleToXML(FileName: String);
var
  PlanData: TPlanData;
  ParentElem: TDataObject;
  gameElem: TDataObject;
  game: TScheduleGame;
  allGames: TSchedule;
  MannschaftHeim, MannschaftGast: TMannschaft;
begin
  PlanData := TPlanData.Create;

  ParentElem := TDataObject.Create(nSchedule, PlanData.root);

  allGames := TSchedule.Create();

  getSchedule(allGames);
  allGames.SortByDate;

  for game in allGames do
  begin
    MannschaftHeim := Mannschaften.FindByName(game.MannschaftHeimName);
    MannschaftGast := Mannschaften.FindByName(game.MannschaftGastName);

    if Assigned(MannschaftHeim) and Assigned(MannschaftGast) then
    begin
      gameElem := TDataObject.Create(nGame, ParentElem);
      gameElem.SetAsDateTime(aDateTime, game.Date);
      gameElem.SetAsString(aHomeTeamName, MannschaftHeim.teamName);
      gameElem.SetAsString(aGuestTeamName, MannschaftGast.teamName);
    end;
  end;

  PlanData.SaveToXML(FileName);
  PlanData.Free;
  allGames.Free;
end;





function TPlan.findGames(const HeimId, GastId: String): TGameList;
var
  Key: String;
  Valid: boolean;
begin
  Key := HeimId + '|' + GastId;

  Valid := cachedGameList.TryGetValue(Key, Result);

  if not Valid then
  begin
    Result := nil;
  end;
end;


procedure TPlan.Load(PlanData: TPlanData);
var
  allGames: TGameList;
  DataElem: TDataObject;
begin
  Clear();

  gender := PlanData.root.GetAsString(aGender);

  DateBeginRueckrundeInXML := PlanData.root.GetAsDate(aMid);
  DateBegin := PlanData.root.GetAsDate(aFrom);
  DateEnd := PlanData.root.GetAsDate(aUntil);
  PlanName := PlanData.root.GetAsString(aName);
  PlanId := PlanData.root.GetAsString(aID);

  loadTeams(PlanData.root, PlanData);

  RemoveNotValidSisterGames();

  AddSisterTeamsInPlan();

  GenerateGames();

  predefinedGames.Clear;
  DataElem := PlanData.root.FindChild(nPredefinedGames);
  if Assigned(DataElem) then
  begin
    allGames := TGameList.Create(True);
    LoadGameDates(allGames, DataElem, PlanData);
    assignPredefinedGames(allGames);
    allGames.Free;
  end;

  LoadNoGameDays(PlanData.root);

  LoadMandatoryGameDays(PlanData.root);

  LoadRanking(PlanData.root);

  existingSchedule.Clear;
  DataElem := PlanData.root.FindChild(nExistingSchedule);
  if Assigned(DataElem) then
  begin
    allGames := TGameList.Create(True);
    LoadGameDates(allGames, DataElem, PlanData);
    AssignGames(allGames);
    assignExistingScheduleGames(allGames);
    allGames.Free;
  end;

  ClearCache();
end;

procedure TPlan.LoadScheduleFromXml(FileName: String);
var
  allGames: TGameList;
  PlanData: TPlanData;
  DataElem: TDataObject;
begin
  PlanData := TPlanData.Create;
  PlanData.LoadFromXML(FileName);

  allGames := TGameList.Create(True);

  DataElem := PlanData.root.FindChild(nSchedule);
  if Assigned(DataElem) then
  begin
    LoadGameDates(allGames, DataElem, PlanData);
  end;

  AssignGames(allGames);
  allGames.Free;
  PlanData.Free;

end;

procedure TPlan.AddSisterTeamsInPlan();
var
  Mannschaft1, Mannschaft2: TMannschaft;
begin
  for Mannschaft1 in Mannschaften do
  begin
    Mannschaft1.SisterTeamsInPlan.Clear;
    for Mannschaft2 in Mannschaften do
    begin
      if Mannschaft1 <> Mannschaft2 then
      begin
        if Mannschaft1.clubID = Mannschaft2.clubID then
        begin
          Mannschaft1.SisterTeamsInPlan.Add(Mannschaft2);
        end;
      end;
    end;
  end;
end;


procedure TPlan.RemoveNotValidDates();
var
  Game: TGame;
begin
  ClearMannschaftWunschTerminRefs();
  ClearMannschaftGameRefs();
  gamesNotAllowed.Clear();

  GenerateMannschaftsRefGames();

  for Game in gamesAllowed do
  begin
    if not GameIsValid(Game) then
    begin
      game.ClearAllValues();
    end
    else
    begin
      // das Spiel wieder im Wunschtermin setzen, haben wir vorher ja gelscht
      if Assigned(game.FWunschTermin) then
      begin
        game.FWunschTermin.assignedGame := game;
      end;
    end;
  end;

  MarkNotNecessaryGames();

  AllDatesAreValid := True;
end;


procedure TPlan.GenerateMannschaftsRefGames();
var
  Game: TGame;
begin
  ClearMannschaftGameRefs();

  for Game in gamesAllowed do
  begin
    Game.MannschaftHeim.gamesRef.Add(Game);
    Game.MannschaftGast.gamesRef.Add(Game);
  end;

  for Game in gamesNotAllowed do
  begin
    Game.MannschaftHeim.gamesRef.Add(Game);
    Game.MannschaftGast.gamesRef.Add(Game);
  end;

  SortMannschaftsDates;
end;

procedure TPlan.RemoveNotValidSisterGames();
var
  Mannschaft: TMannschaft;
  sisterGamesMap: TArray<TSisterGames>;
  sisterGames: TSisterGames;
  game, game1, game2: TSisterGame;
  i, j: integer;

begin
  for Mannschaft in Mannschaften do
  begin
    sisterGamesMap := Mannschaft.sisterGames.toArray();

    for sisterGames in sisterGamesMap do
    begin
      for i := sisterGames.Count-1 downto 0 do
      begin
        game := sisterGames[i];
        if(game.gender = gender) then
        begin
          if Assigned(Mannschaften.FindByName(game.HomeMannschaftName)) or Assigned(Mannschaften.FindByName(game.GuestMannschaftName)) then
          begin
            // Diesen Plan generiere ich gerade -> diese Daten kann ich entfernen
            sisterGames.Delete(i);
          end;
        end;
      end;
    end;
  end;

  // Doppelte entfernen, dass kann vorkommen, wenn zwei tiefere Mannschaften in einer Liga spielen
  // dann ist Hip 4 gegen Hip 5 zweimal aufgefhrt
  for Mannschaft in Mannschaften do
  begin
    sisterGamesMap := Mannschaft.sisterGames.toArray();

    for sisterGames in sisterGamesMap do
    begin
      for i := sisterGames.Count-1 downto 1 do
      begin
        game1 := sisterGames[i];
        for j := i-1 downto 0 do
        begin
          game2 := sisterGames[j];

          if (game1.HomeMannschaftName = game2.HomeMannschaftName)
              and (game1.GuestMannschaftName = game2.GuestMannschaftName) then
          begin
            // Dieses Spiel ist doppelt
            sisterGames.Delete(i);
            break;
          end;
        end;
      end;
    end;
  end;
end;

procedure TPlan.ClearMannschaftGameRefs();
var
  i: integer;
begin
  for i:=0 to Mannschaften.Count-1 do
  begin
    Mannschaften[i].gamesRef.Clear;
  end;
end;


procedure TPlan.ClearMannschaftWunschTerminRefs;
var
  Mannschaft: TMannschaft;
  WunschTermin: TWunschTermin;
begin
  for Mannschaft in Mannschaften do
  begin
    for WunschTermin in Mannschaft.WunschTermine do
    begin
      WunschTermin.assignedGame := nil;
    end;
  end;
end;

procedure TPlan.GenerateGames();
var
  i, j: Integer;
  NewGame: TGame;
  cachedList: TGameList;
  Key: String;
  Valid: boolean;
begin

  CreateRoundInfo();

  ClearMannschaftGameRefs();
  gamesAllowed.Clear;
  gamesNotAllowed.Clear;

  // Hin und Rckrunde erzeugen
  for i:=0 to Mannschaften.Count-1 do
  begin
    for j:=0 to Mannschaften.Count-1 do
    begin
      if (i <> j) then
      begin
        NewGame := TGame.Create(Mannschaften[i], Mannschaften[j]);
        gamesAllowed.Add(NewGame);

        if getOptions().DoubleRound then
        begin
          NewGame := TGame.Create(Mannschaften[i], Mannschaften[j]);
          gamesAllowed.Add(NewGame);
        end;
      end;
    end;
  end;

  cachedGameList.Clear;

  for i:=0 to gamesAllowed.Count-1 do
  begin
    Key := gamesAllowed[i].MannschaftHeim.teamId + '|' + gamesAllowed[i].MannschaftGast.teamId;

    Valid := cachedGameList.TryGetValue(Key, cachedList);

    if not Valid then
    begin
      cachedList := TGameList.Create(false);
      cachedGameList.Add(Key, cachedList);
    end;

    cachedList.Add(gamesAllowed[i]);
  end;

  GenerateMannschaftsRefGames();

end;

function TPlan.GetFirstDate: TDateTime;
var
  i, j: Integer;
begin
  Result := 0.0;
  for j:=0 to Mannschaften.Count - 1 do
  begin
    for i:=0 to Mannschaften[j].WunschTermine.Count - 1 do
    begin
      if Mannschaften[j].WunschTermine[i].Date <> 0.0 then
      begin
        if (Result = 0.0) or (Mannschaften[j].WunschTermine[i].Date < Result) then
        begin
          Result := Mannschaften[j].WunschTermine[i].Date;
        end;
      end;
    end;
  end;
end;


function TPlan.getFirstRealDateInRound(currRound: integer): TDateTime;
var
  Date: TDateTime;
  i, j: Integer;
begin
  Result := 0;
  for j:=0 to Mannschaften.Count - 1 do
  begin
    for i:=0 to Mannschaften[j].WunschTermine.Count - 1 do
    begin
      Date := Mannschaften[j].WunschTermine[i].Date;

      if (Result = 0.0) or (Date < Result) then
      begin
        if currRound = GetRoundNumberByDate(Date) then
        begin
          if not IsFreeGameDate(Date) then
          begin
            Result := Date;
          end;
        end;
      end;
    end;
  end;
end;


function TPlan.getLastRealDateInRound(currRound: integer): TDateTime;
var
  Date: TDateTime;
  i, j: Integer;
begin
  Result := 0;
  for j:=0 to Mannschaften.Count - 1 do
  begin
    for i:=0 to Mannschaften[j].WunschTermine.Count - 1 do
    begin
      Date := Mannschaften[j].WunschTermine[i].Date;

      if (Date > Result) then
      begin
        if currRound = GetRoundNumberByDate(Date) then
        begin
          if not IsFreeGameDate(Date) then
          begin
            Result := Date;
          end;
        end;
      end;
    end;
  end;
end;

procedure AddIndented(Dest, Source: TStrings);
var
  S: String;
begin
  for S in Source do
  begin
    Dest.Add('    ' + S);
  end;
end;


procedure TPlan.getHardErrorMessages(Messages: TStrings);
var
  Strs: TStrings;
  DateStrs: TStrings;
  currDate: TDateTime;
  NumError: integer;
  Mannschaft: TMannschaft;
  ViolatedDates: TList<TDateTime>;
  ViolatedGames: TList<TGame>;
  Game: TGame;
begin
  Strs := TStringList.Create;
  NumError := getNotTerminatedGames(Strs);
  if NumError > 0 then
  begin
    Messages.Add(IntToStr(NumError) + ' Spiele wurden nicht terminiert');
    AddIndented(Messages, Strs);
    Messages.Add('');
  end;
  Strs.Free;

  Strs := TStringList.Create;
  NumError := getNotAllowedGames(Strs);
  if NumError > 0 then
  begin
    Messages.Add(IntToStr(NumError) + ' Spiele sind an nicht erlaubten Tagen');
    AddIndented(Messages, Strs);
    Messages.Add('');
  end;
  Strs.Free;

  NumError := getTeamsNotAtBegin();
  if NumError > 0 then
  begin
    Messages.Add('Vereinsinterne Spiele sind nicht am Anfang der Runde');
    Messages.Add('');
  end;


  Strs := TStringList.Create;
  NumError := 0;

  for Mannschaft in Mannschaften do
  begin
    ViolatedDates := Mannschaft.CreateListOfViolatedSperrTermine(Self);
    if ViolatedDates <> nil then
    begin
      Inc(NumError, ViolatedDates.Count);
      Strs.Add(Mannschaft.teamName);
      DateStrs := TStringList.Create;
      for currDate in ViolatedDates do
      begin
        DateStrs.Add(MyFormatDate(currDate));
      end;
      AddIndented(Strs, DateStrs);
      DateStrs.Free;
      ViolatedDates.Free;
    end;
  end;

  if NumError > 0 then
  begin
    Messages.Add(IntToStr(NumError) + ' Sperrtermine wurden verletzt');
    AddIndented(Messages, Strs);
    Messages.Add('');
  end;
  Strs.Free;


  Strs := TStringList.Create;
  NumError := 0;

  for Mannschaft in Mannschaften do
  begin
    ViolatedGames := Mannschaft.CreateListOfViolated60KmRegel(Self);
    if ViolatedGames <> nil then
    begin
      Inc(NumError, ViolatedGames.Count);
      for Game in ViolatedGames do
      begin
        Strs.Add(Game.AsString());
      end;
      ViolatedGames.Free;
    end;
  end;

  if NumError > 0 then
  begin
    Messages.Add(IntToStr(NumError) + ' mal wurde die 60km Regel verletzt');
    AddIndented(Messages, Strs);
    Messages.Add('');
  end;
  Strs.Free;


  Strs := TStringList.Create;
  NumError := 0;

  for Mannschaft in Mannschaften do
  begin
    DateStrs := TStringList.Create;
    Mannschaft.getMessagesEngeTermine2Days(Self, DateStrs);
    if DateStrs.Count > 0 then
    begin
      Inc(NumError, DateStrs.Count);
      Strs.Add(Mannschaft.teamName);
      AddIndented(Strs, DateStrs);
    end;
    DateStrs.Free;
  end;

  if NumError > 0 then
  begin
    Messages.Add(IntToStr(NumError) + ' Spiele mit weniger als 2 Tagen Abstand');
    AddIndented(Messages, Strs);
    Messages.Add('');
  end;
  Strs.Free;


end;

function TPlan.GetLastDate: TDateTime;
var
  i, j: Integer;
begin
  Result := 0.0;
  for j:=0 to Mannschaften.Count - 1 do
  begin
    for i:=0 to Mannschaften[j].WunschTermine.Count - 1 do
    begin
      if Mannschaften[j].WunschTermine[i].Date <> 0.0 then
      begin
        if (Result = 0.0) or (Mannschaften[j].WunschTermine[i].Date > Result) then
        begin
          Result := Mannschaften[j].WunschTermine[i].Date;
        end;
      end;
    end;
  end;
end;



procedure TPlan.LoadNoGameDays(Data: TDataObject);
var
  node: TDataObject;
  Date: TDateTime;
begin
  FreeGameDays.Clear;
  for node in Data.NamedChilds(nNoGameDay) do
  begin
    Date := node.GetAsDate(aDate);
    if not FreeGameDays.ContainsKey(Date) then
    begin
      FreeGameDays.Add(Date, '');
    end;
  end;
end;

procedure TPlan.LoadRanking(Data: TDataObject);
var
  RankingNode: TDataObject;
  TeamNode: TDataObject;
  TeamName: String;
  Mannschaft: TMannschaft;
  RankingIndex: integer;
  i: integer;
begin
  RankingIndexe.Clear;
  RankingNode := Data.FindChild(nRanking);

  if Assigned(RankingNode) then
  begin
    if RankingNode.GetAsBool(aActive) then
    begin
      for TeamNode in RankingNode.NamedChilds(nTeam) do
      begin
        TeamName := TeamNode.GetAsString(aTeamName);
        RankingIndex := TeamNode.GetAsInt(aRankingIndex);

        Mannschaft := Mannschaften.FindByName(TeamName);

        if Assigned(Mannschaft) then
        begin
          while RankingIndexe.Count <= RankingIndex do
          begin
            RankingIndexe.Add(-1);
          end;
          RankingIndexe[RankingIndex] := Mannschaft.Index;
        end;
      end;

      for i := RankingIndexe.Count-1 downto 0 do
      begin
        if RankingIndexe[i] < 0 then
        begin
          RankingIndexe.Delete(i);
        end;
      end;

      // Fehlende Mannschaften einfgen
      for Mannschaft in Mannschaften do
      begin
        if 0 > RankingIndexe.IndexOf(Mannschaft.Index) then
        begin
          RankingIndexe.Add(Mannschaft.Index);
        end;
      end;
    end;
  end;
end;

procedure TPlan.LoadMandatoryGameDays(Data: TDataObject);
var
  node: TDataObject;
  NewElem: TPlanMandatoryTime;
begin
  MandatoryGameDays.Clear;
  for node in Data.NamedChilds(nMandatoryGames) do
  begin
    NewElem := TPlanMandatoryTime.Create;
    NewElem.DateFrom := node.GetAsDate(aDateFrom);
    NewElem.DateTo := node.GetAsDate(aDateTo);
    NewElem.GameCount := node.GetAsInt(aNumberGames);
    MandatoryGameDays.Add(NewElem);
  end;
end;


procedure TPlan.LoadGameDates(allGames: TGameList; DataObject: TDataObject; PlanData: TPlanData);
var
  date: TDateTime;
  homeTeam, guestTeam: String;
  node: TDataObject;
  game: TGame;
  MannschaftHeim: TMannschaft;
  MannschaftGast: TMannschaft;
begin
  allGames.Clear;
  for node in DataObject.NamedChilds(nGame) do
  begin
    homeTeam := node.GetAsString(aHomeTeamName);
    guestTeam := node.GetAsString(aGuestTeamName);

    date := node.GetAsDateTime(aDateTime);

    MannschaftHeim := Mannschaften.FindByName(homeTeam);
    MannschaftGast := Mannschaften.FindByName(guestTeam);

    if Assigned(MannschaftHeim) and Assigned(MannschaftGast) then
    begin
      game := TGame.Create(MannschaftHeim, MannschaftGast);

      // Koppel und MaxHome hier egal, dass erledigt das AssignDates
      game.setDate(date, [], 0, False, False, nil);

      game.OverrideLocation := PlanData.FixLocation(node.GetAsString(aLocation));

      allGames.Add(game);
    end;
  end;
end;


function TPlan.CreateCache: TPlanDataCache;
begin
  Result := TPlanDataCache.Create;
  Result.Has60KilometerValues := CalcHas60KilometerValues;
  Result.NumWunschDays := CalcNumWunschDays();
  Result.HasAuswaertsKoppelTermine := CalcHasAuswaertsKoppelTermine;
  Result.HasForceSameHomeGames := CalcHasForceSameHomeGames;
  Result.HasMandatoryGameDays := CalcHasMandatoryGameDays;
  Result.HasKoppelTermine := CalcHasKoppelTermine;
  Result.HasAusweichTermine := CalcHasAusweichTermine;
end;


function TPlan.getCache: TPlanDataCache;
begin
  if not Assigned(dataCache) then
  begin
    dataCache := CreateCache();
  end;

  Result := dataCache;

end;

procedure TPlan.getSchedule(gamesList: TSchedule);
var
  Game, predefinedGame: TGame;
  NewGame: TScheduleGame;
begin
  gamesList.Clear;

  for Game in gamesAllowed do
  begin
    NewGame := TScheduleGame.Create();
    NewGame.MannschaftHeimName := Game.MannschaftHeim.teamName;
    NewGame.MannschaftGastName := Game.MannschaftGast.teamName;
    NewGame.Date := Game.Date;
    NewGame.Location := Game.Location;

    predefinedGame := predefinedGames.findSameGame(game);
    if Assigned(predefinedGame) then
    begin
      NewGame.Location := predefinedGame.OverrideLocation;
    end;
    gamesList.Add(NewGame);
  end;

  for Game in gamesNotAllowed do
  begin
    NewGame := TScheduleGame.Create();
    NewGame.MannschaftHeimName := Game.MannschaftHeim.teamName;
    NewGame.MannschaftGastName := Game.MannschaftGast.teamName;
    NewGame.Date := Game.Date;
    NewGame.Location := Game.Location;
    predefinedGame := predefinedGames.findSameGame(game);
    if Assigned(predefinedGame) then
    begin
      NewGame.Location := predefinedGame.OverrideLocation;
    end;
    gamesList.Add(NewGame);
  end;


end;


procedure TPlan.getCloneOfAllGames(gamesList: TGameList);
var
  Game, NewGame: TGame;
begin
  gamesList.Clear;

  for Game in gamesAllowed do
  begin
    NewGame := TGame.Create(Game.MannschaftHeim, Game.MannschaftGast);
    NewGame.setDate(Game.Date, Game.DateOptions, Game.DateMaxHome, False, Game.NotNecessary, nil);
    gamesList.Add(NewGame);
  end;

  for Game in gamesNotAllowed do
  begin
    NewGame := TGame.Create(Game.MannschaftHeim, Game.MannschaftGast);
    NewGame.setDate(Game.Date, Game.DateOptions, Game.DateMaxHome, False, Game.NotNecessary, nil);
    gamesList.Add(NewGame);
  end;

end;




procedure TPlan.assignSchedule(allGames: TSchedule);
var
  gameSource: TScheduleGame;
  gameDest, gameNotAllowed: TGame;
  WunschTermin: TWunschTermin;
  isKoppel: TWunschterminOptions;
  found: boolean;
  MaxHome: integer;
  MannschaftGast, MannschaftHeim: TMannschaft;
  FoundWunschTermin: TWunschTermin;
begin
  clearAllDates();
  GenerateGames();

  ClearMannschaftGameRefs();
  gamesNotAllowed.Clear;

  for gameSource in allGames do
  begin
    found := False;
    // Suche nach diesem Spiel
    for gameDest in gamesAllowed do
    begin
      if not gameDest.DateValid then
      begin
        if (gameDest.MannschaftHeim.teamName = gameSource.MannschaftHeimName)
            and (gameDest.MannschaftGast.teamName = gameSource.MannschaftGastName) then
        begin
          // Suche, ob es ein Koppeltermin ist
          isKoppel := [];
          MaxHome := 0;
          for wunschTermin in gameDest.MannschaftHeim.WunschTermine do
          begin
            if Trunc(WunschTermin.Date) = Trunc(gameSource.Date) then
            begin
              isKoppel := WunschTermin.Options;
              MaxHome := WunschTermin.ParallelGames;
            end;
          end;

          FoundWunschTermin := gameDest.MannschaftHeim.WunschTermine.Find(gameSource.Date);
          gameDest.setDate(gameSource.Date, isKoppel, MaxHome, False, false, FoundWunschTermin);

          found := True;
          break;
        end;
      end;
    end;

    // Jetzt noch die ungltigen kopieren
    if not Found then
    begin
      MannschaftGast := Mannschaften.FindByName(gameSource.MannschaftGastName);
      MannschaftHeim := Mannschaften.FindByName(gameSource.MannschaftHeimName);

      if Assigned(MannschaftGast) and Assigned(MannschaftHeim) then
      begin
        gameNotAllowed := TGame.Create(MannschaftHeim, MannschaftGast);

        // Koppel und MaxHome hier egal, dass ist eh ein Ksetermin
        gameNotAllowed.setDate(gameSource.Date, [], 0, False, False, nil);

        gamesNotAllowed.Add(gameNotAllowed);
      end;
    end;
  end;

  GenerateMannschaftsRefGames();

  MarkNotNecessaryGames();

  AllDatesAreValid := False;
end;


procedure TPlan.assignGames(allGames: TGameList);
var
  gameSource, gameDest, gameNotAllowed: TGame;
  WunschTermin: TWunschTermin;
  isKoppel: TWunschterminOptions;
  found: boolean;
  MaxHome: integer;
  MannschaftGast, MannschaftHeim: TMannschaft;
  FoundWunschTermin: TWunschTermin;
begin
  clearAllDates();
  GenerateGames();

  ClearMannschaftGameRefs();
  gamesNotAllowed.Clear;

  for gameSource in allGames do
  begin
    found := False;
    // Suche nach diesem Spiel
    for gameDest in gamesAllowed do
    begin
      if not gameDest.DateValid then
      begin
        if (gameDest.MannschaftHeim.teamId = gameSource.MannschaftHeim.teamId)
            and (gameDest.MannschaftGast.teamId = gameSource.MannschaftGast.teamId) then
        begin
          // Suche, ob es ein Koppeltermin ist
          isKoppel := [];
          MaxHome := 0;
          for wunschTermin in gameDest.MannschaftHeim.WunschTermine do
          begin
            if Trunc(WunschTermin.Date) = Trunc(gameSource.Date) then
            begin
              isKoppel := WunschTermin.Options;
              MaxHome := WunschTermin.ParallelGames;
            end;
          end;

          FoundWunschTermin := gameDest.MannschaftHeim.WunschTermine.Find(gameSource.Date);
          gameDest.setDate(gameSource.Date, isKoppel, MaxHome, False, gameSource.NotNecessary, FoundWunschTermin);

          found := True;
          break;
        end;
      end;
    end;

    // Jetzt noch die ungltigen kopieren
    if not Found then
    begin
      MannschaftGast := Mannschaften.FindByTeamId(gameSource.MannschaftGast.teamId);
      MannschaftHeim := Mannschaften.FindByTeamId(gameSource.MannschaftHeim.teamId);

      if Assigned(MannschaftGast) and Assigned(MannschaftHeim) then
      begin
        gameNotAllowed := TGame.Create(MannschaftHeim, MannschaftGast);

        // Koppel und MaxHome hier egal, dass ist eh ein Ksetermin
        gameNotAllowed.setDate(gameSource.Date, [], 0, False, False, nil);

        gamesNotAllowed.Add(gameNotAllowed);
      end;
    end;
  end;

  GenerateMannschaftsRefGames();

  AllDatesAreValid := False;
end;


procedure TPlan.assignPredefinedGames(allGames: TGameList);
var
  gameSource, gameDest: TGame;
  MannschaftGast, MannschaftHeim: TMannschaft;
  WunschTermin: TWunschTermin;
begin
  predefinedGames.Clear;

  for gameSource in allGames do
  begin
    MannschaftHeim := Mannschaften.FindByTeamId(gameSource.MannschaftHeim.teamId);
    MannschaftGast := Mannschaften.FindByTeamId(gameSource.MannschaftGast.teamId);
    if Assigned(MannschaftHeim) and Assigned(MannschaftGast) then
    begin
      gameDest := TGame.Create(MannschaftHeim, MannschaftGast);
      WunschTermin := gameDest.MannschaftHeim.WunschTermine.Find(gameSource.Date);
      if Assigned(WunschTermin) then
      begin
        gameDest.setDate(gameSource.Date, WunschTermin.Options, WunschTermin.ParallelGames, False, False, WunschTermin);
      end
      else
      begin
        gameDest.setDate(gameSource.Date, [], 0, False, False, nil);
      end;
      gameDest.OverrideLocation := gameSource.OverrideLocation;
      predefinedGames.Add(gameDest);
    end;
  end;
end;



procedure TPlan.assignExistingScheduleGames(allGames: TGameList);
var
  gameSource, gameDest: TGame;
  MannschaftGast, MannschaftHeim: TMannschaft;
  WunschTermin: TWunschTermin;
begin
  existingSchedule.Clear;

  for gameSource in allGames do
  begin
    MannschaftHeim := Mannschaften.FindByTeamId(gameSource.MannschaftHeim.teamId);
    MannschaftGast := Mannschaften.FindByTeamId(gameSource.MannschaftGast.teamId);
    if Assigned(MannschaftHeim) and Assigned(MannschaftGast) then
    begin
      gameDest := TGame.Create(MannschaftHeim, MannschaftGast);
      WunschTermin := gameDest.MannschaftHeim.WunschTermine.Find(gameSource.Date);
      if Assigned(WunschTermin) then
      begin
        gameDest.setDate(gameSource.Date, WunschTermin.Options, WunschTermin.ParallelGames, False, False, WunschTermin);
      end
      else
      begin
        gameDest.setDate(gameSource.Date, [], 0, False, False, nil);
      end;
      existingSchedule.Add(gameDest);
    end;
  end;
end;




procedure TPlan.loadTeams(Data: TDataObject; PlanData: TPlanData);
var
  Mannschaft: TMannschaft;
  TeamNode: TDataObject;
begin
  Mannschaften.Clear;
  for TeamNode in Data.NamedChilds(nTeam) do
  begin
    Mannschaft := TMannschaft.Create(Mannschaften.Count);
    Mannschaften.Add(Mannschaft);
    Mannschaft.load(TeamNode, Self, PlanData);
  end;

  Mannschaften.Sort;
end;



procedure TPlan.MarkSecondGameAsUnNecessary(game: TGame);
var
  games: TGameList;
  value: TGame;
begin
  games := findGames(game.MannschaftGast.teamId, game.MannschaftHeim.teamId);

  for value in games do
  begin
    if (not value.NotNecessary) and (not Value.DateValid)  then
    begin
      value.setDate(value.Date, [], 0, false, true, nil);
      break;
    end;
  end;
end;

function TPlan.NoGames: boolean;
var
  Game: TGame;
begin
  Result := True;

  for Game in gamesAllowed do
  begin
    if Game.Date <> 0 then
    begin
      Result := False;
      break;
    end;
  end;
end;

procedure TPlan.ProfileFillTermine(TauschPercent: MyDouble);
begin
  TransferRefDateToDate();

  NeuWuerfeln(ttNormal, 0, TauschPercent);

  FillTermine();

end;


function TPlan.getTeamsNotAtBegin: integer;
var
  Mannschaft: TMannschaft;
  Game: TGame;
  Gefunden: integer;
  RoundInfo: TRoundInfo;
begin
  Result := 0;
  for Mannschaft in Mannschaften do
  begin
    if Mannschaft.SisterTeamsInPlan.Count > 0 then
    begin
      for RoundInfo in Rounds do
      begin
        if IsRoundToGenerate(RoundInfo) then
        begin
          Gefunden := 0;
          for Game in Mannschaft.gamesRef do
          begin
            if Game.DateValid then
            begin
              if RoundInfo.DateInRound(Game.Date)
                 then
              begin
                if Mannschaft.SisterTeamsInPlan.Contains(Game.MannschaftHeim) or
                  Mannschaft.SisterTeamsInPlan.Contains(Game.MannschaftGast) then
                begin
                  Inc(Gefunden);
                end
                else
                begin
                  Inc(Result);
                end;
              end;

              if Gefunden >= Mannschaft.SisterTeamsInPlan.Count then
              begin
                break;
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;


function TPlan.CalculateSisterTeamsAmAnfang: MyDouble;
var
  value: MyDouble;
  Fehler: integer;
begin
  Fehler := getTeamsNotAtBegin();
  value := Power(Fehler, 2) * 300 * GetSisterTeamsAmAnfangKostenFaktor();
  Result := Min(cMaxKostenNoHardError, value);
end;


procedure CalculateMaxAbweichung(var EchteAbweichung: integer; var FactorKorrigierteAbweichung: MyDouble; values: TList<Integer>; var MittelWert: MyDouble; FactorLess: MyDouble; FactorMore: MyDouble);
var
  i: Integer;
  Factor: MyDouble;
begin
  EchteAbweichung := 0;
  FactorKorrigierteAbweichung := 0;

  if values.Count <= 1 then
  begin
    MittelWert := 1;
  end
  else
  begin
    MittelWert := 0;

    for i:=0 to values.Count-1 do
    begin
      MittelWert := MittelWert + values[i];
    end;

    MittelWert := MittelWert / (values.Count * 1.0);

    for i:=0 to values.Count-1 do
    begin
      if values[i] < MittelWert then
      begin
        Factor := FactorLess;
      end
      else
      begin
        Factor := FactorMore;
      end;


      FactorKorrigierteAbweichung := Max(Factor *(Abs(values[i] - MittelWert)), FactorKorrigierteAbweichung);

      EchteAbweichung := Max(Trunc((Abs(values[i] - MittelWert) + 0.5)), EchteAbweichung);
    end;

  end;

end;


function CalculateStandardAbweichung(values: TList<Integer>; var MittelWert: MyDouble): MyDouble; overload;
var
  i: Integer;
  Value: MyDouble;
begin
  if values.Count <= 1 then
  begin
    MittelWert := 1;
    Result := 0;
  end
  else
  begin
    MittelWert := 0;

    for i:=0 to values.Count-1 do
    begin
      MittelWert := MittelWert + values[i];
    end;

    MittelWert := MittelWert / (values.Count * 1.0);

    Result := 0;
    for i:=0 to values.Count-1 do
    begin
      Value := values[i] - MittelWert;
      Result := Result + (Value * Value);
    end;
    Result := sqrt(Result / (values.Count * 1.0));
  end;
end;



function CalculateStandardAbweichung(values: TList<MyDouble>; var MittelWert: MyDouble): MyDouble; overload;
var
  i: Integer;
  Value: MyDouble;
begin
  if values.Count <= 1 then
  begin
    MittelWert := 1;
    Result := 0;
  end
  else
  begin
    MittelWert := 0;

    for i:=0 to values.Count-1 do
    begin
      MittelWert := MittelWert + values[i];
    end;

    MittelWert := MittelWert / (values.Count * 1.0);

    Result := 0;
    for i:=0 to values.Count-1 do
    begin
      Value := values[i] - MittelWert;
      Result := Result + (Value * Value);
    end;
    Result := sqrt(Result / (values.Count * 1.0));
  end;
end;



procedure TPlan.CalculateSpielTagKosten(var KostenBreite, KostenOverlap, LastKostenBreite, LastKostenOverlap: MyDouble);
var
  i, j: Integer;
  SpielTageCount: Integer;
  value: MyDouble;
  MaxDate, MinDate, ActDate: TDateTime;
  Diff: MyDouble;
  StdAbweichung, IntervallLaenge, MittelWert: MyDouble;
  DateValues: TList<Integer>;
  Faktor: MyDouble;
  LastMaxDateValues: TList<TDateTime>;
begin

  KostenOverlap := 0;
  KostenBreite := 0;
  SpielTageCount := 0;
  LastKostenBreite := 0;
  LastKostenOverlap := 0;

  for i:=0 to Mannschaften.Count-1 do
  begin
    SpielTageCount := Max(SpielTageCount, Mannschaften[i].gamesRef.Count);
  end;

  LastMaxDateValues := TList<TDateTime>.Create;

  for i:=0 to SpielTageCount-1 do
  begin
    DateValues := TList<Integer>.Create;
    MaxDate := 0.0;
    MinDate := 0.0;
    for j:=0 to Mannschaften.Count-1 do
    begin
      ActDate := 0.0;
      if Mannschaften[j].gamesRef.Count > i then
      begin
        ActDate := Mannschaften[j].gamesRef[i].Date;
      end;

      if (ActDate <> 0.0) and (IsInRoundToGenerate(ActDate)) then
      begin
        DateValues.Add(Trunc(ActDate));
        MaxDate := Max(MaxDate, ActDate);

        if MinDate = 0.0 then
        begin
          MinDate := ActDate;
        end
        else
        begin
          MinDate := Min(MinDate, ActDate);
        end;
      end;
    end;

    if (MaxDate > 0) and (MinDate > 0) then
    begin
      //berlappung zum vorherigen Spieltag ermitteln
      Faktor := 5;
      for j := LastMaxDateValues.Count-1 downto 0 do
      begin
        if Trunc(MinDate) <= Trunc(LastMaxDateValues[j]) then
        begin
          // Eine Verletzung zhlt 1, sind es mehr als 8 Tage, nochmal 1 (im Prinzip die Tage sind die Nachkommastelle)
          Diff := (1 + (Trunc(LastMaxDateValues[j]) - Trunc(MinDate)) / 8);

          // Fr berschneidung von 3 Spieltagen gibt es einen hheren Faktor
          value := Diff * Faktor;

          // Letzter Spieltag
          if SpielTageCount-1 = i then
          begin
            LastKostenOverlap := value * 20.0;
          end;


          KostenOverlap := KostenOverlap + value;
        end;
        // berlappung zum nchsten Spieltag wird mal 10 gerechnet
        Faktor := Faktor * 5;
      end;

      LastMaxDateValues.Add(MaxDate);
    end;

    IntervallLaenge := Trunc(MaxDate) - Trunc(MinDate);

    // Je weniger Mannschaften, desto breiter darf der Spieltag sein
    IntervallLaenge := IntervallLaenge * (Mannschaften.Count / 10.0);

    Diff := max(0, IntervallLaenge - 6.0) / 7.0;

    if IntervallLaenge > 0 then
    begin
      StdAbweichung := CalculateStandardAbweichung(DateValues, MittelWert);
      Diff := Diff + min(0.99, StdAbweichung / (IntervallLaenge / 2));
    end;

    DateValues.Free;

    // Letzter Spieltag
    if SpielTageCount-1 = i then
    begin
      LastKostenBreite := Power(Diff, 3) * GetLastSpielTageKostenFaktor() * 5.0;
    end;


    value := Power(Diff, 3) * GetSpielTageKostenFaktor();

    KostenBreite := KostenBreite + value;

  end;

  LastKostenBreite := Min(cMaxKostenNoHardError, LastKostenBreite);

  KostenBreite := Min(cMaxKostenNoHardError, KostenBreite);

  KostenOverlap := KostenOverlap * GetSpielOverlapKostenFaktor();

  KostenOverlap := Min(cMaxKostenNoHardError, KostenOverlap);

  LastKostenOverlap := LastKostenOverlap * GetLastSpielOverlapKostenFaktor();

  LastKostenOverlap := Min(cMaxKostenNoHardError, LastKostenOverlap);


  LastMaxDateValues.Free;
end;

function TPlan.GetSpielTageKostenFaktor(): MyDouble;
begin
  result := cCalculateOptionsValues[getOptions().SpieltagGewichtung];
end;


function TPlan.GetSpielOverlapKostenFaktor(): MyDouble;
begin
  result := cCalculateOptionsValues[getOptions().SpieltagOverlapp];
end;

function TPlan.GetLastSpielTageKostenFaktor(): MyDouble;
begin
  result := cCalculateOptionsValues[getOptions().LastSpieltagGewichtung];
end;


function TPlan.GetLastSpielOverlapKostenFaktor(): MyDouble;
begin
  result := cCalculateOptionsValues[getOptions().LastSpieltagOverlapp];
end;

function TPlan.GetSisterTeamsAmAnfangKostenFaktor(): MyDouble;
begin
  result := cCalculateOptionsValues[getOptions().SisterTeamsAmAnfangGewichtung];
end;


const
  CalcSequenceFastPerformance: array [0 .. 10] of TMannschaftsKostenType =
  (
  mktEngeTermine,
  mkt2SpieleProWoche,
  mkt60Kilometer,
  mktAuswaertsKoppelTermine,
  mktMandatoryGames,
  mktKoppelTermine,
  mktZahlHeimSpielTermine,
  mktWechselHeimAuswaerts,
  mktAbstandHeimAuswaerts,
  mktRanking,
  mktSperrTermine
  );

  CalcSequenceLowPerformance: array [0 .. 4] of TMannschaftsKostenType =
  (
  mktAusweichTermine,
  mktHalleBelegt,
  mktSisterGames,
  mktForceSameHomeGames,
  mktSpielverteilung
  );


procedure TestCalcSequence();
var
  KostenType, KostenType2: TMannschaftsKostenType;
  valid: boolean;
  isDouble: boolean;
begin
  for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    valid := False;
    isDouble := False;
    for KostenType2 in CalcSequenceFastPerformance do
    begin
      if KostenType = KostenType2 then
      begin
        valid := True;
      end;
    end;

    for KostenType2 in CalcSequenceLowPerformance do
    begin
      if KostenType = KostenType2 then
      begin
        if valid then
        begin
          isDouble := True;
        end;
        valid := True;
      end;
    end;


    if not Valid then
    begin
      raise Exception.Create('CalcSequence: ein Element fehlt');
    end;

    if isDouble then
    begin
      raise Exception.Create('CalcSequence: ein Element doppelt');
    end;

  end;

end;



function TPlan.CalculateKosten(BreakByValue: MyDouble (* = -1*)): MyDouble;
var
  KostenType: TMannschaftsKostenType;
  Breite, Overlap, LastBreite, LastOverlap: MyDouble;
begin

  Result := CalculateFehlterminKosten();

  if (BreakByValue >= 0) and (Result > BreakByValue) then
  begin
    Result := -1;
    Exit;
  end;

  Result := Result + CalculateFreeGameDateKosten();

  if (BreakByValue >= 0) and (Result > BreakByValue) then
  begin
    Result := -1;
    Exit;
  end;


  if getOptions().SisterTeamsAmAnfangGewichtung <> coIgnore then
  begin
    Result := Result + CalculateSisterTeamsAmAnfang();
  end;


  for KostenType in CalcSequenceFastPerformance do
  begin
    if getOptions().GewichtungMainMannschaftsKosten[KostenType] <> coIgnore then
    begin
      if (BreakByValue >= 0) and (Result > BreakByValue) then
      begin
        Result := -1;
        Exit;
      end;

      Result := Result + CalculateMannschaftsKosten(KostenType);
    end;
  end;

  if (BreakByValue >= 0) and (Result > BreakByValue) then
  begin
    Result := -1;
    Exit;
  end;

  if (getOptions().SpieltagGewichtung <> coIgnore) or (getOptions().SpieltagOverlapp <> coIgnore) then
  begin
    CalculateSpielTagKosten(Breite, Overlap, LastBreite, LastOverlap);
    Result := Result + Breite + Overlap + LastBreite + LastOverlap;
  end;


  for KostenType in CalcSequenceLowPerformance do
  begin
    if getOptions().GewichtungMainMannschaftsKosten[KostenType] <> coIgnore then
    begin
      if (BreakByValue >= 0) and (Result > BreakByValue) then
      begin
        Result := -1;
        Exit;
      end;

      Result := Result + CalculateMannschaftsKosten(KostenType);
    end;
  end;

  if (BreakByValue >= 0) and (Result > BreakByValue) then
  begin
    Result := -1;
    Exit;
  end;

  Result := Result + CalculateNotAllowedKosten();


end;


function TPlan.getNotTerminatedGames(Messages: TStrings): integer;
var
  i: integer;
begin

  Result := 0;
  for i:=0 to gamesAllowed.Count-1 do
  begin
    if gamesAllowed[i].Date = 0.0 then
    begin
      if not gamesAllowed[i].NotNecessary then
      begin
        Inc(Result);
        if Messages <> nil then
        begin
          Messages.Add(gamesAllowed[i].AsString);
        end;
      end;
    end;
  end;
end;



function TPlan.getNumSollGamesIntern(DateFrom, DateTo: TDateTime): MyDouble;
var
  RoundNum: integer;
  DateCount: integer;
  DateInIntervall: integer;
  Mannschaft: TMannschaft;
  Wunschtermin: TWunschTermin;
  currCount: integer;
  FaktorWunschToDate: myDouble;
begin
  Result := 0;
  RoundNum := GetRoundNumberByDate(DateFrom);
  DateCount := 0;
  DateInIntervall := 0;

  for Mannschaft in Mannschaften do
  begin
    for Wunschtermin in Mannschaft.WunschTermine do
    begin
      if RoundNum = GetRoundNumberByDate(Wunschtermin.Date) then
      begin
        if not IsFreeGameDate(Wunschtermin.Date) then
        begin
          currCount := 1;
          if OptionIsKoppel(Wunschtermin.Options) then
          begin
            currCount := 2;
          end;

          DateCount := DateCount + currCount;

          if (Trunc(DateFrom) <= Trunc(Wunschtermin.Date))
            and (Trunc(DateTo) > Trunc(Wunschtermin.Date))
          then
          begin
            DateInIntervall := DateInIntervall + currCount;
          end;
        end;
      end;
    end;
  end;

  if (DateCount > 0) and (Mannschaften.Count > 0)  then
  begin
    FaktorWunschToDate := (Mannschaften.Count - 1) / (1.0 * DateCount);

    Result := FaktorWunschToDate * DateInIntervall;
  end;
end;


function TPlan.getNumSollGames(DateFrom, DateTo: TDateTime): MyDouble;
var
  Key: String;
  sollDates: TDictionary<string, MyDouble>;
begin
  sollDates := getCache().getSollGamesByDate();
  Key := IntToStr(Trunc(DateFrom)) + ' ' + IntToStr(Trunc(DateTo));

  if not sollDates.TryGetValue(Key, Result) then
  begin
    Result := getNumSollGamesIntern(DateFrom, DateTo);
    sollDates.Add(Key, Result);
  end;
end;


function TPlan.CalculateFehlterminKosten(): MyDouble;
var
  FehlTermine: integer;
begin
  FehlTermine := getNotTerminatedGames(nil);
  Result := FehlTermine * 10.0 * cMaxKostenNoHardError;
end;


function TPlan.getNotAllowedGames(Messages: TStrings): integer;
var
  i: integer;
  Game: TGame;
begin
  Result := 0;

  // Diese Prfung dauert recht lange und kann nur bei Koppelterminen mit den Anfangszeitenverschiebungen passieren
  if not AllDatesAreValid or (HasKoppelTermine) or (HasAuswaertsKoppelTermine) then
  begin
    Result := gamesNotAllowed.Count;

    if Messages <> nil then
    begin
      for Game in gamesNotAllowed do
      begin
        Messages.Add(Game.AsString());
      end;
    end;


    for i:=0 to gamesAllowed.Count-1 do
    begin
      if gamesAllowed[i].Date <> 0.0 then
      begin
        if not GameIsValid(gamesAllowed[i]) then
        begin
          if not gamesAllowed[i].NotNecessary then
          begin
            Inc(Result);

            if Messages <> nil then
            begin
              Messages.Add(gamesAllowed[i].AsString());
            end;
          end;
        end;
      end;
    end;
  end;
end;




function TPlan.CalculateNotAllowedKosten(): MyDouble;
var
  FehlTermine: integer;
begin
  FehlTermine := getNotAllowedGames(nil);
  Result := FehlTermine * 10.0 * cMaxKostenNoHardError;
end;


function TPlan.IsFreeGameDate(Date: TDateTime): boolean;
begin
  Result := FreeGameDays.ContainsKey(Trunc(Date));
end;


function TPlan.CalculateFreeGameDateKosten(): MyDouble;
var
  i: integer;
  FehlTermine: integer;
begin

  FehlTermine := 0;
  if FreeGameDays.Count > 0 then
  begin
    for i:=0 to gamesAllowed.Count-1 do
    begin
      if IsFreeGameDate(gamesAllowed[i].Date) then
      begin
        if IsInRoundToGenerate(gamesAllowed[i].Date) then
        begin
          Inc(FehlTermine);
        end;
      end;
    end;
  end;

  Result := FehlTermine * 10.0 * cMaxKostenNoHardError;
end;


function TPlan.CalculateMannschaftsKosten(KostenType: TMannschaftsKostenType): MyDouble;
var
  i: integer;
  MannschftKosten: TMannschaftsKosten;
begin
  Result := 0;
  for i:=0 to Mannschaften.Count - 1 do
  begin
    MannschftKosten := TMannschaftsKosten.Create;

    Mannschaften[i].CalculateKostenForType(Self, KostenType, MannschftKosten, tmNoMessage);

    Result := Result + MannschftKosten.Values[KostenType].GesamtKosten;

    MannschftKosten.Free;
  end;
end;

function TPlan.GameIsValid(Game: TGame): Boolean;
var
  WunschTerminTemp: TWunschTermin;
  WunschTermin: TWunschTermin;
begin
  // Testen, ob dieser Termin belegt ist
  Result := True;

  if not IsInRoundToGenerate(Game.Date) then
  begin
    Result := False;
    if getOptions.RoundPlaning in [rpSecondOnly, rpCorona] then
    begin
      // Bei der Genmerierung der Rckrunde sind diese Spiele vordefiniert
      // und nicht ungltig. Deshalb drfen die auch nicht gelscht werden
      // Bei der Vorrunde wird die Rckrunde aber schon ungltig
      if Assigned(existingSchedule.findSameGame(game)) then
      begin
        Result := True;
      end;
      if Assigned(predefinedGames.findSameGame(game)) then
      begin
        Result := True;
      end;
    end;
    exit;
  end;


  WunschTermin := nil;
  for WunschTerminTemp in game.MannschaftHeim.WunschTermine do
  begin
    if WunschTerminTemp.Date = Game.Date then
    begin
      WunschTermin := WunschTerminTemp;
      break;
    end;
  end;

  if WunschTermin <> nil then
  begin
    Result := TerminIsValid(WunschTermin, Game, Game);
  end
  else
  begin
    if nil = predefinedGames.findSameGame(game) then
    begin
      // Kein Wunschtermin und auch kein fest eingestellter Termin
      Result := False;
    end;
  end;

end;


function TPlan.TerminIsValid(WunschTermin: TWunschTermin; Game: TGame; IgnoreGame: TGame = nil; AllSecondTimeGamesAreValid: boolean = false): Boolean;
var
  testGame: TGame;
  testGames: TGameList;
  roundNumber, roundNumberTest: integer;
begin
  Result := True;

  if FreeGameDays.Count > 0  then
  begin
    if IsFreeGameDate(WunschTermin.Date) then
    begin
      Result := False;
      Exit;
    end;
  end;

  if not Game.MannschaftHeim.IsTerminFree(WunschTermin, self, Game.MannschaftHeim, Game.MannschaftGast, IgnoreGame, AllSecondTimeGamesAreValid) then
  begin
    Result := False;
    Exit;
  end;


  if not Game.MannschaftGast.IsTerminFree(WunschTermin, self, Game.MannschaftHeim, Game.MannschaftGast, IgnoreGame, AllSecondTimeGamesAreValid) then
  begin
    Result := False;
    Exit;
  end;


  roundNumber := GetRoundNumberByDate(WunschTermin.Date);

  // Finde die anderen Spiele die die beiden Mannschaft bestreiten

  // erst gleiche Spiele
  testGames := findGames(Game.MannschaftHeim.teamId, Game.MannschaftGast.teamId);

  if Assigned(testGames) then
  begin
    for testGame in testGames do
    begin
      if (testGame <> IgnoreGame) and (testGame.DateValid) then
      begin
        roundNumberTest := GetRoundNumberByDate(testGame.Date);
        // Test, ob die in der gleichen Hauptrunde sind
        if (roundNumber div 2) = (roundNumberTest div 2) then
        begin
          // Gleiche Spiele in der gleichen Hauptrunde sind nicht erlaubt. Doppelrunde 2 mal gleiches Spiel vor Weihnachten. Und normale Runde -> Spiele wurde schon gemacht
          result := False;
          Exit;
        end;
      end;
    end;
  end;


  // jetzt die Rckspiele
  testGames := findGames(Game.MannschaftGast.teamId, Game.MannschaftHeim.teamId);

  if Assigned(testGames) then
  begin
    for testGame in testGames do
    begin
      if (testGame <> IgnoreGame) and (testGame.DateValid) then
      begin
        roundNumberTest := GetRoundNumberByDate(testGame.Date);

        if roundNumber = roundNumberTest then
        begin
          // Rckspiel darf nicht in der gleichen Runde sein
          result := False;
          Exit;
        end;
      end;
    end;
  end;
end;

procedure TPlan.FillTermin(Game: TGame; AllSecondTimeGamesAreValid: boolean);
var
  i: Integer;
  IndexList: array of Integer;
  IndexListCount: integer;
begin
  SetLength(IndexList, Game.MannschaftHeim.WunschTermine.Count);
  IndexListCount := Game.MannschaftHeim.WunschTermine.Count;
  for i:=0 to IndexListCount-1 do
  begin
    IndexList[i] := i;
  end;

  FillTerminIntern(IndexList, Game, AllSecondTimeGamesAreValid);
end;


// Versuche einen Wunschtemin zu finden, der maximal 1 Tag Abstand zu Game hat und versuche den einzuplanen
procedure TPlan.FillAuswaertsKoppelTermin(Game: TGame; Date: TDateTime; AllSecondTimeGamesAreValid: boolean);
var
  i: Integer;
  IndexList: array of Integer;
  NumFoundDates: integer;
begin
  NumFoundDates := 0;
  SetLength(IndexList, Game.MannschaftHeim.WunschTermine.Count + 1);
  for i:=0 to Game.MannschaftHeim.WunschTermine.Count-1 do
  begin
    if 1 >= Abs(Trunc(Date) - Trunc(Game.MannschaftHeim.WunschTermine[i].Date)) then
    begin
      IndexList[NumFoundDates] := i;
      Inc(NumFoundDates);
    end;
  end;

  if NumFoundDates > 0 then
  begin
    SetLength(IndexList, NumFoundDates);
    FillTerminIntern(IndexList, Game, AllSecondTimeGamesAreValid);
  end;
end;



procedure TPlan.FillTerminIntern(WunschTerminIndexList: array of Integer; Game: TGame; AllSecondTimeGamesAreValid: boolean);
var
  i: Integer;
  IndexListCount: integer;
  WunschTerminIndex: Integer;
  WunschTermin: TWunschTermin;
  Valid: boolean;
begin
  IndexListCount := Length(WunschTerminIndexList);

  while IndexListCount > 0 do
  begin
    i := Random(IndexListCount);

    WunschTerminIndex := WunschTerminIndexList[i];

    WunschTermin := Game.MannschaftHeim.WunschTermine[WunschTerminIndex];

    Valid := True;

    if Assigned(WunschTermin.assignedGame) then
    begin
      // Schon vergeben
      Valid := False;
    end;

    if not IsInRoundToGenerate(WunschTermin.Date) then
    begin
      // Nicht erlaubt
      Valid := False;
    end;


    if Valid then
    begin
      if woAuswaertsKoppelSecondTime in WunschTermin.Options then
      begin
        if not (woKoppelSecondTime in WunschTermin.Options) then
        begin
          if not Game.MannschaftGast.IsMannschaftInAuswaertsKoppelWunschAtOneDay(Self, Game.MannschaftHeim) then
          begin
            Valid := False;
          end;
        end;
      end;
    end;


    if Valid and TerminIsValid(WunschTermin, Game, nil, AllSecondTimeGamesAreValid) then
    begin
      Game.setDate(WunschTermin.Date, WunschTermin.Options, WunschTermin.ParallelGames, False, False, WunschTermin);
      if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
      begin
        MarkSecondGameAsUnNecessary(game);
      end;
      Break;
    end;

    // Der Termin ist nicht valid -> nehme ihn raus und probiere den nchsten
    // das Konstrukt hier entspricht IndexList.Delete(i)
    Dec(IndexListCount);
    System.Move(WunschTerminIndexList[i+1], WunschTerminIndexList[i], (IndexListCount - i) * sizeof(Integer));
  end;
end;


procedure TPlan.RemoveNotValidSecondTimeGames();
var
  game: TGame;
begin
  for game in gamesAllowed do
  begin
    if (woKoppelSecondTime in Game.DateOptions) or (woAuswaertsKoppelSecondTime in Game.DateOptions) then
    begin
      if not GameIsValid(game) then
      begin
        Game.setEmptyDate();
      end;
    end;
  end;
end;



procedure TPlan.FillTermine;
begin
  FillTermineIntern(True);
  if HasAuswaertsKoppelTermine or HasKoppelTermine then
  begin
    RemoveNotValidSecondTimeGames();
    FillTermineIntern(False);
  end;
end;



procedure TPlan.FillTermineIntern(AllSecondTimeGamesAreValid: boolean);
var
  i, j: Integer;
  ListOpenGames, GameList: TGameList;
  Game: TGame;
begin
  if getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpSecondOnly, rpCorona] then
  begin
    MarkNotNecessaryGames();
  end;

  ListOpenGames := TGameList.Create(False);
  for i:=0 to gamesAllowed.Count-1 do
  begin
    if gamesAllowed[i].Date = 0.0 then
    begin
      ListOpenGames.Add(gamesAllowed[i]);
    end;
  end;

  // Alle Spiele zufllig versuchen zu terminieren
  while ListOpenGames.Count > 0 do
  begin
    i := Random(ListOpenGames.Count);
    Game := ListOpenGames[i];

    if Game.NotNecessary then
    begin
      ListOpenGames.Delete(i);
      continue;
    end;

    FillTermin(Game, AllSecondTimeGamesAreValid);

    if HasAuswaertsKoppelTermine then
    begin
      if PercentRandom(70) then
      begin
        // Falls es ein Spiel ist, zu dem ein Auswtzskoppeltermin gewscht ist, dann versuche diesen zu terminieren
        GameList := Game.MannschaftGast.CreateAuswaertsKoppeledGamesDependenciesList(Self, Game.MannschaftHeim);

        if Assigned(GameList) then
        begin
          if GameList.Count > 0 then
          begin
            j := Random(GameList.Count);
            FillAuswaertsKoppelTermin(GameList[j], Game.Date, AllSecondTimeGamesAreValid);
          end;

          GameList.Free;
        end;
      end;
    end;

    ListOpenGames.Delete(i);
  end;

  ListOpenGames.Free;

  SortMannschaftsDates();
end;


procedure TPlan.TransferDateToRefDate;
var
  i: Integer;
  game: TGame;
begin
  for i:=0 to gamesAllowed.Count-1 do
  begin
    Game := gamesAllowed[i];
    game.FDateRef := game.FDate;
    game.FDateRefOptions := game.FDateOptions;
    game.FDateRefMaxHome := game.FDateMaxHome;
    game.FWunschTerminRef := game.FWunschTermin;
    game.FNotNecessaryRef := game.FNotNecessary;
  end;

  for i := 0 to Mannschaften.Count-1 do
  begin
    Mannschaften[i].KostenCacheRef := Mannschaften[i].KostenCache;
  end;

end;

procedure TPlan.TransferRefDateToDate;
var
  i: Integer;
  game: TGame;
begin
  for i:=0 to gamesAllowed.Count-1 do
  begin
    Game := gamesAllowed[i];
    if not game.FixedDate then
    begin
      game.setDate(game.FDateRef, game.FDateRefOptions, game.FDateRefMaxHome, False, game.FNotNecessaryRef, game.FWunschTerminRef);
    end;
  end;

  for i := 0 to Mannschaften.Count-1 do
  begin
    Mannschaften[i].KostenCache := Mannschaften[i].KostenCacheRef;
  end;

end;


procedure TPlan.FillPredefinedGame(gamePredefined: TGame);
var
  gameDest: TGame;
  games: TGameList;
  GameFound: TGame;
  WunschTermin, WunschTerminFound: TWunschTermin;
begin
  games := findGames(gamePredefined.MannschaftHeim.teamId, gamePredefined.MannschaftGast.teamId);

  if Assigned(games) and (games.Count > 0) then
  begin
    GameFound := nil;
    for gameDest in games do
    begin
      if gameDest.Date = gamePredefined.Date then
      begin
        GameFound := gameDest;
      end;
    end;

    if not Assigned(GameFound) then
    begin
      GameFound := games[0];
      for gameDest in games do
      begin
        if not gameDest.DateValid then
        begin
          GameFound := gameDest;
        end;
      end;
    end;

    WunschTerminFound := nil;
    for WunschTermin in GameFound.MannschaftHeim.WunschTermine do
    begin
      if WunschTermin.Date = gamePredefined.Date then
      begin
        WunschTerminFound := WunschTermin;
      end;
    end;

    if Assigned(WunschTerminFound) then
    begin
      GameFound.setDate(gamePredefined.Date, WunschTerminFound.Options, WunschTerminFound.ParallelGames, True, False, WunschTerminFound);
    end
    else
    begin
      // Das vordefinierte Spiel ist an keinem Wunschtermin
      GameFound.setDate(gamePredefined.Date, [], 0, True, False, nil);
    end;
  end;
end;


procedure TPlan.FillPredefinedGames();
var
  gamePredefined: TGame;
begin
  for gamePredefined in predefinedGames do
  begin
    FillPredefinedGame(gamePredefined);
  end;

  if getOptions.RoundPlaning in [rpSecondOnly, rpCorona] then
  begin
    for gamePredefined in existingSchedule do
    begin
      if not IsInRoundToGenerate(gamePredefined.Date) then
      begin
        FillPredefinedGame(gamePredefined);
      end;
    end;
  end;
end;

function Multiply(Value: MyDouble; MulValue: TCalculateOptionsValues): MyDouble;
begin
  if MulValue = coIgnore then
  begin
    Result := 0;
  end
  else
  begin
    Result := Value * (cCalculateOptionsValues[MulValue] / 100.0);
  end;
end;


function TPlan.GetMannschaftKostenFaktor(Mannschaft: TMannschaft; AktType: TMannschaftsKostenType): MyDouble;
begin
  Result := 100;

  Result := Multiply(Result, getOptions().GewichtungMainMannschaftsKosten[AktType]);
  Result := Multiply(Result, getOptions().GewichtungMannschaften.getMain(Mannschaft.teamId));
  Result := Multiply(Result, getOptions().GewichtungMannschaften.getDetail(Mannschaft.teamId, AktType));
end;

function Add(Value: integer; MulValue: TCalculateOptionsValues): integer;
begin
  if Value <= -1000 then
  begin
    Result := Value;
  end
  else
  if MulValue = coIgnore then
  begin
    Result := -1000;
  end
  else
  begin
    Result := Value + cCalculateOptionsDisplayInteger[MulValue];
  end;
end;


function TPlan.GetMannschaftKostenDisplayInteger(Mannschaft: TMannschaft; AktType: TMannschaftsKostenType): integer;
begin
  Result := 0;

  Result := Add(Result, getOptions().GewichtungMainMannschaftsKosten[AktType]);
  Result := Add(Result, getOptions().GewichtungMannschaften.getMain(Mannschaft.teamId));
  Result := Add(Result, getOptions().GewichtungMannschaften.getDetail(Mannschaft.teamId, AktType));
end;


function TPlan.getMaxGamePerMannschaft: integer;
var
  Mannschaft: TMannschaft;
begin
  Result := 0;

  for Mannschaft in Mannschaften do
  begin
    Result := max(Result, Mannschaft.gamesRef.Count);
  end;

end;

function TPlan.getOptions: TCalculateOptions;
begin
  Result := FOptions;
end;

function TPlan.GetRoundNumberByDate(Date: TDateTime): Integer;
var
  i: Integer;
begin
  Result := -1;
  for i := 0 to Rounds.Count-1 do
  begin
    if Rounds[i].DateInRound(Date) then
    begin
      Result := i;
      break;
    end;
  end;
end;

procedure TPlan.clearAllDates;
var
  i: integer;
  game: TGame;
begin
  ClearMannschaftWunschTerminRefs();
  ClearMannschaftGameRefs();
  for i:=0 to gamesAllowed.Count-1 do
  begin
    game := gamesAllowed[i];
    game.ClearAllValues;
  end;
  gamesNotAllowed.Clear;
  GenerateMannschaftsRefGames();
  ClearCache();
end;


procedure TPlan.SortMannschaftsDates;
var
  i: integer;
begin
  for i:=0 to Mannschaften.Count-1 do
  begin
    Mannschaften[i].gamesRef.SortByDate();
  end;
end;

{ TWunschTermin }

procedure TWunschTermin.Assign(Source: TWunschTermin);
begin
  Date := Source.Date;
  ParallelGames := Source.ParallelGames;
  KoppelSecondTime := Source.KoppelSecondTime;
  Location := Source.Location;
  Options := Source.Options;
end;

{ TWunschTerminList }

procedure TWunschTerminList.Assign(Source: TWunschTerminList);
var
  i: integer;
  NewElem: TWunschTermin;
begin
  Clear;
  for i:=0 to Source.Count-1 do
  begin
    NewElem := TWunschTermin.Create;
    NewElem.Assign(Source[i]);
    Add(NewElem);
  end;
end;

type
  TWunschTerminComparer = class(TComparer<TWunschTermin>)
    function Compare(const Left, Right: TWunschTermin): Integer; override;
  end;

function TWunschTerminComparer.Compare(const Left, Right: TWunschTermin): Integer;
begin
  Result := 0;
  if Left.Date > Right.Date then
    Result := 1;

  if Left.Date < Right.Date then
    Result := -1;

  if Left.Date = Right.Date then
    Result := 0;

end;


constructor TWunschTerminList.Create(OwnsObjects: boolean);
begin
  inherited Create(TWunschTerminComparer.Create, OwnsObjects);
end;


function TWunschTerminList.Exists(Date: TDateTime): boolean;
begin
  Result := Assigned(Find(Date));
end;

function TWunschTerminList.Find(Date: TDateTime): TWunschTermin;
var
  i: Integer;
begin
  Result := nil;
  for i := 0 to Count-1 do
  begin
    if Items[i].Date = Date then
    begin
      Result := Items[i];
      break;
    end;
  end;
end;



function TWunschTerminList.ExistsIgnoreTime(Date: TDateTime): boolean;
var
  i: Integer;
begin
  Result := False;
  for i := 0 to Count-1 do
  begin
    if Trunc(Items[i].Date) = Trunc(Date) then
    begin
      Result := True;
    end;
  end;
end;


{ TMannschaft }

procedure TMannschaft.Assign(Source: TMannschaft);
begin
  teamName := Source.teamName;
  clubID := Source.clubID;
  teamId := Source.teamId;
  DefaultLocation := Source.DefaultLocation;
  teamNumber := Source.teamNumber;
  sisterGames.Assign(Source.sisterGames);
  noWeekGame.Clear;
  noWeekGame.AddRange(Source.noWeekGame.ToArray);

  SperrTermine.Assign(Source.SperrTermine);
  SperrTermine.Sort;

  AusweichTermine.Assign(Source.AusweichTermine);
  AusweichTermine.Sort;

  AuswaertsKoppel.Assign(Source.AuswaertsKoppel);

  HomeRightCountRoundOne := Source.HomeRightCountRoundOne;
  HomeRights.Assign(Source.HomeRights);


  WunschTermine.Assign(Source.WunschTermine);
  KostenAnzahl := Source.KostenAnzahl;
  KostenAbstand := Source.KostenAbstand;
  KostenWochentag := Source.KostenWochentag;

  SisterTeamsInPlan.Clear;

  ClearCache();
end;

procedure TMannschaft.CalculateKostenForType(Plan: TPlan; KostenType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
begin
  case KostenType of
    mktHalleBelegt:
        CalculateKostenHallenBelegung(Plan, KostenType, Kosten, WithMessages);
    mktSisterGames:
        CalculateKostenParallelGames(Plan, KostenType, Kosten, WithMessages);
    mktForceSameHomeGames:
        CalculateKostenSameHomeDates(Plan, KostenType, Kosten, WithMessages);
    mktSperrTermine:
        CalculateKostenSperrTermine(Plan, KostenType, Kosten, WithMessages);
    mktAusweichTermine:
        CalculateKostenAusweichTermine(Plan, KostenType, Kosten, WithMessages);
    mkt60Kilometer:
        CalculateKosten60kMRegel(Plan, KostenType, Kosten, WithMessages);
    mktEngeTermine:
        CalculateKostenEngeTermine(Plan, KostenType, Kosten, WithMessages);
    mktSpielverteilung:
        CalculateKostenSpielverteilung(Plan, KostenType, Kosten, WithMessages);
    mkt2SpieleProWoche:
        CalculateKosten2SpieleDieWoche(Plan, KostenType, Kosten, WithMessages);
    mktKoppelTermine:
        CalculateKostenKoppelTermine(Plan, KostenType, Kosten, WithMessages);
    mktAuswaertsKoppelTermine:
        CalculateKostenAuswaertsKoppelTermine(Plan, KostenType, Kosten, WithMessages);
    mktMandatoryGames:
        CalculateKostenMandatoryGames(Plan, KostenType, Kosten, WithMessages);
    mktZahlHeimSpielTermine:
        CalculateKostenZahlHeimSpiele(Plan, KostenType, Kosten, WithMessages);
    mktWechselHeimAuswaerts:
        CalculateKostenWechselHeimAuswaerts(Plan, KostenType, Kosten, WithMessages);
    mktAbstandHeimAuswaerts:
        CalculateKostenAbstandHeimAuswaerts(Plan, KostenType, Kosten, WithMessages);
    mktRanking:
        CalculateKostenRanking(Plan, KostenType, Kosten, WithMessages);
    else
      raise Exception.Create('Keine Berechnungsfunktion fr Kostentyp');
  end;
end;

function  TMannschaft.getCache(Plan: TPlan): TMannschaftDataCache;
var
  Elem: TAuswaertsKoppel;
  TempSisterGames: TSisterGames;
  TempsisterGame: TSisterGame;
  TempsisterGamesMap: TArray<TSisterGames>;
begin
  if DataCache <> nil then
  begin
    Result := DataCache;
  end
  else
  begin
    Result := TMannschaftDataCache.Create;
    // Die Liste der Mannschaften aufbauen, bei welchen an einem Tag gespielt werden soll
    Result.CachedAuswaertsKoppelTeams.TeamsOnDay.Clear;
    for Elem in AuswaertsKoppel do
    begin
      if Elem.Option in [ktSameDay, ktAllDay] then
      begin
        if Assigned(Elem.getMannschaftA(Plan)) and Assigned(Elem.getMannschaftB(Plan)) then
        begin

          if not Result.CachedAuswaertsKoppelTeams.TeamsOnDay.Contains(Elem.TeamA) then
             Result.CachedAuswaertsKoppelTeams.TeamsOnDay.Add(Elem.TeamA);

          if not Result.CachedAuswaertsKoppelTeams.TeamsOnDay.Contains(Elem.TeamB) then
             Result.CachedAuswaertsKoppelTeams.TeamsOnDay.Add(Elem.TeamB);

        end;
      end;
    end;

    // Pro Mannschaft die Liste der mglichen 2. Mannschaft zum Koppeltermin aufbauen
    Result.CachedAuswaertsKoppelTeams.TeamIdToTeamId.Clear;
    for Elem in AuswaertsKoppel do
    begin
      if Assigned(Elem.getMannschaftA(Plan)) and Assigned(Elem.getMannschaftB(Plan)) then
      begin
        Result.CachedAuswaertsKoppelTeams.AddTeamIdToTeamId(Elem.getMannschaftA(Plan).teamId, Elem.getMannschaftB(Plan).teamId);
        Result.CachedAuswaertsKoppelTeams.AddTeamIdToTeamId(Elem.getMannschaftB(Plan).teamId, Elem.getMannschaftA(Plan).teamId);
      end;
    end;

    // Pro Mannschaft die Liste der SisterTeams mit denen am gleichen Tag die Heimspiele Stattfinden sollen
    Result.CachedTeamsForForceHomeGames := TStringList.Create;
    Result.CachedTeamsForForceHomeGames.Duplicates := dupIgnore;
    Result.CachedTeamsForForceHomeGames.Sorted := True;

    TempSisterGamesMap := sisterGames.toArray();
    for TempSisterGames in TempSisterGamesMap do
    begin
      for TempSisterGame in TempSisterGames do
      begin
        if TempSisterGame.ForceParallelHomeGames then
        begin
          Result.CachedTeamsForForceHomeGames.Add(TempsisterGame.gender + #10 + TempsisterGame.teamName);
        end;
      end;
    end;

    FreeAndNil(DataCache);
    DataCache := Result;
  end;
end;

procedure TMannschaft.CalculateKosten(Plan: TPlan; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  KostenType: TMannschaftsKostenType;
begin
  for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    CalculateKostenForType(Plan, KostenType, Kosten, WithMessages);
  end;
end;


const
  MAX_MANNSCHAFT = 30;
  MAX_ROUND = 10;


procedure TMannschaft.CalculateKostenAbstandHeimAuswaerts(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i, j: Integer;
  values: TList<Integer>;
  value: MyDouble;
  MannschaftGegner: TMannschaft;
  Mittelwert: MyDouble;
  // Fr jede Mannschaft eine Liste, an welchen Spieltagen diese Mannschaften aufeinandertreffen.
  PosNumberPerMannschaft: array[0..MAX_MANNSCHAFT, -1..MAX_ROUND] of Integer;
  MannschaftsIndex: integer;
  GameIndex: Integer;
  EchteAbweichung: integer;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktAbstandHeimAuswaerts);
  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  if (Plan.getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpCorona]) and (not Plan.getOptions.DoubleRound) then
  begin
    // Diese Kostenart ergibt hier keinen Sinn
    Exit;
  end;


  if MAX_MANNSCHAFT < Plan.Mannschaften.Count then
  begin
    raise Exception.Create('Der Plan enthlt zu viele Mannschaften. Fr maximal ' + IntToStr(MAX_MANNSCHAFT) + ' kann der Plan generiert werden');
  end;


  for i := 0 to MAX_MANNSCHAFT do
  begin
    PosNumberPerMannschaft[i, -1] := 0;
  end;

  values := TList<Integer>.Create;

  for i:=0 to gamesRef.Count-1 do
  begin
    if not gamesRef[i].DateValid then
    begin
      continue;
    end;

    if gamesRef[i].MannschaftHeim = Self then
    begin
      MannschaftGegner := gamesRef[i].MannschaftGast;
    end
    else
    begin
      MannschaftGegner := gamesRef[i].MannschaftHeim;
    end;

    MannschaftsIndex := MannschaftGegner.Index;
    GameIndex := PosNumberPerMannschaft[MannschaftsIndex, -1];
    if GameIndex >= MAX_ROUND-1  then
    begin
      // Viel zu viele Spiele gegen diese Mannschaft. Knnte theoretisch passieren, wenn der Plan in Click-TT total Gaga ist
      continue;
    end;

    PosNumberPerMannschaft[MannschaftsIndex, GameIndex] := i;
    Inc(GameIndex);
    PosNumberPerMannschaft[MannschaftsIndex, -1] := GameIndex;


  end;

  values.Capacity := Plan.Mannschaften.Count * 4;
  for j := 0 to Plan.Mannschaften.Count-1 do
  begin
    if j <> Self.Index then
    begin
      for i := 0 to PosNumberPerMannschaft[j, -1]-2 do
      begin
        //Abstand zum nchsten Spiel mit der gleichen Mannschaft in die Values
        values.Add(PosNumberPerMannschaft[j, i+1] - PosNumberPerMannschaft[j, i+0]);
      end;
    end;
  end;

  CalculateMaxAbweichung(EchteAbweichung, value, values, Mittelwert, 2, 1);
  Kosten.Values[AktType].Anzahl := EchteAbweichung;

  if WithMessages <> tmNoMessage then
  begin
    Kosten.Messages.Add('Maximale Abweichung Reihenfolge Heim- und Auswrtsspiel: ' + MyFormatFloat(Kosten.Values[AktType].Anzahl));
  end;

  value := (value / Mittelwert);

  value := Power(value * 6, 6) / 5000.0 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  values.Free;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;

procedure TMannschaft.CalculateKostenRanking(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  values: TList<Integer>;
  value: MyDouble;
  MannschaftGegner: TMannschaft;
  EchteAbweichung: integer;
  CachedValue: MyDouble;
  currZiel: integer;
  currRound: integer;
  diff: integer;
  Faktor: MyDouble;
begin
  Assert(AktType = mktRanking);
  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  if (not Plan.HasRanking) then
  begin
    // Nicht aktiv
    Exit;
  end;

  values := TList<Integer>.Create;

  getRankingDiffs(Plan, values);

  currZiel := Plan.Mannschaften.Count - 1;
  currRound := -1;

  value := 0;
  EchteAbweichung := 0;

  Faktor := 1.0;

  for i:=0 to gamesRef.Count-1 do
  begin
    if not gamesRef[i].DateValid then
    begin
      continue;
    end;

    if currRound <> Plan.GetRoundNumberByDate(gamesRef[i].Date) then
    begin
      currZiel := Plan.Mannschaften.Count - 1;
      currRound := Plan.GetRoundNumberByDate(gamesRef[i].Date);

      if currRound = Plan.Rounds.Count-1 then
      begin
        // Letzte Runde wird hher bewertet
        Faktor := 4.0;
      end;
    end
    else
    begin
      Dec(currZiel);
    end;

    diff := values[i] - currZiel;
    if (diff > 0) and (currZiel > 0) then
    begin
      value := value + Faktor * (diff / power(currZiel, 1.25));

      Inc(EchteAbweichung);

      if WithMessages = tmDetailMessage then
      begin

        if gamesRef[i].MannschaftHeim = Self then
        begin
          MannschaftGegner := gamesRef[i].MannschaftGast;
        end
        else
        begin
          MannschaftGegner := gamesRef[i].MannschaftHeim;
        end;

        Kosten.Messages.Add('Spiel gegen ' + MannschaftGegner.teamName + ' sollte ' + IntToStr(diff) + ' Spieltage frher stattfinden');
      end;

    end;
  end;

  Kosten.Values[AktType].Anzahl := EchteAbweichung;

  value := Power(value / 8.0, 3) * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  values.Free;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;


procedure TMannschaft.getRankingDiffs(Plan: TPlan; ResultValues: TList<Integer>);
var
  i: integer;
  MannschaftGegner: TMannschaft;
  MyRank: integer;
  GegnerRank: integer;
begin
  ResultValues.Clear;
  MyRank := Plan.RankingIndexe.IndexOf(Self.Index);
  for i:=0 to gamesRef.Count-1 do
  begin
    if gamesRef[i].MannschaftHeim = Self then
    begin
      MannschaftGegner := gamesRef[i].MannschaftGast;
    end
    else
    begin
      MannschaftGegner := gamesRef[i].MannschaftHeim;
    end;

    GegnerRank := Plan.RankingIndexe.IndexOf(MannschaftGegner.Index);

    if (GegnerRank < 0) or (MyRank < 0) then
    begin
      ResultValues.Add(0);
    end
    else
    begin
      ResultValues.Add(Abs(MyRank - GegnerRank));
    end;
  end;
end;


procedure TMannschaft.CalculateKostenMandatoryGames(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  i: Integer;
  MandatoryElem: TPlanMandatoryTime;
  Count: integer;
  AllCount: integer;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktMandatoryGames);

  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  AllCount := 0;
  for MandatoryElem in Plan.MandatoryGameDays do
  begin
    if (Plan.IsInRoundToGenerate(MandatoryElem.DateFrom)) then
    begin
      Inc(AllCount, MandatoryElem.GameCount);
      Count := 0;
      for i:= gamesRef.getFirstGameRefIndexByDate(MandatoryElem.DateFrom) to gamesRef.Count-1 do
      begin
        if gamesRef[i].DateValid then
        begin
          if (Trunc(gamesRef[i].Date) >= MandatoryElem.DateFrom)
              and (Trunc(gamesRef[i].Date) <= MandatoryElem.DateTo)
          then
          begin
            Inc(Count);
          end;

          if (Trunc(gamesRef[i].Date) > MandatoryElem.DateTo) then
          begin
            break;
          end;
        end;
      end;

      if Count < MandatoryElem.GameCount then
      begin
        Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + (MandatoryElem.GameCount - Count);
        if WithMessages <> tmNoMessage then
        begin
          Kosten.Messages.Add(IntToStr((MandatoryElem.GameCount - Count)) + ' Spiel(e) zu wenig vom ' + MyFormatDate(MandatoryElem.DateFrom) + ' - ' + MyFormatDate(MandatoryElem.DateTo));
        end;
      end;
    end;
  end;

  value := Kosten.Values[AktType].Anzahl * 1.0;
  value := Power(value, 2) * 1000 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;
  Kosten.Values[AktType].AllCount := AllCount;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;




procedure TMannschaft.CalculateKostenWechselHeimAuswaerts(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  NumSame: Integer;
  LastIsHeim, AktIsHeim: Boolean;
  value: MyDouble;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktWechselHeimAuswaerts);
  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  NumSame := 0;
  LastIsHeim := False;
  value := 0;
  for i:=0 to gamesRef.Count-1 do
  begin
    if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date))  then
    begin
      AktIsHeim := (gamesRef[i].MannschaftHeim = Self);
      if( i <> 0) then
      begin
        if AktIsHeim = LastIsHeim then
        begin
          // Bestimme, ob der Vorgnger ein Koppeltermin ist
          if gamesRef[i-1] = getKoppeledGame(i) then
          begin
            // Der Vorgnger ist ein Koppeltermin, und da ist das ja gewnscht -> mache nix
          end
          else
          begin
            Inc(NumSame);
            Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + 1;
            value := value + Power(NumSame, 4);

          end;
        end
        else
        begin
          NumSame := 0;
        end;
      end;
      LastIsHeim := AktIsHeim;
    end;
  end;


  if WithMessages <> tmNoMessage then
  begin
    Kosten.Messages.Add('Kein Wechsel Heimspiel und Auswrtsspiel: ' + IntToStr(Trunc(Kosten.Values[AktType].Anzahl)));
  end;

  value := (Power(value, 2) / 30.0) * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;




procedure TMannschaft.CalculateKostenZahlHeimSpiele(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  NumInRound: TList<Integer>;
  i: Integer;
  RoundNumber: Integer;
  ZielNum: MyDouble;
  Fehler: Integer;
  S: String;
  CachedValue: MyDouble;
  ValueHome: THomeRightValue;
  ZielIsHome, IsHome: boolean;
  TeamHomeRight: integer;
  IgnoreRounds: boolean;
  game: TGame;
begin
  Assert(AktType = mktZahlHeimSpielTermine);
  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  IgnoreRounds := (Plan.getOptions.RoundPlaning = rpCorona) or ((Plan.getOptions.RoundPlaning = rpHalfRound) and (not Plan.getOptions.DoubleRound));

  NumInRound := TList<Integer>.Create();

  if IgnoreRounds then
  begin
    NumInRound.Add(0);
  end
  else
  begin
    for i := 0 to Plan.Rounds.Count-1 do
    begin
      NumInRound.Add(0);
    end;
  end;

  TeamHomeRight := 0;
  if HomeRights.Count > 0 then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if gamesRef[i].DateValid then
      begin
        if gamesRef[i].MannschaftHeim = Self then
        begin
          ValueHome := HomeRights.getValueForTeam(gamesRef[i].MannschaftGast.teamName);
          IsHome := True;
        end
        else
        begin
          ValueHome := HomeRights.getValueForTeam(gamesRef[i].MannschaftHeim.teamName);
          IsHome := False;
        end;

        if ValueHome <> hrNone then
        begin
          if IgnoreRounds then
          begin
            RoundNumber := 0;
          end
          else
          begin
            RoundNumber := Plan.GetRoundNumberByDate(gamesRef[i].Date);
          end;

          ZielIsHome := RoundNumber mod 2 = 0;

          if ValueHome = hrRound2 then
          begin
            ZielIsHome := not ZielIsHome;
          end;

          if IsHome <> ZielIsHome then
          begin
            Inc(TeamHomeRight);

            if WithMessages <> tmNoMessage then
            begin
              S := gamesRef[i].AsString;

              if ZielIsHome then
              begin
                S := S + ' sollte ein Heimspiel sein';
              end
              else
              begin
                S := S + ' sollte ein Auswrtsspiel sein';
              end;
              Kosten.Messages.Add(S);
            end;
          end;
        end;
      end;
    end;
  end;

  for i:=0 to gamesRef.Count-1 do
  begin
    if gamesRef[i].DateValid then
    begin
      if gamesRef[i].MannschaftHeim = Self then
      begin
        if IgnoreRounds then
        begin
          RoundNumber := -1;
          if gamesRef[i].Date <> 0 then
          begin
            RoundNumber := 0;
          end;
        end
        else
        begin
          RoundNumber := Plan.GetRoundNumberByDate(gamesRef[i].Date);
        end;

        if RoundNumber >= 0 then
        begin
          NumInRound[RoundNumber] := NumInRound[RoundNumber] + 1;
        end;
      end;
    end;
  end;

  if IgnoreRounds then
  begin
    ZielNum := 0;
    for game in gamesRef do
    begin
      if not game.NotNecessary then
      begin
        ZielNum := ZielNum + 1;
      end;
    end;
    ZielNum := (ZielNum * 0.5) / 1;
  end
  else
  begin
    ZielNum := (gamesRef.Count * 0.5) / Plan.Rounds.Count;
    if (Plan.getOptions.RoundPlaning = rpHalfRound) and (Plan.getOptions.DoubleRound) then
    begin
      ZielNum := (gamesRef.Count * 0.25) / Plan.Rounds.Count;
    end;

  end;

  Kosten.Values[AktType].Anzahl := 0;

  if IgnoreRounds then
  begin
    // Mehr oder 1 Spiel Unterschied zum Ziel -> ergibt Fehler.
    // Ist Zielspielzahl z.B. 4.5, dann ist der Fehler 0, wegen Trunc
    Fehler := Trunc(Abs(NumInRound[0] - ZielNum));
    if (Fehler > 0) and (WithMessages <> tmNoMessage) and (HomeRightCountRoundOne >= 0) then
    begin
      S := 'Anzahl Heimspiele: ';
      S := S + IntToStr(NumInRound[0]) + ' anstatt ';
      S := S + IntToStr(Trunc(ZielNum));
      Kosten.Messages.Add(S);
    end;
    Kosten.Values[AktType].Anzahl := Fehler + Kosten.Values[AktType].Anzahl;
  end
  else
  begin
    for i := 0 to Plan.Rounds.Count-1 do
    begin
      if Plan.IsRoundToGenerate(Plan.Rounds[i]) then
      begin
        if HomeRightCountRoundOne >= 0 then
        begin
          if i mod 2 = 0 then
          begin
            ZielNum := HomeRightCountRoundOne;
          end
          else
          begin
            ZielNum := Plan.Mannschaften.Count - HomeRightCountRoundOne - 1;
          end;
        end;
        // Mehr oder 1 Spiel Unterschied zum Ziel -> ergibt Fehler.
        // Ist Zielspielzahl z.B. 4.5, dann ist der Fehler 0, wegen Trunc
        Fehler := Trunc(Abs(NumInRound[i] - ZielNum));

        if (Fehler > 0) and (WithMessages <> tmNoMessage) and (HomeRightCountRoundOne >= 0) then
        begin
          if i mod 2 = 0 then
            S := 'Anzahl Heimspiele in Vorrunde: '
          else
            S := 'Anzahl Heimspiele in Rckrunde: ';

          S := S + IntToStr(NumInRound[i]) + ' anstatt ';
          S := S + IntToStr(Trunc(ZielNum));

          Kosten.Messages.Add(S);
        end;


        Kosten.Values[AktType].Anzahl := Fehler + Kosten.Values[AktType].Anzahl;
      end;
    end;
  end;

  if not IgnoreRounds then
  begin
    if Kosten.Values[AktType].Anzahl <> 0 then
    begin
      if (WithMessages <> tmNoMessage) and (HomeRightCountRoundOne < 0) then
      begin

        S := 'Anzahl Heimspiele in den Runden unterschiedlich: ';

        for i := 0 to Plan.Rounds.Count-1 do
        begin
          if (not (Plan.getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpCorona])) or (Plan.IsRoundToGenerate(Plan.Rounds[i])) then
          begin
            if i <> 0 then
            begin
              S := S + ' - ';
            end;
            S := S + IntToStr(NumInRound[i]);
          end;
        end;

        Kosten.Messages.Add(S);
      end;
    end;
  end;

  value := Kosten.Values[AktType].Anzahl + TeamHomeRight * 2;

  value := Power(value, 2) * 300 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  NumInRound.Free;

  Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + TeamHomeRight;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;





procedure TMannschaft.ClearCache();
var
  index: TMannschaftsKostenType;
begin
  for index := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    KostenCache.Values[Index].GesamtKosten := -1;
  end;
  KostenCache.Valid := False;

  for index := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    KostenCacheRef.Values[Index].GesamtKosten := -1;
  end;
  KostenCacheRef.Valid := False;

  FreeAndNil(DataCache);
end;

procedure TMannschaft.ClearKostenCache();
var
  index: TMannschaftsKostenType;
begin
  if KostenCache.Valid then
  begin
    for index := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      KostenCache.Values[Index].GesamtKosten := -1;
    end;
  end;
  KostenCache.Valid := False;
end;

type TKoppelDatesPerRound = record
   HighPrio: integer;
   LowPrio: integer;
   ResultValuePrioHigh: integer;
   ResultValuePrioLow: integer;
   OptimumValue: MyDouble;
   ResultValue: MyDouble;
end;


procedure TMannschaft.CalculateKostenKoppelTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  valueResult: MyDouble;
  koppelType: TWunschterminOptions;
  i: Integer;
  AllCount: Integer;
  CachedValue: MyDouble;
  KoppelDatesPerRound: array [0..MAX_ROUND] of TKoppelDatesPerRound;
  RoundNo: integer;
  MaxKoppelPerRound: integer;
  OptHigh, OptLow, RealHigh, RealLow: integer;

const
  LOW_PRIO_FACTOR = 0.25;
begin
  Assert(AktType = mktKoppelTermine);
  Kosten.ClearValues(AktType);

  FillMemory(@KoppelDatesPerRound, sizeof(KoppelDatesPerRound), 0);

  if not Plan.HasKoppelTermine then
  begin
    Exit;
  end;

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;


  AllCount := 0;

  for i := 0 to WunschTermine.Count-1 do
  begin
    if OptionIsKoppel(WunschTermine[i].Options) then
    begin
      if Plan.IsInRoundToGenerate(WunschTermine[i].Date) then
      begin
        Inc(AllCount);
        RoundNo := Plan.GetRoundNumberByDate(WunschTermine[i].Date);

        if OptionIsKoppelHard(WunschTermine[i].Options) then
        begin
          Inc(KoppelDatesPerRound[RoundNo].HighPrio);
        end;

        if OptionIsKoppelSoft(WunschTermine[i].Options) then
        begin
          Inc(KoppelDatesPerRound[RoundNo].LowPrio);
        end;
      end;
    end;
  end;

  if AllCount > 0 then
  begin
    // Zwei gekoppelte Wunschtermine sind ein Koppeltermin
    AllCount := AllCount div 2;

    // Jetzt ausrechnen, was der Optimalwert wre. Pro Runde sind nur ein Viertel der gesamten Spiele als Koppeltermin mglich
    MaxKoppelPerRound := (Plan.Mannschaften.Count-1) div 4;
    for i := 0 to Plan.Rounds.Count-1 do
    begin
      // Zwei gekoppelte Wunschtermine sind ein Koppeltermin
      KoppelDatesPerRound[i].HighPrio := KoppelDatesPerRound[i].HighPrio div 2;
      KoppelDatesPerRound[i].LowPrio := KoppelDatesPerRound[i].LowPrio div 2;

      if KoppelDatesPerRound[i].HighPrio >= MaxKoppelPerRound then
      begin
        KoppelDatesPerRound[i].HighPrio := MaxKoppelPerRound;
        KoppelDatesPerRound[i].LowPrio := MaxKoppelPerRound;
      end;

      if KoppelDatesPerRound[i].LowPrio + KoppelDatesPerRound[i].HighPrio > MaxKoppelPerRound then
      begin
        KoppelDatesPerRound[i].LowPrio := MaxKoppelPerRound - KoppelDatesPerRound[i].HighPrio;
      end;

      KoppelDatesPerRound[i].OptimumValue := KoppelDatesPerRound[i].HighPrio + (KoppelDatesPerRound[i].LowPrio * LOW_PRIO_FACTOR);
    end;



    Kosten.Values[AktType].AllCount := AllCount div 2;

    for i:=0 to gamesRef.Count-1 do
    begin
      if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
      begin
        koppelType := IsKoppelTermin(gamesRef[i]);
        if (woKoppelDoubleDatePossible in koppelType) or (woKoppelPossible in koppelType) then
        begin
          // Ist zwar erlaubt, erzeugt aber keine Kosten, wenn es nicht erfllt ist
          Continue;
        end;

        if OptionIsKoppel(koppelType) then
        begin
          // Nur der wird als Koppeltermin gezhlt, der einen Nachfolgetermin hat
          // sonst wrde in der Schleife jeder Koppeltermin doppelt gezhlt
          if Assigned(getKoppeledGameAfter(i)) then
          begin
            RoundNo := Plan.GetRoundNumberByDate(gamesRef[i].Date);
            if (woKoppelHard in koppelType) or (woKoppelDoubleDateHard in koppelType) then
            begin
              KoppelDatesPerRound[RoundNo].ResultValue := KoppelDatesPerRound[RoundNo].ResultValue + 1.0;
              Inc(KoppelDatesPerRound[RoundNo].ResultValuePrioHigh);
            end
            else
            begin
              KoppelDatesPerRound[RoundNo].ResultValue := KoppelDatesPerRound[RoundNo].ResultValue + LOW_PRIO_FACTOR;
              Inc(KoppelDatesPerRound[RoundNo].ResultValuePrioLow);
            end;
          end;
        end;



        (*
        if OptionIsKoppel(koppelType) then
        begin
          if not Assigned(getKoppeledGame(i)) then
          begin
            Kosten.Values[AktType].AnzahlFehler := Kosten.Values[AktType].AnzahlFehler + 1;

            if (woKoppelHard in koppelType) or (woKoppelDoubleDateHard in koppelType) then
            begin
              valueResult := valueResult + 1;
            end
            else
            begin
              valueResult := valueResult + 0.25;
            end;

            if WithMessages then
            begin
              if woKoppelHard in koppelType then
              begin
                Kosten.Messages.Add('Kein unbedingter Koppeltermin: ' + MyFormatDate(gamesRef[i].Date));
              end
              else
              if woKoppelDoubleDateHard in koppelType then
              begin
                Kosten.Messages.Add('Kein gewnschter Doppelspieltag (hohe Prio): ' + MyFormatDate(gamesRef[i].Date));
              end
              else
              if woKoppelDoubleDateSoft in koppelType then
              begin
                Kosten.Messages.Add('Kein gewnschter Doppelspieltag: ' + MyFormatDate(gamesRef[i].Date));
              end
              else
              begin
                Kosten.Messages.Add('Kein "wenn mglich" Koppeltermin: ' + MyFormatDate(gamesRef[i].Date));
              end;
            end;
          end;
        end;
        *)
      end;
    end;

    valueResult := 0;
    Kosten.Values[AktType].Anzahl := 0;
    Kosten.Values[AktType].AllCount := 0;

    for i := 0 to Plan.Rounds.Count-1 do
    begin
      if KoppelDatesPerRound[i].OptimumValue > KoppelDatesPerRound[i].ResultValue then
      begin
        valueResult := valueResult + (KoppelDatesPerRound[i].OptimumValue - KoppelDatesPerRound[i].ResultValue);
      end;

      Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + KoppelDatesPerRound[i].ResultValuePrioHigh + KoppelDatesPerRound[i].ResultValuePrioLow;
      Kosten.Values[AktType].AllCount := Kosten.Values[AktType].AllCount + KoppelDatesPerRound[i].HighPrio + KoppelDatesPerRound[i].LowPrio;
    end;

    // Die nicht vergebenen anzeigen. Das ist das Maximum - den vergebenen.
    Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].AllCount - Kosten.Values[AktType].Anzahl;



    if WithMessages <> tmNoMessage then
    begin
      OptHigh := 0;
      OptLow := 0;
      RealHigh := 0;
      RealLow := 0;
      for i := 0 to Plan.Rounds.Count-1 do
      begin
        Inc(OptHigh, KoppelDatesPerRound[i].HighPrio);
        Inc(OptLow, KoppelDatesPerRound[i].LowPrio);
        Inc(RealHigh, KoppelDatesPerRound[i].ResultValuePrioHigh);
        Inc(RealLow, KoppelDatesPerRound[i].ResultValuePrioLow);
      end;

      if OptHigh > RealHigh then
      begin
        Kosten.Messages.Add('Unbedingte Heimkoppeltermine/Doppelspieltage wurden nicht vergeben: ' + IntToStr(OptHigh - RealHigh) + ' mal');
      end;
      if OptLow > RealLow then
      begin
        Kosten.Messages.Add('"Wenn mglich" Heimkoppeltermine/Doppelspieltage wurden nicht vergeben: ' + IntToStr(OptLow - RealLow) + ' mal');
      end;
    end;


    valueResult := valueResult * 10.0;
    valueResult := Power(valueResult, 2) * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := valueResult;
  end;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;

procedure TMannschaft.CalculateKostenAuswaertsKoppelTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  Koppel: TAuswaertsKoppel;
  i: Integer;
  game1: TGame;
  game2: TGame;
  TeamA, TeamB: TMannschaft;
  valueResult: MyDouble;
  KoppelMannschaftCounts: array[0 .. MAX_MANNSCHAFT] of Integer;
  ZielValue: integer;
  Factor: MyDouble;
  WunschCount: integer;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktAuswaertsKoppelTermine);
  Kosten.ClearValues(AktType);
  Kosten.Values[AktType].AllCount := 0;

  if (not Plan.HasAuswaertsKoppelTermine) or (AuswaertsKoppel.Count = 0) then
  begin
    Exit;
  end;

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  FillMemory(@KoppelMannschaftCounts, sizeof(KoppelMannschaftCounts), 0);

  ZielValue := Plan.Rounds.Count div 2;

  for i:=0 to AuswaertsKoppel.Count-1 do
  begin
    Koppel := AuswaertsKoppel[i];
    TeamA := Koppel.getMannschaftA(Plan);
    TeamB := Koppel.getMannschaftB(Plan);

    if Assigned(TeamA) and Assigned(TeamB) then
    begin
      KoppelMannschaftCounts[TeamA.Index] := ZielValue;
      KoppelMannschaftCounts[TeamB.Index] := ZielValue;
    end;
  end;

  WunschCount := 0;
  for i := 0 to MAX_MANNSCHAFT do
  begin
    if KoppelMannschaftCounts[i] > 0 then
    begin
      Inc(WunschCount);
    end;
  end;

  for i := 0 to gamesRef.Count-1 do
  begin
    game1 := gamesRef[i];
    if game1.MannschaftGast = Self then
    begin
      game2 := getKoppeledGameAfter(i);
      if Assigned(game2) then
      begin
        teamA := game1.MannschaftHeim;
        teamB := game2.MannschaftHeim;
        KoppelMannschaftCounts[TeamA.Index] := KoppelMannschaftCounts[TeamA.Index] - 1;
        KoppelMannschaftCounts[TeamB.Index] := KoppelMannschaftCounts[TeamB.Index] - 1;
      end;
    end;
  end;

  for i := 0 to MAX_MANNSCHAFT do
  begin
    if KoppelMannschaftCounts[i] > 0 then
    begin
      Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + (KoppelMannschaftCounts[i]);

      if WithMessages <> tmNoMessage then
      begin
        Kosten.Messages.Add('Kein Auswrtskoppeltermin bei: ' + Plan.Mannschaften[i].teamName);
      end;
    end;
  end;

  Factor := Max(0, (Min(6, WunschCount - 2))) / 6;
  Factor := 1.0 - 0.75 * Factor;

  valueResult := Kosten.Values[AktType].Anzahl * 5.0;
  valueResult := Power(valueResult, 2) * 4.0 * Plan.GetMannschaftKostenFaktor(Self, AktType) * Factor;

  Kosten.Values[AktType].GesamtKosten := valueResult;

  // 2 Fehler sind eigentlich nur einer (zwei Mannschaften wurden nicht bei Auswrtskoppel bercksichtigt)
  Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl div 2;
  Kosten.Values[AktType].AllCount := WunschCount div 2;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;




procedure TMannschaft.CalculateKosten2SpieleDieWoche(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  DiffWeeks: Integer;
  i: Integer;
  CachedValue: MyDouble;
begin
  Assert(AktType = mkt2SpieleProWoche);

  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  for i:=0 to gamesRef.Count-2 do
  begin
    if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
    begin
      DiffWeeks := InternalWeekNumber(gamesRef[i+1].Date) - InternalWeekNumber(gamesRef[i].Date);

      if DiffWeeks = 0 then
      begin
        if not (gamesRef[i+1] = getKoppeledGame(i)) then
        begin
          Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + 1;
          if WithMessages <> tmNoMessage then
          begin
            Kosten.Messages.Add('Zwei Spiele die Woche: ' + MyFormatDate(gamesRef[i].Date) + ' - ' + MyFormatDate(gamesRef[i+1].Date));
          end;
        end;
      end;
    end;
  end;

  value := Kosten.Values[AktType].Anzahl * 1.0;
  value := Power(value, 2) * 400 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;



function TMannschaft.getKoppeledGameAfter(IndexInRefGames: integer): TGame;
begin
  Result := nil;
  if IndexInRefGames < gamesRef.Count-1 then
  begin
    if InternIsKoppeledGames(gamesRef[IndexInRefGames], gamesRef[IndexInRefGames+1]) then
    begin
      if Assigned(getKoppeledGameAfter(IndexInRefGames+1)) then
      begin
        // Der Nachfolgetermin ist schon gekoppelt mit dem nchsten Nachfolger -> 3 Koppeltermine gibt es nicht
        Result := nil;
      end
      else
      begin
        Result := gamesRef[IndexInRefGames+1];
      end;
    end;
  end;
end;

function TMannschaft.getKoppeledGameBefore(IndexInRefGames: integer): TGame;
begin
  Result := nil;
  if IndexInRefGames > 0 then
  begin
    // Die Afterfunktion ist die fhrende Bestimmung der Koppeltermine
    if Assigned(getKoppeledGameAfter(IndexInRefGames-1)) then
    begin
      Result := gamesRef[IndexInRefGames-1];
    end;
  end;
end;


function TMannschaft.getKoppeledGame(IndexInRefGames: integer): TGame;
begin
  Result := getKoppeledGameAfter(IndexInRefGames);

  if not Assigned(Result) then
  begin
    if IndexInRefGames > 0 then
    begin
      if gamesRef[IndexInRefGames] = getKoppeledGameAfter(IndexInRefGames-1) then
      begin
        Result := gamesRef[IndexInRefGames - 1];
      end;
    end;
  end;
end;


function TMannschaft.getKoppeledGame(Game: TGame): TGame;
begin
  Result := getKoppeledGame(gamesRef.IndexOf(Game));
end;



function TMannschaft.IsKoppelTermin(game: TGame): TWunschterminOptions;
begin
  Result := [];
  if game.MannschaftHeim = Self then
  begin
    Result := game.DateOptions;
  end;
end;

procedure TMannschaft.CalculateKostenEngeTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  value: MyDouble;
  DiffDays: Integer;
  CachedValue: MyDouble;
const
  cErrorPower = 4.3;
begin
  Assert(AktType = mktEngeTermine);

  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  value := 0.0;
  for i:=0 to gamesRef.Count-2 do
  begin
    if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
    begin
      DiffDays := Trunc(gamesRef[i+1].Date) - Trunc(gamesRef[i].Date);
      if (DiffDays <= 3) and (DiffDays >= 0)  then
      begin
        if not (gamesRef[i+1] = getKoppeledGame(i)) then
        begin
          Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + 1;
          // Kein Tag Abstand muss per Power viel schlechter bewertet werden als 3 Tage
          value := value + Power((4.0 / (DiffDays + 1)), cErrorPower);
          if WithMessages <> tmNoMessage then
          begin
            Kosten.Messages.Add('Weniger als drei Tage Abstand: ' + MyFormatDate(gamesRef[i].Date) + ' - ' + MyFormatDate(gamesRef[i+1].Date));
          end;
        end;
      end;
    end;
  end;

  value := Power(value, 2) * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;


{$Hints Off}
procedure TMannschaft.CalculateKostenSpielverteilung(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  value: MyDouble;
  DiffDays: Integer;
  CachedValue: MyDouble;
  SollGameCount: MyDouble;
  CurrValue: MyDouble;
  StartDate: TDateTime;
  EndDate: TDateTime;
  IsFirstGame: boolean;
  IsLastGame: boolean;
  currRound: integer;
  Values: TList<MyDouble>;
  MittelWert: MyDouble;


const
  cErrorPower = 4.3;
begin
  Assert(AktType = mktSpielverteilung);

  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  value := 0.0;
  Values := TList<MyDouble>.Create;


  if gamesRef.Count > 0 then
  begin
    for currRound:=0 to Plan.Rounds.Count-1 do
    begin
      if Plan.IsRoundToGenerate(Plan.Rounds[currRound]) then
      begin
        for i:=-1 to gamesRef.Count-1 do
        begin
          IsFirstGame := false;
          IsLastGame := false;

          EndDate := 0;
          StartDate := 0;

          if (i < 0) or (currRound > Plan.GetRoundNumberByDate(gamesRef[i].Date)) then
          begin
            if (i+1 >= gamesRef.Count) or (currRound <> Plan.GetRoundNumberByDate(gamesRef[i+1].Date)) then
            begin
              // Ganz am Ende oder wir sind in einer ganz anderen Runde
              continue;
            end;

            IsFirstGame := true;
            StartDate := Plan.getFirstRealDateInRound(currRound);
            EndDate := gamesRef[i+1].Date;
          end
          else
          if (i >= gamesRef.Count-1) or (currRound < Plan.GetRoundNumberByDate(gamesRef[i+1].Date)) then
          begin
            if (currRound <> Plan.GetRoundNumberByDate(gamesRef[i].Date)) then
            begin
              // Ganz andere Runde
              continue;
            end;
            IsLastGame := true;
            EndDate := Plan.getLastRealDateInRound(currRound) + 1;
            StartDate := gamesRef[i].Date;
          end
          else
          begin
            StartDate := gamesRef[i].Date;
            EndDate := gamesRef[i+1].Date;
          end;


          if (EndDate > 0) and (StartDate > 0) then
          begin
            DiffDays := Trunc(EndDate) - Trunc(StartDate);
            SollGameCount := Plan.getNumSollGames(StartDate, EndDate);

            if (not IsLastGame) and Assigned(getKoppeledGameAfter(i+1)) then
            begin
              SollGameCount := SollGameCount / 1.5;
            end;

            if (not IsFirstGame) and Assigned(getKoppeledGameBefore(i)) then
            begin
              SollGameCount := SollGameCount / 1.5;
            end;

            if IsFirstGame or IsLastGame then
            begin
              SollGameCount := SollGameCount + 0.5;
            end;


            Values.Add(SollGameCount);
          end;
        end;
      end;
    end;
  end;

  if Values.Count > 0 then
  begin
    Value := CalculateStandardAbweichung(Values, MittelWert);
  end;

  value := Power(value * 5, 5) / 10.0 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  value := value / gamesRef.Count * 20;

  Kosten.Values[AktType].Anzahl := -1;
  Kosten.Values[AktType].GesamtKosten := value;

  setCachedKosten(AktType, Kosten.Values[AktType]);

  Values.free;

end;
{$Hints On}



procedure TMannschaft.getMessagesEngeTermine2Days(Plan: TPlan; Messages: TStrings);
var
  i: Integer;
  DiffDays: Integer;
begin
  for i:=0 to gamesRef.Count-2 do
  begin
    if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
    begin
      DiffDays := Trunc(gamesRef[i+1].Date) - Trunc(gamesRef[i].Date);
      if (DiffDays <= 2) and (DiffDays >= 0)  then
      begin
        if not (gamesRef[i+1] = getKoppeledGame(i)) then
        begin
          Messages.Add(MyFormatDate(gamesRef[i].Date) + ' - ' + MyFormatDate(gamesRef[i+1].Date));
        end;
      end;
    end;
  end;
end;



function TMannschaft.CreateListOfViolatedSperrTermine(Plan: TPlan): TList<TDateTime>;
var
  i: Integer;
  currDate: TDateTime;
  Dummy: Integer;
begin
  Result := nil;
  if SperrTermine.Count > 0 then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      currDate := Trunc(gamesRef[i].Date);
      if (Plan.IsInRoundToGenerate(currDate)) then
      begin
        if SperrTermine.BinarySearch(currDate, Dummy) then
        begin
          if Result = nil then
          begin
            Result := TList<TDateTime>.Create;
          end;
          Result.Add(currDate);
        end;
      end;
    end;
  end;
end;



procedure TMannschaft.CalculateKostenSameHomeDates(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  value: MyDouble;
  NumParallelGames: Integer;
  ParallelGamesString: String;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktForceSameHomeGames);

  Kosten.ClearValues(AktType);

  if (WithMessages = tmNoMessage) and (SisterTeamsInPlan.Count = 0) then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  if getCache(Plan).CachedTeamsForForceHomeGames.Count > 0 then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
      begin
        if gamesRef[i].MannschaftHeim = Self then
        begin
          GetMissingParallelHomeGames(gamesRef[i].Date, Plan, NumParallelGames, ParallelGamesString, WithMessages <> tmNoMessage);

          if NumParallelGames > 0 then
          begin
            Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + NumParallelGames;

            if WithMessages <> tmNoMessage then
            begin
              Kosten.Messages.Add('Kein gleichzeitiges Heimspiel: ' + MyFormatDate(gamesRef[i].Date) + ' ' + ParallelGamesString);
            end;
          end;
        end;
      end;
    end;

    value := Kosten.Values[AktType].Anzahl;
    value := Power(value, 3) * 10 * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := value;
  end;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;

procedure TMannschaft.CalculateKostenSperrTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  currDate: TDateTime;
  CachedValue: MyDouble;
  ViolatedDates: TList<TDateTime>;
begin
  Assert(AktType = mktSperrTermine);

  Kosten.ClearValues(AktType);

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  Kosten.Values[AktType].AllCount := SperrTermine.Count;

  ViolatedDates := CreateListOfViolatedSperrTermine(Plan);
  if ViolatedDates <> nil then
  begin
    if WithMessages <> tmNoMessage then
    begin
      for currDate in ViolatedDates do
      begin
        Kosten.Messages.Add('Sperrtermin wurde verletzt: ' + MyFormatDate(currDate));
      end;
    end;

    Kosten.Values[AktType].Anzahl := ViolatedDates.Count;



    value := (ViolatedDates.Count * 1.0) / (SperrTermine.Count) * 20.0;

    // Je hher die Anzahl der Spieltage sind, desto hher werden die Sperrtermine bewertet
    // Bei einer Liga, die unter der Woche spielt knnen mehr Sperrtermine eingehalten werden,
    // deshalb sind die Strafkosten bei Ligen, die nur am Samstag spielen geringer und damit werden die Sperrtermine eher
    // nicht bercksichtigt
    value := value * ((Plan.GetNumWunschDays * 1.0) / 50.0);

    value := Power(value, 3) * 200;


    value := value * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := value;

    ViolatedDates.Free;

  end;

  setCachedKosten(AktType, Kosten.Values[AktType]);
end;



procedure TMannschaft.CalculateKostenAusweichTermine(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  CachedValue: MyDouble;
  i: integer;
begin
  Assert(AktType = mktAusweichTermine);

  Kosten.ClearValues(AktType);

  if not Plan.HasAusweichTermine then
  begin
    Exit;
  end;

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;

  Kosten.Values[AktType].AllCount := AusweichTermine.Count;

  if Kosten.Values[AktType].AllCount > 0 then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if gamesRef[i].MannschaftHeim = Self then
      begin
        if (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
        begin
          if isAusWeichTermin(Trunc(gamesRef[i].Date)) then
          begin
            Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + 1;
            if WithMessages <> tmNoMessage then
            begin
              Kosten.Messages.Add('Ausweichtermin wurde verwendet: ' + MyFormatDate(Trunc(gamesRef[i].Date)));
            end;
          end;
        end;
      end;
    end;
  end;

  value := Kosten.Values[AktType].Anzahl * 1.0;
  value := Power(value, 2) * 10 * Plan.GetMannschaftKostenFaktor(Self, AktType);

  Kosten.Values[AktType].GesamtKosten := value;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;





procedure TMannschaft.CalculateKostenHallenBelegung(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  value: MyDouble;
  BelegungNum: integer;
  BelegungString: String;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktHalleBelegt);

  Kosten.ClearValues(AktType);

  if (WithMessages = tmNoMessage) and (SisterTeamsInPlan.Count = 0) then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;



  if (not sisterGames.IsEmpty()) or (SisterTeamsInPlan.Count > 0) then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
      begin
        if (gamesRef[i].DateMaxHome > 0) and (gamesRef[i].MannschaftHeim = Self) then
        begin
          GetHallenBelegung(gamesRef[i].Date, gamesRef[i].Location, Plan, BelegungNum, BelegungString, WithMessages <> tmNoMessage, False);

          if BelegungNum >= gamesRef[i].DateMaxHome then
          begin
            Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + BelegungNum - gamesRef[i].DateMaxHome + 1;
            if WithMessages <> tmNoMessage then
            begin
              Kosten.Messages.Add('Halle belegt von ' + BelegungString + ': ' + MyFormatDate(gamesRef[i].Date));
            end;

          end;
        end;
      end;
    end;

    value := Kosten.Values[AktType].Anzahl * 10.0;
    value := Power(value, 2) * 500 * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := value;

  end;

  if (SisterTeamsInPlan.Count = 0) then
  begin
    setCachedKosten(AktType, Kosten.Values[AktType]);
  end;

end;



procedure TMannschaft.CalculateKostenParallelGames(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  i: Integer;
  value: MyDouble;
  NumParallelGames: Integer;
  ParallelGamesString: String;
  CachedValue: MyDouble;
begin
  Assert(AktType = mktSisterGames);

  Kosten.ClearValues(AktType);
  value := 0;

  if (WithMessages = tmNoMessage) and (SisterTeamsInPlan.Count = 0) then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;


  if (not sisterGames.IsEmpty()) or (SisterTeamsInPlan.Count > 0) then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
      begin
        GetParallelGames(gamesRef[i].Date, Plan, NumParallelGames, ParallelGamesString, WithMessages <> tmNoMessage, False);

        if NumParallelGames > 0 then
        begin
          Kosten.Values[AktType].Anzahl := Kosten.Values[AktType].Anzahl + NumParallelGames;

          // Damit 3 Spiele am Tag hher bewertet werden
          value := value + Power(NumParallelGames, 3);

          if WithMessages <> tmNoMessage then
          begin
            Kosten.Messages.Add('Parallele Spiele: ' + MyFormatDate(gamesRef[i].Date) + ' ' + ParallelGamesString);
          end;

        end;
      end;
    end;

    value := value * 1.0;
    value := Power(value, 1.5) * 20.0 * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := value;

  end;

  if (SisterTeamsInPlan.Count = 0) then
  begin
    setCachedKosten(AktType, Kosten.Values[AktType]);
  end;

end;


function TMannschaft.CreateListOfViolated60KmRegel(Plan: TPlan): TList<TGame>;
var
  i: Integer;
begin
  Result := nil;
  if noWeekGame.Count > 0 then
  begin
    for i:=0 to gamesRef.Count-1 do
    begin
      if (gamesRef[i].DateValid) and (Plan.IsInRoundToGenerate(gamesRef[i].Date)) then
      begin
        if gamesRef[i].MannschaftHeim <> Self then
        begin
          if noWeekGame.Contains(gamesRef[i].MannschaftHeim.teamName) and (gamesRef[i].Date <> 0) then
          begin
            if not Plan.DateIsAllowedForWeekendGames(gamesRef[i].Date) then
            begin
              if Result = nil then
              begin
                Result := TList<TGame>.Create;
              end;
              Result.Add(gamesRef[i]);
            end;
          end;
        end;
      end;
    end;
  end;
end;




procedure TMannschaft.CalculateKosten60kMRegel(Plan: TPlan; AktType: TMannschaftsKostenType; Kosten: TMannschaftsKosten; WithMessages: TMessageType);
var
  value: MyDouble;
  CachedValue: MyDouble;
  ViolatedGames: TList<TGame>;
  Game: TGame;
begin
  Assert(AktType = mkt60Kilometer);

  Kosten.ClearValues(AktType);

  if not Plan.Has60KilometerValues then
  begin
    Exit;
  end;

  if WithMessages = tmNoMessage then
  begin
    CachedValue := getCachedKosten(AktType);
    if CachedValue >= 0 then
    begin
      Kosten.Values[aktType].GesamtKosten := CachedValue;
      Exit;
    end;
  end;


  Kosten.Values[AktType].AllCount := noWeekGame.Count;

  ViolatedGames := CreateListOfViolated60KmRegel(Plan);

  if ViolatedGames <> nil then
  begin
    Kosten.Values[AktType].Anzahl := ViolatedGames.Count;
    if WithMessages <> tmNoMessage then
    begin
      for Game in ViolatedGames do
      begin
        Kosten.Messages.Add('60km Regel nicht beachtet: ' + MyFormatDate(Game.Date) + ' beim ' + Game.MannschaftHeim.teamName);
      end;
    end;
    ViolatedGames.Free;

    value := Kosten.Values[AktType].Anzahl * 10.0;
    value := Power(value, 2) * Plan.GetMannschaftKostenFaktor(Self, AktType);

    Kosten.Values[AktType].GesamtKosten := value;

  end;

  setCachedKosten(AktType, Kosten.Values[AktType]);

end;



constructor TMannschaft.Create(Index: integer);
begin
  inherited Create;
  Self.Index := Index;
  SperrTermine := TDateList.Create;
  AusweichTermine := TDateList.Create;
  AuswaertsKoppel := TAuswaertsKoppelList.Create();
  HomeRightCountRoundOne := -1;
  HomeRights := THomeRightList.Create;
  WunschTermine := TWunschTerminList.Create;
  gamesRef := TGameList.Create(false);
  sisterGames := TMapSisterGames.Create;
  noWeekGame := TList<String>.Create;
  SisterTeamsInPlan := TMannschaftList.Create(false);
  DataCache := nil;
  ClearCache();
end;

function TMannschaft.CreateAuswaertsKoppeledGamesDependenciesList(Plan: TPlan; MannschaftHeim: TMannschaft): TGameList;
var
  Games: TGameList;
  Values: TList<String>;
  i: integer;
begin
  Result := nil;

  if getCache(Plan).CachedAuswaertsKoppelTeams.TeamIdToTeamId.Count > 0 then
  begin
    if getCache(Plan).CachedAuswaertsKoppelTeams.TeamIdToTeamId.TryGetValue(MannschaftHeim.teamId, Values) then
    begin
      for i := 0 to Values.Count-1 do
      begin
        Games := Plan.findGames(Values[i], teamId);
        if Assigned(Games) then
        begin
          if not Assigned(Result) then
          begin
            Result := TGameList.Create(false);
          end;
          Result.AddRange(Games);
        end;
      end;
    end;
  end;
end;

destructor TMannschaft.Destroy;
begin
  SperrTermine.Free;
  AusweichTermine.Free;
  AuswaertsKoppel.Free;
  HomeRights.Free;
  DataCache.Free;
  WunschTermine.Free;
  gamesRef.Free;
  sisterGames.Free;
  noWeekGame.Free;
  SisterTeamsInPlan.Free;
  inherited;
end;

function TMannschaft.getCachedKosten(AktType: TMannschaftsKostenType): MyDouble;
begin
  if KostenCache.Valid then
  begin
    Result := KostenCache.Values[AktType].GesamtKosten;
{$ifdef CACHE_HIT_TEST}
    if Result >= 0 then
    begin
      Inc(CacheHit);
    end
    else
    begin
      Inc(CacheMiss);
    end;
{$endif}

{$ifdef CACHE_TEST}
    Result := -1;
{$endif}
  end
  else
  begin
    Result := -1;
  end;
end;

function TMannschaft.getGameCount(FromDate, ToDate: TDateTime): integer;
var
  game: TGame;
begin
  Result := 0;
  for game in gamesRef do
  begin
    if game.DateValid then
    begin
      if (game.Date >= FromDate) and (game.Date <= ToDate) then
      begin
        Inc(Result);
      end;
    end;
  end;
end;

procedure TMannschaft.GetHallenBelegung(Date: TDateTime; Location: String; Plan: TPlan; var BelegungNum: integer; var BelegungString: String; FillBelegungString: boolean; IgnoreThisPlan: boolean);
var
  Liste: TSisterGames;
  i, j: integer;
  SisterMannschaft: TMannschaft;
  SisterGame: TGame;
begin
  BelegungNum := 0;
  BelegungString := '';
  Liste := sisterGames.get(Date);

  if Assigned(Liste) then
  begin
    for j:=0 to Liste.Count-1 do
    begin
      if Liste[j].IsHomeGame then
      begin
        if Abs(Liste[j].Date - Date) <= EncodeTime(1, 29, 0, 0) then
        begin
          if Location = Liste[j].Location then
          begin
            Inc(BelegungNum);
            if FillBelegungString then
            begin
              if BelegungString <> '' then
              begin
                BelegungString := BelegungString + ', ';
              end;
              BelegungString := BelegungString + Liste[j].HomeMannschaftName + ' (' + Liste[j].gender + ')';
            end;
          end;
        end;
      end;
    end;
  end;

  if not IgnoreThisPlan then
  begin
    for i := 0 to SisterTeamsInPlan.Count-1 do
    begin
      SisterMannschaft := SisterTeamsInPlan[i];
      for j:=0 to SisterMannschaft.gamesRef.Count-1 do
      begin
        SisterGame := SisterMannschaft.gamesRef[j];
        if (SisterGame.MannschaftHeim = SisterMannschaft) and (SisterGame.MannschaftGast <> Self) then
        begin
          // Heimspiel der anderen Mannschaft in der Liga, aber nicht gegen die eigene Mannschaft
          if Abs(SisterGame.Date - Date) <= EncodeTime(1, 29, 0, 0) then
          begin
            if SisterGame.Location = Location then
            begin
              Inc(BelegungNum);
              if FillBelegungString then
              begin
                if BelegungString <> '' then
                begin
                  BelegungString := BelegungString + ', ';
                end;
                BelegungString := BelegungString + SisterMannschaft.teamName + ' (' + Plan.gender + ')';
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;


procedure TMannschaft.GetMissingParallelHomeGames(Date: TDateTime; Plan: TPlan; var NumGames: integer; var MessageString: String; FillMessageString: boolean);
var
  j: Integer;
  Liste: TSisterGames;
  Index: integer;
  S, S2: String;
  TeamsForForceHomeGames: TStrings;
begin

  NumGames := 0;

  if FillMessageString then
  begin
    MessageString := '';
  end;


  TeamsForForceHomeGames := TStringList.Create;
  TeamsForForceHomeGames.Assign(getCache(Plan).CachedTeamsForForceHomeGames);
  Liste := sisterGames.get(Date);
  if Assigned(Liste) then
  begin
    for j:=0 to Liste.Count-1 do
    begin
      if Liste[j].IsHomeGame then
      begin
        if Liste[j].ForceParallelHomeGames then
        begin
          // Wirklich zur gleichen Zeit
          if Trunc(Liste[j].Date) = Trunc(Date) then
          begin
            Index := TeamsForForceHomeGames.IndexOf(Liste[j].gender + #10 + Liste[j].teamName);

            if Index >= 0 then
            begin
              TeamsForForceHomeGames.Delete(Index);
            end;
          end;
        end;
      end;
    end;
  end;

  NumGames := TeamsForForceHomeGames.Count;

  if (NumGames > 0) and FillMessageString then
  begin
    for S in TeamsForForceHomeGames do
    begin
      S2 := S.Replace(#10, ' ');
      if MessageString <> '' then
      begin
        MessageString := MessageString + ', ';
      end;
      MessageString := MessageString + S2;
    end;
  end;

  TeamsForForceHomeGames.Free;
end;




procedure TMannschaft.GetParallelGamesIntern(Date: TDateTime; Plan: TPlan; SearchTeamNum: integer; SearchRichtung: TParallelGamesSuchRichtung; var NumGames: integer; var MessageString: String; FillMessageString: boolean; IgnoreThisPlan: boolean);
var
  i, j: Integer;
  Liste: TSisterGames;
  SisterMannschaft: TMannschaft;
  SisterGame: TGame;
  Diff: integer;
  TestThisTeam: boolean;
begin

  if not sisterGames.IsEmpty() then
  begin
    Liste := sisterGames.get(Date);

    if Assigned(Liste) then
    begin
      for j:=0 to Liste.Count-1 do
      begin
        TestThisTeam := false;
        if (SearchRichtung = tpgBoth) then
        begin
          // Bei Both wird nur getestet, ob parallele Spiele mit der Nachbarnmannschaft vermieden werden sollen
          TestThisTeam := Liste[j].PreventParallelGame;
        end
        else
        begin
          // Hier geht es nur um weitere Mannschaften, also 2. 3. ob die auch ein Spiel haben
          // Das wird immer bercksichtigt, dafr gibt es keine Einstellung
          if Liste[j].gender = Plan.gender then
          begin
            // Ist eine echte Nachbarmannschaft
            Diff := SearchTeamNum - Liste[j].Number;
            if ((Diff = -1) and (SearchRichtung = tpgDown)) or
                ((Diff = 1) and (SearchRichtung = tpgTop))
            then
            begin
              TestThisTeam := True;
            end;
          end;
        end;

        if TestThisTeam then
        begin
          // Wirklich zur gleichen Zeit
          if Abs(Liste[j].Date - Date) <= EncodeTime(3, 59, 0, 0) then
          begin
            if SearchRichtung <> tpgBoth then
            begin
              SearchTeamNum := SearchTeamNum;
            end;

            if FillMessageString then
            begin
              if MessageString <> '' then
              begin
                MessageString := MessageString + ', ';
              end;
              MessageString := MessageString + Liste[j].HomeMannschaftName + ' - ' + Liste[j].GuestMannschaftName;
            end;
            Inc(NumGames);

            // Jetzt noch testen, ob eine weiter Mannschaft spielt, also wenn die II spielt, testen ob die III auch spielt.
            if SearchRichtung in [tpgBoth, tpgDown] then
            begin
              GetParallelGamesIntern(Date, Plan, Liste[j].Number, tpgDown, NumGames, MessageString, FillMessageString, IgnoreThisPlan);
            end;
            if SearchRichtung in [tpgBoth, tpgTop] then
            begin
              GetParallelGamesIntern(Date, Plan, Liste[j].Number, tpgTop, NumGames, MessageString, FillMessageString, IgnoreThisPlan);
            end;
          end;
        end;
      end;
    end;
  end;

  if not IgnoreThisPlan then
  begin
    for i := 0 to SisterTeamsInPlan.Count-1 do
    begin
      SisterMannschaft := SisterTeamsInPlan[i];
      for j:=0 to SisterMannschaft.gamesRef.Count-1 do
      begin
        SisterGame := SisterMannschaft.gamesRef[j];

        // Ich spiele mit
        if (SisterGame.MannschaftHeim <> Self) and (SisterGame.MannschaftGast <> Self) then
        begin

          // Ist eine echte NachbarMannschaft
          Diff := SearchTeamNum - SisterMannschaft.teamNumber;
          if ((Diff = -1) and (SearchRichtung in [tpgBoth, tpgDown])) or
              ((Diff = 1) and (SearchRichtung in [tpgBoth, tpgTop]))
          then
          begin
            // Wirklich zur gleichen Zeit
            if Abs(SisterGame.Date - Date) <= EncodeTime(3, 59, 0, 0) then
            begin
              Inc(NumGames);
              if FillMessageString then
              begin
                if MessageString <> '' then
                begin
                  MessageString := MessageString + ', ';
                end;
                MessageString := MessageString + SisterGame.MannschaftHeim.teamName + ' - ' + SisterGame.MannschaftGast.teamName;
              end;

              // Jetzt noch testen, ob eine weiter Mannschaft spielt, also wenn die II spielt, testen ob die III auch spielt.
              if SearchRichtung in [tpgBoth, tpgDown] then
              begin
                GetParallelGamesIntern(Date, Plan, SisterMannschaft.teamNumber, tpgDown, NumGames, MessageString, FillMessageString, IgnoreThisPlan);
              end;
              if SearchRichtung in [tpgBoth, tpgTop] then
              begin
                GetParallelGamesIntern(Date, Plan, SisterMannschaft.teamNumber, tpgTop, NumGames, MessageString, FillMessageString, IgnoreThisPlan);
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;



// 75.000 Plne pro Sekunde, alte Version
procedure TMannschaft.GetParallelGames(Date: TDateTime; Plan: TPlan; var NumGames: integer; var MessageString: String; FillMessageString: boolean; IgnoreThisPlan: boolean);
begin
  NumGames := 0;
  MessageString := '';

  GetParallelGamesIntern(Date, Plan, Self.teamNumber, tpgBoth, NumGames, MessageString, FillMessageString, IgnoreThisPlan);
end;


procedure TMannschaft.getTermineNotPossible(Plan: TPlan; MessageList: TList<String>);
var
  i: Integer;
  valid: boolean;
  OtherMannschaft, TeamA, TeamB: TMannschaft;
  j: Integer;
  WunschTermin, Wunsch1, Wunsch2: TWunschTermin;
  Koppel: TAuswaertsKoppel;
begin
  MessageList.Clear;

  // Teste die SperrTermine
  for i := 0 to Plan.Mannschaften.Count-1 do
  begin
    valid := false;
    OtherMannschaft := Plan.Mannschaften[i];
    if OtherMannschaft <> Self then
    begin
      for j := 0 to OtherMannschaft.WunschTermine.Count-1 do
      begin
        if not SperrTermine.Contains(Trunc(OtherMannschaft.WunschTermine[j].Date)) then
        begin
          valid := True;
        end;
      end;
      if (not Valid) and (OtherMannschaft.WunschTermine.Count > 0) then
      begin
        MessageList.Add('Alle Termine beim ' + OtherMannschaft.teamName + ' wurden gesperrt')
      end;
    end;
  end;


  // Teste die 60km Reglung
  for i := 0 to Plan.Mannschaften.Count-1 do
  begin
    valid := false;
    OtherMannschaft := Plan.Mannschaften[i];
    if OtherMannschaft <> Self then
    begin
      for j := 0 to OtherMannschaft.WunschTermine.Count-1 do
      begin
        if Plan.DateIsAllowedForWeekendGames(OtherMannschaft.WunschTermine[j].Date) then
        begin
          valid := True;
        end
        else
        begin
          if not noWeekGame.Contains(OtherMannschaft.teamName) then
          begin
            valid := True;
          end;
        end;
      end;
      if (not Valid) and (OtherMannschaft.WunschTermine.Count > 0) then
      begin
        MessageList.Add('60km Regel kann beim ' + OtherMannschaft.teamName + ' nicht eingehalten werden')
      end;
    end;
  end;

  // Teste die Koppeltermine
  if Plan.Mannschaften.Count mod 2 = 0 then
  begin
    Valid := False;
    for WunschTermin in WunschTermine do
    begin
      if not OptionIsKoppelSingleDate(WunschTermin.Options) then
      begin
        Valid := True;
        break;
      end;
    end;

    if (not Valid) and (WunschTermine.Count > 0) then
    begin
      MessageList.Add('Es wurden nur Koppeltermine angegeben, die Anzahl der Heimspiele ist ungerade weshalb mindestens ein Koppeltermin nicht eingehalten werden kann.')
    end;
  end;


  // Teste die Auswrtskoppeltermine
  for Koppel in AuswaertsKoppel do
  begin
    Valid := False;

    TeamA := Plan.Mannschaften.FindByName(Koppel.TeamA);
    TeamB := Plan.Mannschaften.FindByName(Koppel.TeamB);

    if Assigned(TeamA) and Assigned(TeamB) then
    begin
      for Wunsch1 in TeamA.WunschTermine do
      begin
        for Wunsch2 in TeamB.WunschTermine do
        begin
          if IsValidAuswaertsKoppelDate(Wunsch1.Date, Wunsch2.Date, TeamA.teamName, TeamB.teamName) then
          begin
            Valid := True;
          end;
        end;
      end;
      if (not Valid) then
      begin
        MessageList.Add('Fr den Auswrtskoppelwunsch bei ' + Koppel.TeamA + ' und ' + Koppel.TeamB + ' existieren keine geeigneten Wunschtermine.')
      end;
    end
    else
    begin
      if not Assigned(TeamA) then
      begin
        MessageList.Add('Datenfehler: Unbekanntes Team ' + Koppel.TeamA + ' in einem Auswrtskoppelwunsch.')
      end;
      if not Assigned(TeamB) then
      begin
        MessageList.Add('Datenfehler: Unbekanntes Team ' + Koppel.TeamB + ' in einem Auswrtskoppelwunsch.')
      end;
    end;
  end;
end;





function TMannschaft.IsTerminFree(Termin: TWunschtermin; Plan: TPlan; HeimMannschaft: TMannschaft; GastMannschaft: TMannschaft; IgnoreGame: TGame; AllSecondTimeGamesAreValid: boolean = false): Boolean;
var
  i: integer;
  tempDate, Date: TDateTime;
begin
  Result := True;
  Date := Termin.Date;

  if not AllSecondTimeGamesAreValid then
  begin
    // Die Zusatztermine sind normalerweise nicht zulssig, nur wenn an einem Tag 2 Koppelspiele sind
    // der Gastmannschaft ist das aber natrlich egal
    if (woKoppelSecondTime in Termin.Options) and (HeimMannschaft = self) then
    begin
      Result := False;
    end;

    // Die Zusatztermine sind normalerweise nicht zulssig, nur wenn an einem Tag 2 Koppelspiele sind
    // bei den der Gastmannschaft ist das aber natrlich egal
    if (woAuswaertsKoppelSecondTime in Termin.Options) and (GastMannschaft = self) then
    begin
      Result := False;
    end;
  end;


  for i:=0 to gamesRef.Count-1 do
  begin
    if gamesRef.List[i] <> IgnoreGame then
    begin
      tempDate := gamesRef.List[i].FDate;
      if (Trunc(tempDate) = Trunc(Date)) then
      begin
        if (not Plan.HasAuswaertsKoppelTermine) and (not Plan.HasKoppelTermine) then
        begin
          // Keine Koppeltermine im Plan -> Dann sind auch keine Termine an einem Tag erlaubt
          Result := False;
          break;
        end;

        // Eigentlich zu eng beisammen, aber Teste noch, ob es ein Koppeltermin sein knnte
        if tempDate > 0 then
        begin
          if MinGameDistance > Abs(tempDate - Date) then
          begin
            // Auch als Koppeltermin zu eng
            Result := False;
            break;
          end;

          if (Plan.HasAuswaertsKoppelTermine) then
          begin
            // Teste Auswrtskoppel

            if (GastMannschaft = self) and (gamesRef.List[i].MannschaftGast = Self) then
            begin
              if IsValidAuswaertsKoppelDate(tempDate, Date, HeimMannschaft.teamName, gamesRef.List[i].MannschaftHeim.teamName) then
              begin
                // Das passt, ist ein Auswrtskoppeltermin
                // Explizit auf True setzten, damit die SecondTime-Termine gltig werden
                Result := True;
                continue;
              end;
            end;
          end;

          if (self <> HeimMannschaft)  then
          begin
            // Koppeltermine gibt es nur bei der Heimmannschaft hier wird der Test fr self als Gastmannschaft aufgerufen -> kein Koppeltermin
            Result := False;
            Break;
          end;

          if not OptionIsKoppel(Termin.Options) then
          begin
            Result := False;
            Break;
          end;

          if not OptionIsKoppel(IsKoppelTermin(gamesRef[i])) then
          begin
            Result := False;
            Break;
          end;

          // Explizit auf True setzten, damit die SecondTime-Termine gltig werden
          // wenn er hier reinkommt ist es ein gltiger Koppeltermin
          Result := True;

        end;
      end;
    end;
  end;
end;


function TMannschaft.IsValidAuswaertsKoppelDate(Date1, Date2: TDateTime; teamA, teamB: String): boolean;
var
  Koppel: TAuswaertsKoppel;
  DateValid: boolean;
begin
  Result := False;

  if 1 < Abs(Trunc(Date1) - Trunc(Date2)) then
  begin
    // Zu weit auseinander, kann kein Koppeltermin sein
    exit;
  end;


  for Koppel in AuswaertsKoppel do
  begin
    if Koppel.Option = ktNone then
    begin
      Continue;
    end;

    DateValid := False;

    if Koppel.Option = ktSameDay then
    begin
      if Trunc(Date1) = Trunc(Date2) then
      begin
        // Zur gleichen Zeit darf das Spiel aber auch nicht stattfinden!
        if MinGameDistance <= Abs(Date1 - Date2) then
        begin
          DateValid := True;
        end;
      end;
    end
    else
    if Koppel.Option = ktDiffDay then
    begin
      if 1 = Abs(Trunc(Date1) - Trunc(Date2)) then
      begin
        DateValid := True;
      end;
    end
    else
    begin
      if 1 >= Abs(Trunc(Date1) - Trunc(Date2)) then
      begin
        // Zur gleichen Zeit darf das Spiel aber auch nicht stattfinden!
        if MinGameDistance <= Abs(Date1 - Date2) then
        begin
          DateValid := True;
        end;
      end;
    end;

    if DateValid then
    begin
      if (teamA = Koppel.TeamA)
        and (teamB = Koppel.TeamB) then
      begin
        Result := True;
      end;

      if (teamA = Koppel.TeamB)
        and (teamB = Koppel.TeamA) then
      begin
        Result := True;
      end;
    end;
  end;
end;


procedure TMannschaft.load(Data: TDataObject; Plan: TPlan; PlanData: TPlanData);
begin
  clubID := Data.GetAsString(aClubID);
  teamId := Data.GetAsString(aTeamId);
  teamNumber := Data.GetAsInt(aTeamNumber);
  teamName := Data.GetAsString(aTeamName);
  DefaultLocation := PlanData.FixLocation(Data.GetAsString(aLocation));
  HomeRightCountRoundOne := Data.GetAsInt(aHomeRights);

  loadGameDays(Plan, Data, PlanData);

  loadSisterGameDays(Plan, Data, PlanData);

end;

procedure TMannschaft.loadGameDays(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
var
  Node: TDataObject;
begin
  for Node in Data.NamedChilds(nHomeGameDay) do
  begin
    loadHomeGame(Plan, Data, Node, PlanData);
  end;

  for Node in Data.NamedChilds(nNoGameDay) do
  begin
    loadNoGame(Plan, Node);
  end;

  for Node in Data.NamedChilds(nRoadCouple) do
  begin
    loadAuswaertsKoppel(Plan, Node);
  end;

  for Node in Data.NamedChilds(nHomeRights) do
  begin
    loadHomeRights(Plan, Node);
  end;


  for Node in Data.NamedChilds(nNoWeekGames) do
  begin
    loadNoWeekGame(Plan, Node);
  end;

  WunschTermine.Sort();
  AusweichTermine.Sort();
  SperrTermine.Sort();

end;


procedure TMannschaft.loadSisterGameDays(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
var
  SisterNode: TDataObject;
begin
  for SisterNode in Data.NamedChilds(nSisterTeam) do
  begin
    loadSisterTeam(Plan, SisterNode, PlanData);
  end;
end;

procedure TMannschaft.loadSisterTeam(Plan: TPlan; Data: TDataObject; PlanData: TPlanData);
var
  gender: string;
  number: Integer;
  gameNode: TDataObject;
  homeTeam, guestTeam: String;
  sisterGame: TSisterGame;
  name: string;
  DefaultLocation: String;
begin
  gender := Data.GetAsString(aGender);
  number := Data.GetAsInt(aTeamNumber);
  name := Data.GetAsString(aTeamName);
  DefaultLocation := PlanData.FixLocation(Data.GetAsString(aLocation));


  for gameNode in Data.NamedChilds(nSisterGame) do
  begin
    homeTeam := gameNode.GetAsString(aHomeTeamName);
    guestTeam := gameNode.GetAsString(aGuestTeamName);

    sisterGame := TSisterGame.Create;
    sisterGame.Date := gameNode.GetAsDateTime(aDateTime);
    sisterGame.Number := number;
    sisterGame.teamName := name;
    sisterGame.gender := gender;
    sisterGame.IsHomeGame := (Name = homeTeam);
    sisterGame.HomeMannschaftName := homeTeam;
    sisterGame.GuestMannschaftName := guestTeam;
    if sisterGame.IsHomeGame then
    begin
      sisterGame.Location := DefaultLocation;
    end;
    sisterGame.PreventParallelGame := Data.GetAsBool(aNoParallelGames);
    sisterGame.ForceParallelHomeGames := Data.GetAsBool(aParallelHomeGames);

    if PlanData.FixLocation(gameNode.GetAsString(aLocation)) <> '' then
    begin
      sisterGame.Location := PlanData.FixLocation(gameNode.GetAsString(aLocation));
    end;

    sisterGames.Add(sisterGame);
  end;
end;


procedure TMannschaft.setCachedKosten(AktType: TMannschaftsKostenType; Kosten: TKostenValue);
begin
{$ifdef CACHE_TEST}
  if KostenCache.Valid and (KostenCache.Values[AktType].GesamtKosten >= 0) then
  begin
    // normalerweise htte der Cache hier zuschalgen sollen, wenn nicht CACHE_TEST gesetzt wre
    // also muss in den Kosten das gleiche drinstehen
    if Kosten.GesamtKosten <> KostenCache.Values[AktType].GesamtKosten then
    begin
      raise Exception.Create('Fehler im Cache');
    end;
  end;
{$endif}
  // Sobald der erste Wert gesetzt wird, dann ist der Cache valid
  KostenCache.Valid := True;
  KostenCache.Values[AktType] := Kosten;
end;

function TMannschaft.IsMannschaftInAuswaertsKoppelWunschAtOneDay(Plan: TPlan; MannschaftHeim: TMannschaft): boolean;
begin
  Result := getCache(Plan).CachedAuswaertsKoppelTeams.TeamsOnDay.Contains(MannschaftHeim.teamName);
end;

procedure TMannschaft.loadHomeGame(Plan: TPlan; TeamNode: TDataObject; Data: TDataObject; PlanData: TPlanData);
var
  DateTime: TDateTime;
  NewElem: TWunschTermin;
  KoppelType: TWunschterminOptions;
begin
  DateTime := Data.GetAsDateTime(aDateTime);

  NewElem := TWunschTermin.Create;
  WunschTermine.Add(NewElem);

  NewElem.Date := DateTime;

  NewElem.ParallelGames := Data.GetAsInt(aParallelGames);

  NewElem.Location := PlanData.FixLocation(TeamNode.GetAsString(aLocation));

  if PlanData.FixLocation(Data.GetAsString(aLocation)) <> '' then
  begin
    NewElem.Location := PlanData.FixLocation(Data.GetAsString(aLocation));
  end;

  KoppelType := [];
  if Data.GetAsBool(aCoupleGameDay) then
  begin
    case Data.GetAsInt(aCouplePrio) of
      1:
        KoppelType := [woKoppelSoft];
      2:
        KoppelType := [woKoppelHard];
      else
        KoppelType := [woKoppelPossible];
    end;

    // Zweiter Termin bei Koppelterminen
    NewElem.KoppelSecondTime := FixDateTime(Trunc(DateTime) + Data.GetAsTime(aCoupleSecondTime));
  end
  else
  if Data.GetAsBool(aDoubleGameDay) then
  begin
    case Data.GetAsInt(aCouplePrio) of
      1:
        KoppelType := [woKoppelDoubleDateSoft];
      2:
        KoppelType := [woKoppelDoubleDateHard];
      else
        KoppelType := [woKoppelDoubleDatePossible];
    end;
  end;

  NewElem.Options := KoppelType;

  if Data.GetAsBool(aAusweichTermin) then
  begin
    AusweichTermine.Add(Trunc(NewElem.Date));
  end;

  if (not OptionIsKoppelSingleDate(NewElem.Options)) and Data.GetAsBool(aCoupleAuswaertsSecondTime) then
  begin
    NewElem.Options := NewElem.Options + [woAuswaertsKoppelHasSecondTime];
    NewElem.KoppelSecondTime := FixDateTime(Trunc(DateTime) + Data.GetAsTime(aCoupleSecondTime));
  end

end;


procedure TMannschaft.loadAuswaertsKoppel(Plan: TPlan; Data: TDataObject);
var
  NewValue: TAuswaertsKoppel;
  SameDay: integer;
begin
  NewValue := TAuswaertsKoppel.Create;
  NewValue.TeamA := Data.GetAsString(aTeamNameA);
  NewValue.TeamB := Data.GetAsString(aTeamNameB);
  SameDay := Data.GetAsInt(aSameDay);

  case SameDay of
    0: NewValue.Option := ktSameDay;
    1: NewValue.Option := ktDiffDay;
    else
      NewValue.Option := ktAllDay;
  end;

  AuswaertsKoppel.Add(NewValue);
end;


procedure TMannschaft.loadNoGame(Plan: TPlan; Data: TDataObject);
var
  Date: TDateTime;
begin
  Date := Data.GetAsDate(aDate);
  SperrTermine.Add(Date);
  SperrTermine.Sort;
end;


procedure TMannschaft.loadNoWeekGame(Plan: TPlan; Data: TDataObject);
var
  team: String;
begin
  team := Data.GetAsString(aTeamName);
  noWeekGame.Add(team);
end;


procedure TMannschaft.loadHomeRights(Plan: TPlan; Data: TDataObject);
var
  team: String;
  value: Integer;
  Elem: THomeRight;
  HomeRight: THomeRightValue;
begin
  team := Data.GetAsString(aTeamName);
  value := Data.GetAsInt(aHomeRight);

  HomeRight := hrNone;

  if value = 1 then
  begin
    HomeRight := hrRound1;
  end;
  if value = 2 then
  begin
    HomeRight := hrRound2;
  end;

  if HomeRight <> hrNone then
  begin
    Elem := THomeRight.Create;
    Elem.Team := team;
    Elem.Option := HomeRight;
    HomeRights.Add(Elem);
  end;
end;

function TMannschaft.IsAuswaertsKoppeledGames(game1, game2: TGame): boolean;
begin
  Result := IsValidAuswaertsKoppelDate(game1.Date, game2.Date, game1.MannschaftHeim.teamName, game2.MannschaftHeim.teamName);
end;

function TMannschaft.isAusWeichTermin(Date: TDateTime): boolean;
var
  Dummy: Integer;
begin
  Result := AusweichTermine.BinarySearch(Trunc(Date), Dummy);
end;

function TMannschaft.InternIsKoppeledGames(game1, game2: TGame): Boolean;
begin
  Result := False;

  if 1 < Abs(Trunc(game1.Date) - Trunc(game2.Date)) then
  begin
    // Knnen wir gleich raus -> Performance
    Exit;
  end;

  // Teste Heimkoppelspiele

  if (game1.MannschaftHeim = Self) and (game2.MannschaftHeim = Self) then
  begin
    if 1 >= Abs(Trunc(game1.Date) - Trunc(game2.Date)) then
    begin
      if (OptionIsKoppel(IsKoppelTermin(game1))) and (OptionIsKoppel(IsKoppelTermin(game2))) then
      begin
        Result := True;
      end;
    end;
  end;

  // Teste Auswrtskoppelspiele
  if (game1.MannschaftGast = Self) and (game2.MannschaftGast = Self) then
  begin
    if game1.MannschaftGast.IsAuswaertsKoppeledGames(game1, game2) then
    begin
      Result := True;
    end;
  end;
end;


{ TMannschaftList }

procedure TMannschaftList.Assign(Source: TMannschaftList; Plan: TPlan);
var
  i: integer;
  NewElem: TMannschaft;
begin
  Clear;
  for i:=0 to Source.Count-1 do
  begin
    NewElem := TMannschaft.Create(Count);
    NewElem.Assign(Source[i]);
    Add(NewElem);
  end;

  for i:=0 to Count-1 do
  begin
    Items[i].ClearKostenCache;
  end;
end;


type
  TMannschaftComparer = class(TComparer<TMannschaft>)
    function Compare(const Left, Right: TMannschaft): Integer; override;
  end;

function TMannschaftComparer.Compare(const Left, Right: TMannschaft): Integer;
begin
  Result := compareStr(Left.teamName, right.teamName);
end;


constructor TMannschaftList.Create(OwnsObjects: boolean);
begin
  inherited Create(TMannschaftComparer.Create, OwnsObjects);
end;

function TMannschaftList.FindByName(const Name: String): TMannschaft;
var
  Mannschaft: TMannschaft;
begin
  Result := nil;

  for Mannschaft in Self do
  begin
    if Mannschaft.teamName = Name then
    begin
      Result := Mannschaft;
      break;
    end;
  end;
end;


function TMannschaftList.FindByTeamId(const TeamId: String): TMannschaft;
var
  Mannschaft: TMannschaft;
begin
  Result := nil;

  for Mannschaft in Self do
  begin
    if Mannschaft.teamId = TeamId then
    begin
      Result := Mannschaft;
      break;
    end;
  end;
end;


procedure TMannschaftList.Sort;
var
  i: integer;
begin
  inherited Sort();

  // Index wieder richtig setzen
  for i := 0 to Count-1 do
  begin
    Self[i].Index := i;
  end;

end;

{ TGame }


constructor TGame.Create(MannschaftHeim, MannschaftGast: TMannschaft);
begin
  inherited Create;
  FDate := 0;
  FDateOptions := [];
  FDateMaxHome := 0;
  FFixedDate := False;
  FDateRef := 0;
  FDateRefOptions := [];
  FDateRefMaxHome := 0;
  FWunschTermin := nil;
  FWunschTerminRef := nil;
  FNotNecessaryRef := false;


  Self.MannschaftHeim := MannschaftHeim;
  Self.MannschaftGast := MannschaftGast;

end;

function TGame.DateValid: Boolean;
begin
  Result := Date <> 0.0;
end;

function TGame.AsString: String;
begin
  Result := '';
  if DateValid then
  begin
    Result := MyFormatDateTime(Date) + ': ';
  end;
  Result := Result + MannschaftHeim.teamName + ' - ' + MannschaftGast.teamName;
end;

procedure TGame.ClearAllValues;
begin
  FDate := 0.0;
  FDateOptions := [];
  FDateMaxHome := 0;
  FFixedDate := False;
  FDateRef := 0.0;
  FDateRefOptions := [];
  FDateRefMaxHome := 0;
  FWunschTerminRef := nil;
  FWunschTermin := nil;
  FNotNecessary := false;
  FNotNecessaryRef := false;
  OverrideLocation := '';
end;


destructor TGame.Destroy;
begin

  inherited;
end;

function TGame.getLocation: String;
begin
  Result := '';

  if Assigned(FWunschTermin) then
  begin
    Result := FWunschTermin.Location;
  end;

  if OverrideLocation <> '' then
  begin
    Result := OverrideLocation;
  end;
end;

function TGame.isTheSame(game: TGame): boolean;
begin
  if (FDate = game.FDate)
      and (game.MannschaftHeim.teamId = MannschaftHeim.teamId)
      and (game.MannschaftGast.teamId = MannschaftGast.teamId)
  then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;
end;

procedure TGame.setEmptyDate();
begin
  setDate(0.0, [], 0, False, False, nil);
end;

procedure TGame.setDate(Date: TDateTime; IsKoppel: TWunschterminOptions; MaxHome: Integer; fixedDate: boolean; NotNecessary: boolean; WunschTermin: TWunschTermin);
begin
  MannschaftHeim.ClearKostenCache();
  MannschaftGast.ClearKostenCache();
  FDate := Date;
  FDateOptions := IsKoppel;
  FDateMaxHome := MaxHome;
  FfixedDate := fixedDate;
  FNotNecessary := NotNecessary;
  if Assigned(FWunschTermin) then
  begin
    FWunschTermin.assignedGame := nil;
  end;
  FWunschTermin := WunschTermin;
  if Assigned(FWunschTermin) then
  begin
    FWunschTermin.assignedGame := Self;
  end;

end;

{ TGameList }


function TGameComparer.Compare(const game1, game2: TGame): Integer;
begin

  if game1.Date > game2.Date then
  begin
    Result := 1;
    Exit;
  end;

  if game1.Date < game2.Date then
  begin
    Result := -1;
    Exit;
  end;

  Result := compareStr(game1.MannschaftHeim.teamName + ' ' + game1.MannschaftGast.teamName, game2.MannschaftHeim.teamName + ' ' + game2.MannschaftGast.teamName);
end;



constructor TGameList.Create(OwnsObjects: boolean);
begin
  inherited Create(OwnsObjects);
  gameComparer := TGameComparer.Create;
end;


destructor TGameList.Destroy;
begin
  gameComparer.Free;
  inherited;
end;

function TGameList.findSameGame(game: TGame): TGame;
var
  GameTest: TGame;
begin
  Result := nil;
  for GameTest in Self do
  begin
    if GameTest.isTheSame(game) then
    begin
      Result := GameTest;
      break;
    end;
  end;
end;


// Intervallschachtelung auf die sortierten Spiele um das Spiel zu finden, das spter oder gleich dem Datum ist
function TGameList.getFirstGameRefIndexByDate(Date: TDateTime): integer;
var
  L, H, I, C: Integer;
begin
  L := 0;
  H := Count - 1;
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := CompareValue(Self[I].Date, Date);
    if C < 0 then L := I + 1 else
    begin
      H := I - 1;
      if C = 0 then
      begin
        L := I;
        break;
      end;
    end;
  end;
  Result := L;
end;


procedure TGameList.GetRefListe(Result: TGameList);
var
  i: Integer;
begin
  Result.Clear;

  for i:=0 to Count - 1 do
  begin
    Result.Add(Items[i]);
  end;
end;


function compGameListByDate(const game1: TGame; const game2: TGame): Integer;
begin

  if game1.Date > game2.Date then
  begin
    Result := 1;
    Exit;
  end;

  if game1.Date < game2.Date then
  begin
    Result := -1;
    Exit;
  end;

  Result := compareStr(game1.MannschaftHeim.teamName + ' ' + game1.MannschaftGast.teamName, game2.MannschaftHeim.teamName + ' ' + game2.MannschaftGast.teamName);

end;

procedure TGameList.SortByDate();
var
  i: Integer;
  game1: TGame;
  game2: TGame;
begin

  // Einfacher Bubblesort, da die Liste in der Regel eh schon sortiert ist

  i := 0;
  while(i<Count-1) do
  begin
    game1 := Items[i];
    game2 := Items[i+1];
    if 0 < compGameListByDate(game1, game2) then
    begin
      Exchange(i, i+1);
      i := max(0, i-1);
    end
    else
    begin
      Inc(i);
    end;
  end;
end;


(*
procedure TGameList.SortByDate();
begin
  Sort(GameComparer);
end;
*)




{ TMannschaftsKosten }

procedure TMannschaftsKosten.ClearValues(KostenType: TMannschaftsKostenType);
begin
  Values[KostenType].Anzahl := 0;
  Values[KostenType].AllCount := -1;
  Values[KostenType].GesamtKosten := 0;
end;

constructor TMannschaftsKosten.Create;
begin
  inherited Create;
  FMessages := nil;
end;

destructor TMannschaftsKosten.Destroy;
begin
  FreeAndNil(FMessages);
  inherited;
end;

function TMannschaftsKosten.getGesamtKosten: MyDouble;
var
  AktType: TMannschaftsKostenType;
begin
  Result := 0.0;

  for AktType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    Result := Result + values[AktType].GesamtKosten;
  end;
end;



function TMannschaftsKosten.GetMessages: TStrings;
begin
  if not Assigned(FMessages) then
  begin
    FMessages := TStringList.Create;
  end;
  Result := FMessages;
end;

{ TSisterGame }

procedure TSisterGame.Assign(Source: TSisterGame);
begin
  Date := Source.Date;
  gender := Source.gender;
  Number := Source.Number;
  teamName := Source.teamName;
  IsHomeGame := Source.IsHomeGame;
  HomeMannschaftName := Source.HomeMannschaftName;
  GuestMannschaftName := Source.GuestMannschaftName;
  Location := Source.Location;
  PreventParallelGame := Source.PreventParallelGame;
  ForceParallelHomeGames := Source.ForceParallelHomeGames;
end;

{ TMapSisterGames }

constructor TMapSisterGames.Create;
begin
  inherited;
  values := TObjectDictionary<Integer, TSisterGames>.Create([doOwnsValues]);
end;

destructor TMapSisterGames.Destroy;
begin
  values.Free;
  inherited;
end;

function TMapSisterGames.get(Date: TDateTime): TSisterGames;
var
  Key: Integer;
  Valid: boolean;
begin
  Key := Trunc(Date);

  Valid := values.TryGetValue(Key, Result);

  if not Valid then
  begin
    Result := nil;
  end;
end;

procedure TMapSisterGames.Add(sisterGame: TSisterGame);
var
  Key: Integer;
  sisterGames: TSisterGames;
begin
  Key := Trunc(sisterGame.Date);

  if values.ContainsKey(Key) then
  begin
    sisterGames := values.Items[Key];
  end
  else
  begin
    sisterGames := nil;
  end;

  if not Assigned(sisterGames) then
  begin
    sisterGames := TSisterGames.Create;
    values.Add(Key, sisterGames);
  end;

  sisterGames.Add(sisterGame);
end;

procedure TMapSisterGames.Assign(Source: TMapSisterGames);
var
  i, j: Integer;
  sisterGames: TSisterGames;
  newSisterGame: TSisterGame;
  MapValues: TArray<TSisterGames>;
begin
  values.Clear;

  MapValues := Source.values.Values.ToArray;

  for i:=Low(MapValues) to High(MapValues) do
  begin
    sisterGames := MapValues[i];

    for j:=0 to sisterGames.Count-1 do
    begin
      newSisterGame := TSisterGame.Create;
      newSisterGame.Assign(sisterGames[j]);
      Add(newSisterGame);
    end;
  end;
end;


function TMapSisterGames.IsEmpty: Boolean;
begin
  Result := values.Count = 0;
end;

function TMapSisterGames.toArray: TArray<TSisterGames>;
begin
  Result := values.Values.ToArray();
end;


function TMapSisterGames.toSortedArray: TArray<TSisterGames>;
var
  Comparer: TSisterGamesComparer;
begin
  Result := values.Values.ToArray();

  Comparer := TSisterGamesComparer.Create;

  TArray.Sort<TSisterGames>(Result, Comparer);

  Comparer.Free;
end;


{ TCalculateOptions }

procedure TCalculateOptions.Clear;
var
  i: TMannschaftsKostenType;
begin
  FreitagIsAllowedBy60km := True;
  RoundPlaning := rpBoth;
  DoubleRound := False;
  AutomaticMidDate := True;
  MidDate1 := 0;
  MidDate2 := 0;

  SpieltagGewichtung := coNormal;
  SpieltagOverlapp := coNormal;
  LastSpieltagGewichtung := coNormal;
  LastSpieltagOverlapp := coNormal;
  SisterTeamsAmAnfangGewichtung := coNormal;

  for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    GewichtungMainMannschaftsKosten[i] := coNormal;
  end;

  GewichtungMannschaften.Clear;

end;

constructor TCalculateOptions.Create;
begin
  inherited;
  GewichtungMannschaften := TGewichtungMannschaften.Create;
  Clear();
end;


destructor TCalculateOptions.Destroy;
begin
  GewichtungMannschaften.Free;
  inherited;
end;

function TCalculateOptions.getMidDate1(Plan: TPlan): TDateTime;
begin
  if not AutomaticMidDate then
  begin
    Result := MidDate1;
  end
  else
  begin
    Result := getAutomaticMidDate1(Plan);
  end;
end;

function TCalculateOptions.getMidDate2(Plan: TPlan): TDateTime;
begin
  if not AutomaticMidDate then
  begin
    Result := MidDate2;
  end
  else
  begin
    Result := getAutomaticMidDate2(Plan);
  end;
end;

function TCalculateOptions.getOptionsValueFromAttribute(Plan: TPlan; node: IXMLNode; attName: String): TCalculateOptionsValues;
var
  value: String;
  valueInt: Integer;
begin
  Result := coNormal;
  value := Plan.getAttributValue(node, attName, '');
  valueInt := GetEnumValue(System.TypeInfo(TCalculateOptionsValues), value);
  if valueInt >= 0 then
  begin
    Result := TCalculateOptionsValues(valueInt);
  end;
end;


procedure TCalculateOptions.Load(Plan: TPlan; FileName: String);
var
  document: IXMLDocument;
  root: IXMLNode;
  subNode: IXMLNode;
  subNode2: IXMLNode;
  KostenType: TMannschaftsKostenType;
  value: String;
  i, valueInt: integer;
begin
  Clear();
  if FileExists(FileName) then
  begin
    try
      document := LoadXMLDocument(FileName);

      root := document.DocumentElement;

      subNode := Plan.findChildNode(root, 'weighting');

      if Assigned(subNode) then
      begin
        FreitagIsAllowedBy60km := 'true' = Plan.getAttributValue(subNode, 'friday-is-part-of-weekend', '');
        SpieltagGewichtung := getOptionsValueFromAttribute(Plan, subNode, 'gameday');
        SpieltagOverlapp := getOptionsValueFromAttribute(Plan, subNode, 'gameday-overlapp');
        LastSpieltagGewichtung := getOptionsValueFromAttribute(Plan, subNode, 'last-gameday');
        LastSpieltagOverlapp := getOptionsValueFromAttribute(Plan, subNode, 'last-gameday-overlapp');
        SisterTeamsAmAnfangGewichtung := getOptionsValueFromAttribute(Plan, subNode, 'sister-game-at-start');
        for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
        begin
          GewichtungMainMannschaftsKosten[KostenType] := getOptionsValueFromAttribute(Plan, subNode, GetEnumName(System.TypeInfo(TMannschaftsKostenType), Ord(KostenType)));
        end;

        for i:=0 to subNode.childNodes.Count-1 do
        begin
          subNode2 := subNode.childNodes[i];
          if subNode2.NodeName = 'team' then
          begin
            value := Plan.getAttributValue(subNode2, 'teamid', '');
            if value <> '' then
            begin
              GewichtungMannschaften.addMain(value, getOptionsValueFromAttribute(Plan, subNode2, 'main'));
              for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
              begin
                GewichtungMannschaften.addDetail(value, KostenType, getOptionsValueFromAttribute(Plan, subNode2, GetEnumName(System.TypeInfo(TMannschaftsKostenType), Ord(KostenType))));
              end;
            end;
          end;
        end;

      end;

      subNode := Plan.findChildNode(root, 'round-planing');

      if Assigned(subNode) then
      begin
        RoundPlaning := rpBoth;
        value := Plan.getAttributValue(subnode, 'planing', '');
        valueInt := GetEnumValue(System.TypeInfo(TRoundPlaning), value);
        if valueInt >= 0 then
        begin
          RoundPlaning := TRoundPlaning(valueInt);
        end;
      end;


      subNode := Plan.findChildNode(root, 'double-round');

      if Assigned(subNode) then
      begin
        DoubleRound := 'true' = Plan.getAttributValue(subNode, 'active', '');
        AutomaticMidDate := 'true' = Plan.getAttributValue(subNode, 'automatic-mid-date', '');

        value := Plan.getAttributValue(subNode, 'mid-date-1', '');
        if value <> '' then
          MidDate1 := DateFromString(value);

        value := Plan.getAttributValue(subNode, 'mid-date-2', '');
        if value <> '' then
          MidDate2 := DateFromString(value);

      end;
    except
      Clear;
    end;
  end;
end;






procedure TCalculateOptions.Save(Plan: TPlan; FileName: String);
var
  XMLDoc: IXMLDocument;
  root: IXMLNode;
  subNode: IXMLNode;
  subNode2: IXMLNode;
  KostenType: TMannschaftsKostenType;
  teamId: String;
begin
  XMLDoc := TXMLDocument.Create(nil);
  XMLDoc.Active := True;
  XmlDoc.Version:='1.0';
  XMLDoc.Encoding := 'iso-8859-15';

  root := XMLDoc.AddChild('andigenerator-options');

  subNode := root.AddChild('weighting');

  if FreitagIsAllowedBy60km then
  begin
    Plan.setAttributValue(subNode, 'friday-is-part-of-weekend', 'true');
  end
  else
  begin
    Plan.setAttributValue(subNode, 'friday-is-part-of-weekend', 'false');
  end;

  Plan.setAttributValue(subNode, 'gameday', GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(SpieltagGewichtung)));
  Plan.setAttributValue(subNode, 'gameday-overlapp', GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(SpieltagOverlapp)));
  Plan.setAttributValue(subNode, 'last-gameday', GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(LastSpieltagGewichtung)));
  Plan.setAttributValue(subNode, 'last-gameday-overlapp', GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(LastSpieltagOverlapp)));
  Plan.setAttributValue(subNode, 'sister-game-at-start', GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(SisterTeamsAmAnfangGewichtung)));

  for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    Plan.setAttributValue(subNode,
                      GetEnumName(System.TypeInfo(TMannschaftsKostenType), Ord(KostenType)),
                      GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(GewichtungMainMannschaftsKosten[KostenType])));
  end;

  for teamId in GewichtungMannschaften.TeamIds do
  begin
    subNode2 := subnode.AddChild('team');
    Plan.setAttributValue(subNode2, 'teamid', teamId);

    Plan.setAttributValue(subNode2,
                      'main',
                      GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(GewichtungMannschaften.getMain(teamId))));

    for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      Plan.setAttributValue(subNode2,
                        GetEnumName(System.TypeInfo(TMannschaftsKostenType), Ord(KostenType)),
                        GetEnumName(System.TypeInfo(TCalculateOptionsValues), Ord(GewichtungMannschaften.getDetail(teamId, KostenType))));
    end;
  end;



  subNode := root.AddChild('round-planing');

  Plan.setAttributValue(subNode,
                      'planing',
                      GetEnumName(System.TypeInfo(TRoundPlaning), Ord(RoundPlaning)));


  subNode := root.AddChild('double-round');

  if DoubleRound then
  begin
    Plan.setAttributValue(subNode, 'active', 'true');
  end
  else
  begin
    Plan.setAttributValue(subNode, 'active', 'false');
  end;

  if AutomaticMidDate then
  begin
    Plan.setAttributValue(subNode, 'automatic-mid-date', 'true');
  end
  else
  begin
    Plan.setAttributValue(subNode, 'automatic-mid-date', 'false');
    if MidDate1 <> 0 then
      Plan.setAttributValue(subNode, 'mid-date-1', FormatDateForExport(MidDate1));
    if MidDate2 <> 0 then
      Plan.setAttributValue(subNode, 'mid-date-2', FormatDateForExport(MidDate2));
  end;


  XMLDoc.SaveToFile(FileName);

end;


procedure TCalculateOptions.setGewichtungPlan(KostenType: TPlanKostenType; value: TCalculateOptionsValues);
begin
  case KostenType of
    pktGameDayOverlap:
          SpieltagOverlapp := Value;
    pktGameDayLength:
          SpieltagGewichtung := Value;
    pktLastGameDayOverlap:
          LastSpieltagOverlapp := Value;
    pktLastGameDayLength:
          LastSpieltagGewichtung := Value;
    pktSisterGamesAtBegin:
          SisterTeamsAmAnfangGewichtung := Value;
  end;
end;

function TCalculateOptions.getAutomaticMidDate1(Plan: TPlan): TDateTime;
var
  SpieltageMondays: TList<TDateTime>;
  DateFrom, DateTo: TDateTime;
begin
  DateFrom := Plan.GetFirstDate();
  DateTo := Plan.DateBeginRueckrundeInXML;

  if Plan.getOptions.RoundPlaning = rpHalfRound then
  begin
    DateFrom := Plan.DateBegin;
    DateTo := Plan.DateEnd;
  end;


  Result := ((DateTo - DateFrom) / 2) + DateFrom;
  SpieltageMondays := TList<TDateTime>.Create;
  Plan.getSpieltagMondays(SpieltageMondays, DateFrom, DateTo);
  if SpieltageMondays.Count > 2 then
  begin
    Result := SpieltageMondays[SpieltageMondays.Count div 2];
  end;
  SpieltageMondays.Free;
end;

function TCalculateOptions.getAutomaticMidDate2(Plan: TPlan): TDateTime;
var
  SpieltageMondays: TList<TDateTime>;
begin
  Result := ((Plan.GetLastDate() - Plan.DateBeginRueckrundeInXML) / 2) + Plan.DateBeginRueckrundeInXML;
  SpieltageMondays := TList<TDateTime>.Create;
  Plan.getSpieltagMondays(SpieltageMondays, Plan.DateBeginRueckrundeInXML, Plan.GetLastDate());
  if SpieltageMondays.Count > 2 then
  begin
    Result := SpieltageMondays[SpieltageMondays.Count div 2];
  end;
  SpieltageMondays.Free;
end;


procedure TCalculateOptions.getWeightMessagesDiffToDefault(Plan: TPlan; Messages: TStrings);
var
  i: TMannschaftsKostenType;
  Mannschaft: TMannschaft;
begin
  if SpieltagOverlapp <> coNormal then
    Messages.Add('  ' + 'berlappung der Spieltage: ' + cCalculateOptionsNamen[SpieltagOverlapp]);

  if SpieltagGewichtung <> coNormal then
    Messages.Add('  ' + 'Lnge der Spieltage: ' + cCalculateOptionsNamen[SpieltagGewichtung]);

  if SisterTeamsAmAnfangGewichtung <> coNormal then
    Messages.Add('  ' + 'Vereinsinterne Spiele am Anfang: ' + cCalculateOptionsNamen[SisterTeamsAmAnfangGewichtung]);

  if LastSpieltagOverlapp <> coNormal then
    Messages.Add('  ' + 'berlappung letzer Spieltage: ' + cCalculateOptionsNamen[LastSpieltagOverlapp]);

  if LastSpieltagGewichtung <> coNormal then
    Messages.Add('  ' + 'Lnge letzter Spieltage: ' + cCalculateOptionsNamen[LastSpieltagGewichtung]);


  for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    if GewichtungMainMannschaftsKosten[i] <> coNormal then
      Messages.Add('  ' + cMannschaftsKostenTypeNamen[i] + ': ' + cCalculateOptionsNamen[GewichtungMainMannschaftsKosten[i]]);
  end;

  for Mannschaft in Plan.Mannschaften do
  begin
    if GewichtungMannschaften.getMain(Mannschaft.teamId) <> coNormal then
      Messages.Add('  ' + Mannschaft.teamName + ': ' + cCalculateOptionsNamen[GewichtungMannschaften.getMain(Mannschaft.teamId)]);

    for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      if GewichtungMannschaften.getDetail(Mannschaft.teamId, i) <> coNormal then
        Messages.Add('  ' + Mannschaft.teamName + ' ' + cMannschaftsKostenTypeNamen[i] + ': ' + cCalculateOptionsNamen[GewichtungMannschaften.getDetail(Mannschaft.teamId, i)]);
    end;

  end;

end;


procedure TCalculateOptions.getDiffToDefault(Plan: TPlan; Messages: TStrings);
var
  WeightMessages: TStrings;
begin
  WeightMessages := TStringList.Create;

  getWeightMessagesDiffToDefault(Plan, WeightMessages);

  if WeightMessages.Count > 0 then
  begin
    Messages.Add('Gewichtung');
    Messages.AddStrings(WeightMessages);
    Messages.Add('');
  end;
  WeightMessages.Free;


  if not FreitagIsAllowedBy60km then
  begin
    Messages.Add('Freitag ist bei der 60km-Regel nicht erlaubt');
    Messages.Add('');
  end;

  if RoundPlaning  in [rpHalfRound] then
  begin
    Messages.Add('Halbrunde wird generiert');
    Messages.Add('');
  end;

  if RoundPlaning in [rpFirstOnly] then
  begin
    Messages.Add('Nur Vorrunde wird generiert');
    Messages.Add('');
  end;

  if RoundPlaning = rpSecondOnly then
  begin
    Messages.Add('Nur Rckrunde wird generiert');
    Messages.Add('');
  end;

  if RoundPlaning = rpCorona then
  begin
    Messages.Add('Nur die "Corona"-Rckrunde wird generiert');
    Messages.Add('');
  end;

  if DoubleRound then
  begin
    Messages.Add('Doppelrunde ist aktiv');
    Messages.Add('');
  end;

end;

function TCalculateOptions.getGewichtungPlan(KostenType: TPlanKostenType): TCalculateOptionsValues;
begin
  Result := coNormal;
  case KostenType of
    pktGameDayOverlap:
      Result := SpieltagOverlapp;
    pktGameDayLength:
      Result := SpieltagGewichtung;
    pktLastGameDayOverlap:
      Result := LastSpieltagOverlapp;
    pktLastGameDayLength:
      Result := LastSpieltagGewichtung;
    pktSisterGamesAtBegin:
      Result := SisterTeamsAmAnfangGewichtung;
  end;
end;

function TCalculateOptions.ToString: string;
var
  i: TMannschaftsKostenType;
begin
  Result := BoolToStr(FreitagIsAllowedBy60km) + ', ' + GetEnumName(TypeInfo(TRoundPlaning), Ord(RoundPlaning)) + ', ' + BoolToStr(DoubleRound) + ', '
            + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(SpieltagGewichtung))  + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(SisterTeamsAmAnfangGewichtung))
            + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(LastSpieltagGewichtung))
            + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(LastSpieltagOverlapp));

  Result := Result + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(SpieltagOverlapp))  + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(SisterTeamsAmAnfangGewichtung));

  for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    Result := Result + ', ' + GetEnumName(TypeInfo(TCalculateOptionsValues), Ord(GewichtungMainMannschaftsKosten[i]));
  end;
end;

procedure TCalculateOptions.Assign(Source: TCalculateOptions);
var
  i: TMannschaftsKostenType;
begin
  if Source <> Self then
  begin
    FreitagIsAllowedBy60km := Source.FreitagIsAllowedBy60km;
    RoundPlaning := Source.RoundPlaning;
    DoubleRound := Source.DoubleRound;
    AutomaticMidDate := Source.AutomaticMidDate;
    MidDate1 := Source.MidDate1;
    MidDate2 := Source.MidDate2;

    SpieltagGewichtung := Source.SpieltagGewichtung;
    SpieltagOverlapp := Source.SpieltagOverlapp;
    LastSpieltagGewichtung := Source.LastSpieltagGewichtung;
    LastSpieltagOverlapp := Source.LastSpieltagOverlapp;
    SisterTeamsAmAnfangGewichtung := Source.SisterTeamsAmAnfangGewichtung;

    for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      GewichtungMainMannschaftsKosten[i] := Source.GewichtungMainMannschaftsKosten[i];
    end;

    GewichtungMannschaften.Assign(Source.GewichtungMannschaften);

  end;
end;


{ TWunschTerminComparer }


{ TRoundInfo }

function TRoundInfo.DateInRound(Date: TDateTime): boolean;
begin
  if (Date >= DateFrom) and (Date < DateTo) then
  begin
    Result := True;
  end
  else
  begin
    Result := False;
  end;

end;

{ TSavedPlan }

constructor TSavedPlan.Create;
begin
  inherited;
  Plan := TPlan.Create;
end;

destructor TSavedPlan.Destroy;
begin
  Plan.Free;
  inherited;
end;

procedure DoTests();
begin
  TestCalcSequence;
  TestDateRoutines();
end;


{ TSisterGamesComparer }

function TSisterGamesComparer.Compare(const game1,
  game2: TSisterGames): Integer;
var
  Date1, Date2: TDateTime;
begin
  if game1.Count = 0 then
  begin
    Date1 := 0;
  end
  else
  begin
    Date1 := game1[0].Date;
  end;

  if game2.Count = 0 then
  begin
    Date2 := 0;
  end
  else
  begin
    Date2 := game2[0].Date;
  end;


  if Date1 > Date2 then
  begin
    Result := 1;
    Exit;
  end;

  if Date1 < Date2 then
  begin
    Result := -1;
    Exit;
  end;

  Result := 0;

end;

{ TSisterGames }

function TSisterGames.toSortedArray: TArray<TSisterGame>;
var
  Comparer: TSisterGameComparer;
begin
  Result := ToArray();

  Comparer := TSisterGameComparer.Create;

  TArray.Sort<TSisterGame>(Result, Comparer);

  Comparer.Free;
end;



{ TSisterGameComparer }

function TSisterGameComparer.Compare(const game1, game2: TSisterGame): Integer;
var
  Date1, Date2: TDateTime;
begin
  Date1 := game1.Date;

  Date2 := game2.Date;


  if Date1 > Date2 then
  begin
    Result := 1;
    Exit;
  end;

  if Date1 < Date2 then
  begin
    Result := -1;
    Exit;
  end;

  Result := 0;

end;

{ TAuswaertsKoppel }

procedure TAuswaertsKoppel.Assign(Source: TAuswaertsKoppel);
begin
  TeamA := Source.TeamA;
  TeamB := Source.TeamB;
  Option := Source.Option;
end;

function TAuswaertsKoppel.AsString: String;
begin
  Result := TeamA + ' und ' + TeamB + ' ' + cAuswaertsKoppelTypeName[Option];
end;

function TAuswaertsKoppel.getMannschaftA(Plan: TPlan): TMannschaft;
begin
  if not Assigned(CachedMannschaftA) then
  begin
    CachedMannschaftA := Plan.Mannschaften.FindByName(teamA);
  end;
  Result := CachedMannschaftA;
end;

function TAuswaertsKoppel.getMannschaftB(Plan: TPlan): TMannschaft;
begin
  if not Assigned(CachedMannschaftB) then
  begin
    CachedMannschaftB := Plan.Mannschaften.FindByName(teamB);
  end;
  Result := CachedMannschaftB;
end;

{ TAuswaertsKoppelList }

procedure TAuswaertsKoppelList.Assign(Source: TAuswaertsKoppelList);
var
  i: integer;
  NewValue: TAuswaertsKoppel;
begin
  Clear;
  for i:=0 to Source.Count-1 do
  begin
    NewValue := TAuswaertsKoppel.Create;
    NewValue.Assign(Source[i]);
    Add(NewValue);
  end;
end;



{ TPlanNoGameDay }

procedure TPlanNoGameDay.Assign(Source: TPlanNoGameDay);
begin
  DateFrom := Source.DateFrom;
  DateTo := Source.DateTo;
  Comment := Source.Comment;
end;

{ TPlanNoGameDayList }

procedure TPlanNoGameDayList.Assign(Source: TPlanNoGameDayList);
var
  Elem, NewElem: TPlanNoGameDay;
begin
  Clear();

  for Elem in Source do
  begin
    NewElem := TPlanNoGameDay.Create;
    NewElem.Assign(Elem);
    Add(NewElem);
  end;
end;

{ TPlanMandatoryTime }

procedure TPlanMandatoryTime.Assign(Source: TPlanMandatoryTime);
begin
  DateFrom := Source.DateFrom;
  DateTo := Source.DateTo;
  GameCount := Source.GameCount;
end;

function TPlanMandatoryTime.AsString: String;
var
  S: String;
begin
  if GameCount = 0 then
  begin
    S := ' wurde entfernt';
  end
  else
  begin
    S := ' mindestens ' + IntToStr(GameCount) + ' Spiele';
  end;

  Result := MyFormatDate(DateFrom) + ' - ' + MyFormatDate(DateTo) + S;

end;

function TPlanMandatoryTime.IsTheSame(Source: TPlanMandatoryTime): boolean;
begin
  Result := (DateFrom = Source.DateFrom)
              and (DateTo = Source.DateTo)
              and (GameCount = Source.GameCount);
end;

{ TPlanMandatoryTimeList }

type
  TPlanMandatoryTimeComparer = class(TComparer<TPlanMandatoryTime>)
    function Compare(const Left, Right: TPlanMandatoryTime): Integer; override;
  end;

function TPlanMandatoryTimeComparer.Compare(const Left, Right: TPlanMandatoryTime): Integer;
begin
  Result := CompareDateTime(Left.DateFrom, Right.DateFrom);

  if Result = 0 then
  begin
    Result := CompareDateTime(Left.DateTo, Right.DateTo);
  end;

  if Result = 0 then
  begin
    Result := Left.GameCount - Right.GameCount;
  end;


end;


procedure TPlanMandatoryTimeList.Assign(Source: TPlanMandatoryTimeList);
var
  Elem, NewElem: TPlanMandatoryTime;
begin
  if Self <> Source then
  begin
    Clear;
    for Elem in Source do
    begin
      NewElem := TPlanMandatoryTime.Create;
      NewElem.Assign(Elem);
      Add(NewElem);
    end;
  end;
end;

constructor TPlanMandatoryTimeList.Create;
begin
  inherited Create(TPlanMandatoryTimeComparer.Create, True);
end;

function TPlanMandatoryTimeList.FindValueByDate(Value: TPlanMandatoryTime): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Count-1 do
  begin
    if (Value.DateFrom = Self[i].DateFrom)
        and (Value.DateTo = Self[i].DateTo)
    then
    begin
      Result := i;
    end;
  end;
end;

{ TAuswaertsKoppelCachedTeams }


constructor TAuswaertsKoppelCachedTeams.Create;
begin
  inherited Create();
  TeamsOnDay := TList<String>.Create;
  TeamIdToTeamId := TObjectDictionary<String, TList<String>>.Create([doOwnsValues]);
end;

destructor TAuswaertsKoppelCachedTeams.Destroy;
begin
  TeamsOnDay.Free;
  TeamIdToTeamId.Free;
  inherited;
end;


procedure TAuswaertsKoppelCachedTeams.AddTeamIdToTeamId(TeamA, TeamB: String);
var
  Values: TList<String>;
begin
  if not TeamIdToTeamId.ContainsKey(TeamA) then
  begin
    Values := TList<String>.Create;
    TeamIdToTeamId.Add(TeamA, Values);
  end
  else
  begin
    Values := TeamIdToTeamId[TeamA];
  end;

  if not Values.Contains(TeamB) then
  begin
    Values.Add(TeamB);
  end;
end;


{ TPlanDataCache }

constructor TPlanDataCache.Create;
begin
  inherited Create;
  sollGamesByDate := nil;
end;

destructor TPlanDataCache.Destroy;
begin
  sollGamesByDate.Free;
  inherited;
end;

function TPlanDataCache.getSollGamesByDate: TDictionary<string, MyDouble>;
begin
  if not Assigned(sollGamesByDate) then
  begin
    sollGamesByDate := TDictionary<String, MyDouble>.Create;
  end;
  Result := sollGamesByDate;
end;

{ TGewichtungMannschaften }

procedure TGewichtungMannschaften.addDetail(teamId: String; Art: TMannschaftsKostenType; value: TCalculateOptionsValues);
var
  ArrayValue: TMannschaftsKostenArray;
begin
  AddTeam(teamId);

  ArrayValue := GewichtungMannschaftenDetail[teamId];
  ArrayValue[Art] := value;
  GewichtungMannschaftenDetail[teamId] := ArrayValue;
end;

procedure TGewichtungMannschaften.addMain(teamId: String; value: TCalculateOptionsValues);
begin
  AddTeam(teamId);

  GewichtungMannschaften[teamId] := value;
end;

procedure TGewichtungMannschaften.AddTeam(teamId: String);
var
  ArrayValue: TMannschaftsKostenArray;
  KostenType: TMannschaftsKostenType;
begin
  if not GewichtungMannschaften.ContainsKey(teamId) then
  begin
    GewichtungMannschaften.Add(teamId, coNormal);

    for KostenType := low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      ArrayValue[KostenType] := coNormal;
    end;
    GewichtungMannschaftenDetail.Add(teamId, ArrayValue);
  end;
end;

procedure TGewichtungMannschaften.Assign(Source: TGewichtungMannschaften);
var
  Id: String;
  KostenType: TMannschaftsKostenType;
begin
  Clear;

  for Id in Source.TeamIds do
  begin
    addMain(Id, Source.getMain(id));

    for KostenType := low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      addDetail(Id, KostenType, Source.getDetail(id, KostenType));
    end;

  end;
end;

procedure TGewichtungMannschaften.Clear;
begin
  GewichtungMannschaften.Clear;
  GewichtungMannschaftenDetail.Clear;
end;

constructor TGewichtungMannschaften.Create;
begin
  inherited;
  GewichtungMannschaften := TDictionary<String, TCalculateOptionsValues>.Create;
  GewichtungMannschaftenDetail := TDictionary<String, TMannschaftsKostenArray>.Create;
end;

destructor TGewichtungMannschaften.Destroy;
begin
  GewichtungMannschaften.Free;
  GewichtungMannschaftenDetail.Free;
  inherited;
end;

function TGewichtungMannschaften.getDetail(teamId: String; Art: TMannschaftsKostenType): TCalculateOptionsValues;
var
  Ret: TMannschaftsKostenArray;
begin
  Result := coNormal;

  if GewichtungMannschaftenDetail.TryGetValue(teamId, Ret) then
  begin
    Result := Ret[Art];
  end;
end;

function TGewichtungMannschaften.getMain(teamId: String): TCalculateOptionsValues;
var
  Ret: TCalculateOptionsValues;
begin
  Result := coNormal;

  if GewichtungMannschaften.TryGetValue(teamId, Ret) then
  begin
    Result := Ret;
  end;
end;

function TGewichtungMannschaften.GetTeamIds: TEnumerable<String>;
begin
  Result := GewichtungMannschaften.Keys;
end;

{ TScheduleGame }

function TScheduleGame.DateValid: boolean;
begin
  Result := Date <> 0;
end;

{ TSchedule }

constructor TSchedule.Create;
begin
  inherited Create(true);
  ScheduleComparer := TScheduleComparer.Create;
end;

destructor TSchedule.Destroy;
begin
  ScheduleComparer.Free;
  inherited;
end;

procedure TSchedule.SortByDate;
begin
  Sort(ScheduleComparer);
end;

{ TScheduleComparer }

function TScheduleComparer.Compare(const game1, game2: TScheduleGame): Integer;
begin
  if game1.Date > game2.Date then
  begin
    Result := 1;
    Exit;
  end;

  if game1.Date < game2.Date then
  begin
    Result := -1;
    Exit;
  end;

  Result := compareStr(game1.MannschaftHeimName + ' ' + game1.MannschaftGastName, game2.MannschaftHeimName + ' ' + game2.MannschaftGastName);

end;

{ TMannschaftDataCache }

constructor TMannschaftDataCache.Create;
begin
  inherited Create;
  CachedAuswaertsKoppelTeams := TAuswaertsKoppelCachedTeams.Create;
end;

destructor TMannschaftDataCache.Destroy;
begin
  CachedAuswaertsKoppelTeams.Free;
  CachedTeamsForForceHomeGames.Free;
  inherited;
end;

{ THomeRight }

procedure THomeRight.Assign(Source: THomeRight);
begin
  Team := Source.Team;
  Option := Source.Option;
end;

function THomeRight.AsString: String;
begin
  Result := '';

  if Option = hrRound1 then
  begin
    Result := 'Heimspiel gegen ' + Team + ' in der Vorrunde';
  end;

  if Option = hrRound2 then
  begin
    Result := 'Heimspiel gegen ' + Team + ' in der Rckrunde';
  end;

end;

{ THomeRightList }

procedure THomeRightList.Assign(Source: THomeRightList);
var
  i: integer;
  NewValue: THomeRight;
begin
  Clear;
  for i:=0 to Source.Count-1 do
  begin
    NewValue := THomeRight.Create;
    NewValue.Assign(Source[i]);
    Add(NewValue);
  end;
end;

function THomeRightList.getValueForTeam(teamName: String): THomeRightValue;
var
  i: Integer;
begin
  Result := hrNone;
  for i := 0 to Count-1 do
  begin
    if teamName = Self[i].Team then
    begin
      result := Self[i].Option;
      break;
    end;
  end;
end;

initialization
  MinGameDistance := EncodeTime(3, 30, 0, 0);
{$ifdef CACHE_HIT_TEST}
  CacheHit := 0;
  CacheMiss := 0;
{$endif}
{$ifopt D+}
  DoTests();
{$endif}
end.
