unit PlanPanel;

(*
  Copyright 2017 Andreas Hofmann
  Lizenz:
  Der Sourcecode steht unter der GPL V3
  siehe <http://www.gnu.org/licenses/>.
*)

interface

uses
  System.Types, System.UITypes, Windows, Messages, SysUtils, Variants, Classes,
  Graphics, Controls, Forms,
  Dialogs, PlanTypes, PlanUtils, StdCtrls, math, ExtCtrls, ComCtrls, StrUtils,
  System.Generics.Collections, PrintUtil, DialogPrintSelect, System.DateUtils,
  DialogForSingleGewichtungsOptions, Vcl.ToolWin, Vcl.ImgList, System.ImageList;

type
  TFormPlanPanel = class(TForm)
    PaintBoxTermine: TPaintBox;
    ScrollBoxTermine: TScrollBox;
    PanelMain: TPanel;
    pgcMain: TPageControl;
    tsPlanView: TTabSheet;
    tsKosten: TTabSheet;
    ScrollBoxKosten: TScrollBox;
    PaintBoxKosten: TPaintBox;
    tsDiagrams: TTabSheet;
    ScrollBoxVerteilung: TScrollBox;
    PaintBoxVerteilung: TPaintBox;
    tsTerminMeldung: TTabSheet;
    ScrollBoxTerminMeldung: TScrollBox;
    PaintBoxTerminMeldung: TPaintBox;
    tsSisterTeams: TTabSheet;
    ScrollBoxSisterTeams: TScrollBox;
    PaintBoxSisterTeams: TPaintBox;
    ToolBarTerminAuswahl: TToolBar;
    ToolButtonSpielplan: TToolButton;
    ToolButtonMannschaftsplaene: TToolButton;
    ToolButtonMannschaftsplaeneMitNachbarn: TToolButton;
    ImageListToolbar: TImageList;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure PaintBoxPaint(Sender: TObject);
    procedure PaintBoxMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure PaintBoxMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ToolButtonMannschaftsplaeneClick(Sender: TObject);
    procedure ToolButtonSpielplanClick(Sender: TObject);
    procedure ToolButtonMannschaftsplaeneMitNachbarnClick(Sender: TObject);
  private
    FZoomValue: MyDouble;
    ActionRects: TList<TActionRect>;
    function PaintVerteilungMannschaft(Canvas: TZoomCanvas;
      PosY, PosYNext: Integer; Mannschaft: TMannschaft;
      NextMannschaft: TMannschaft; PaintOverlay: boolean): Integer;
    function PaintSpielPlanVerteilung(Canvas: TZoomCanvas; Y: Integer): TSize;
    function xPosByDate(Date, FirstDate: TDateTime): Integer;
    procedure PaintDateLine(Canvas: TZoomCanvas; PosY: Integer);
    function PaintSpielPlanMeldungen(Canvas: TZoomCanvas; Y: Integer): TSize;
    function PaintMeldungMannschaft(Canvas: TZoomCanvas; PosY: Integer;
      Mannschaft: TMannschaft): Integer;
    function PaintSpielAbstandHinRueck(Canvas: TZoomCanvas; Y: Integer): TSize;
    function PaintSpielAbstandMannschaft(Canvas: TZoomCanvas; PosY: Integer;
      Mannschaft: TMannschaft): TSize;
    procedure PaintKostenPaintWithBackColor(Canvas: TZoomCanvas;
      X, Y, Length: Integer; Fehler: Integer; AllCount: Integer;
      Kosten, GesamtKosten: MyDouble; Gewichtung: Integer; ForPrint: boolean);
    function PaintTerminMeldung(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    function PaintSisterTeams(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    function PaintKosten(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    function PaintTermineMain(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    function PaintSpielplan(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    function PaintMannschaftsPlaene(Canvas: TZoomCanvas;
      ForPrint: boolean): TSize;
    function PaintMannschaftsPlaeneWithSisterTeams(Canvas: TZoomCanvas;
      ForPrint: boolean): TSize;
    function PaintMannschaftsPlaeneIntern(Canvas: TZoomCanvas;
      ForPrint: boolean; Title: String; WithSisterTeams: boolean): TSize;
    function PaintVerteilung(Canvas: TZoomCanvas; ForPrint: boolean): TSize;
    procedure InvalidateAll;
    function PaintSpielPlanOverlap(Canvas: TZoomCanvas;
      Top, Height: Integer): TSize;
    function PaintSpielWechselHeimAuswaerts(Canvas: TZoomCanvas;
      Y: Integer): TSize;
    function PaintWechselHeimAuswaertsMannschaft(Canvas: TZoomCanvas;
      PosY: Integer; Mannschaft: TMannschaft): Integer;
    function PaintSpielAbstaende(Canvas: TZoomCanvas; Y: Integer): TSize;
    function PaintSpielAbstaendeMannschaft(Canvas: TZoomCanvas; PosY: Integer;
      Mannschaft: TMannschaft): Integer;
    procedure PaintGameBubble(Canvas: TZoomCanvas; PosX, PosY: Integer;
      Game: TGame; HeimSpiel, KoppelTermin: boolean);
    function PaintOneTermin(Canvas: TZoomCanvas; yPos: Integer; Game: TGame;
      WithSisterTeams: boolean; OnlyShowSisterGamesWithThatMannschaften
      : array of String): Integer;
    function PaintGamesPerWeek(Canvas: TZoomCanvas; Y: Integer): TSize;
    function PaintWeekValues(Canvas: TZoomCanvas; Y: Integer;
      SpieltageMondays: TList<TDateTime>; Title: String; Values: TList<String>;
      ValuesPrintRed: TList<boolean>): TSize;
    function getXBySpieltageMondays(SpieltageMondays: TList<TDateTime>;
      index: Integer): Integer;
    function PaintGamesPerWeekOneRound(Canvas: TZoomCanvas; Y: Integer;
      SpieltageMondays: TList<TDateTime>): TSize;
    procedure PaintGamesPerWeekHeader(Canvas: TZoomCanvas; Y: Integer;
      SpieltageMondays: TList<TDateTime>);
    procedure PaintDateRaster(Canvas: TZoomCanvas;
      PosY, Count, Height: Integer);
    procedure PaintTableBackground(Canvas: TZoomCanvas;
      PosY, Count, Height, width: Integer);
    function PaintSpielRanking(Canvas: TZoomCanvas; Y: Integer): TSize;
    function PaintSpielRankingMannschaft(Canvas: TZoomCanvas; PosY: Integer;
      Mannschaft: TMannschaft): Integer;
    procedure ClearActionRects(PaintBox: TObject);
    procedure AddActionRects(PaintBox: TObject; Values: TList<TActionRect>;
      Canvas: TZoomCanvas);
    function PaintOneTerminSisterGames(Canvas: TZoomCanvas; yPos: Integer;
      Game: TGame; OnlyShowGamesWithThatMannschaften: array of String): Integer;
    { Private-Deklarationen }
  public
    { Public-Deklarationen }
    Plan: TPlan;
    procedure SetPanel(ParentPanel: TPanel);
    procedure SetPlan(Plan: TPlan);
    procedure Print();
    procedure SetZoom(Value: MyDouble);
  end;

  TActionRectKosten = class(TActionRectBase)
  end;

  TActionRectKostenPlan = class(TActionRectKosten)
    KostenType: TPlanKostenType;
  public
    constructor Create(KostenType: TPlanKostenType);
    procedure DoAction(ScreenPosition: TPoint); override;
  end;

  TActionRectKostenTypeMannschaft = class(TActionRectKosten)
    KostenType: TMannschaftsKostenType;
  public
    constructor Create(KostenType: TMannschaftsKostenType);
    procedure DoAction(ScreenPosition: TPoint); override;
  end;

  TActionRectKostenMannschaftMain = class(TActionRectKosten)
    teamId: String;
  public
    constructor Create(teamId: String);
    procedure DoAction(ScreenPosition: TPoint); override;
  end;

  TActionRectKostenMannschaftDetail = class(TActionRectKosten)
    teamId: String;
    KostenType: TMannschaftsKostenType;
  public
    constructor Create(teamId: String; KostenType: TMannschaftsKostenType);
    procedure DoAction(ScreenPosition: TPoint); override;
  end;

var
  FormPlanPanel: TFormPlanPanel;

implementation

uses
  AndiGeneratorMain;

{$R *.dfm}

const
  cFontBig = -18;
  cFontNormal = -12;

var
  cColorLineGray: TColor;
  cColorBrushGray: TColor;
  cColorHomeGame: TColor;
  cColorAwayGame: TColor;
  cColorGameDayLine1: TColor;
  cColorGameDayLine2: TColor;
  cColorHomeGameBubble: TColor;
  cColorAwayGameBubble: TColor;
  cColorNormalWunsch: TColor;
  cColorKoppelWunsch: TColor;
  cColorSperrWunsch: TColor;

procedure TFormPlanPanel.FormCreate(Sender: TObject);
begin
  Plan := TPlan.Create;
  pgcMain.ActivePageIndex := 0;
  FZoomValue := 1.0;
  ActionRects := TList<TActionRect>.Create;

  FixControls(Self);
end;

procedure TFormPlanPanel.FormDestroy(Sender: TObject);
begin
  Plan.Free;
  ActionRects.Free;
end;

function TFormPlanPanel.PaintOneTermin(Canvas: TZoomCanvas; yPos: Integer;
  Game: TGame; WithSisterTeams: boolean; OnlyShowSisterGamesWithThatMannschaften
  : array of String): Integer;
var
  Meldung: String;
  Found: boolean;
  WunschTermin: TWunschTermin;
  testGame, gameRef: TGame;
  predefinedGame: TGame;
  OldColor: TColor;
begin
  Result := 20;
  if Game.Date = 0.0 then
  begin
    Canvas.TextOut(0, yPos, '------');
  end
  else
  begin
    Canvas.TextOut(0, yPos, MyFormatDate(Game.Date));
    Canvas.TextOut(100, yPos, MyFormatTime(Game.Date));
  end;

  Canvas.TextOut(150, yPos, Game.MannschaftHeim.teamName);

  Canvas.TextOut(400, yPos, '-');

  Canvas.TextOut(420, yPos, Game.MannschaftGast.teamName);

  Meldung := '';

  if Plan.FreeGameDays.ContainsKey(Trunc(Game.Date)) then
  begin
    Meldung := 'spielfreier Tag';
  end;

  if Game.DateValid then
  begin

    predefinedGame := Plan.predefinedGames.findSameGame(Game);

    if (Assigned(predefinedGame)) then
    begin
      if Meldung <> '' then
        Meldung := Meldung + ', ';
      Meldung := Meldung + 'manuell festgelegter Termin';
    end;

    gameRef := Game.MannschaftHeim.gamesRef.findSameGame(Game);
    if Assigned(gameRef) then
    begin
      if Assigned(gameRef.MannschaftHeim.getKoppeledGame(gameRef)) then
      begin
        if Meldung <> '' then
          Meldung := Meldung + ', ';
        Meldung := Meldung + 'Heimkoppelspiel/Doppelspieltag';
      end;
    end;

    gameRef := Game.MannschaftGast.gamesRef.findSameGame(Game);
    if Assigned(gameRef) then
    begin
      if Assigned(gameRef.MannschaftGast.getKoppeledGame(gameRef)) then
      begin
        if Meldung <> '' then
          Meldung := Meldung + ', ';
        Meldung := Meldung + 'Auswrtskoppelspiel';
      end;
    end;

    Found := False;
    for WunschTermin in Game.MannschaftHeim.WunschTermine do
    begin
      if Game.Date = WunschTermin.Date then
      begin
        Found := True;
        if Game.MannschaftHeim.isAusWeichTermin(WunschTermin.Date) then
        begin
          if Meldung <> '' then
            Meldung := Meldung + ', ';
          Meldung := Meldung + 'Ausweichtermin';
        end;

        if (woAuswaertsKoppelSecondTime in WunschTermin.Options) or
          (woKoppelSecondTime in WunschTermin.Options) then
        begin
          if Meldung <> '' then
            Meldung := Meldung + ', ';
          Meldung := Meldung + 'wegen Koppeltermin genderte Uhrzeit';
        end;

        if (not Assigned(predefinedGame)) and (WunschTermin.Location <> '') then
        begin
          if Meldung <> '' then
            Meldung := Meldung + ', ';
          Meldung := Meldung + 'Spiellokal: ' + WunschTermin.Location;
        end;
        break;
      end;
    end;

    if Assigned(predefinedGame) and (predefinedGame.OverrideLocation <> '') then
    begin
      if Meldung <> '' then
        Meldung := Meldung + ', ';
      Meldung := Meldung + 'Spiellokal: ' + predefinedGame.OverrideLocation;
    end;

    if not Found then
    begin
      if Meldung <> '' then
        Meldung := Meldung + ', ';
      Meldung := Meldung + 'kein Wunschtermin';
    end;

    if nil <> Plan.gamesNotAllowed.findSameGame(Game) then
    begin
      if Meldung <> '' then
        Meldung := Meldung + ', ';
      Meldung := Meldung + 'doppeltes Spiel';
    end;

    if Found then
    begin
      testGame := Plan.gamesAllowed.findSameGame(Game);
      if testGame <> nil then
      begin
        if not Plan.GameIsValid(testGame) then
        begin
          if Meldung <> '' then
            Meldung := Meldung + ', ';
          Meldung := Meldung + 'ungltiger Termin';
        end;
      end;
    end;
  end;

  Canvas.TextOut(720, yPos, Meldung);

  OldColor := Canvas.FontColor;
  if WithSisterTeams then
  begin
    Canvas.FontColor := RGB(128, 128, 128);
    Result := Result + PaintOneTerminSisterGames(Canvas, yPos, Game,
      OnlyShowSisterGamesWithThatMannschaften);
  end;
  Canvas.FontColor := OldColor;
end;

function RemoveTeamNumber(teamName: String): String;
var
  I: Integer;
  PostFix: String;
begin
  Result := teamName;
  for I := 1 to 30 do
  begin
    PostFix := ' ' + DecToRom(I);
    if teamName.EndsWith(PostFix) then
    begin
      Result := Copy(teamName, 1, Length(teamName) - Length(PostFix));
      break;
    end;
  end;
  Result := TrimRight(Result);
end;

// Die Mannschaft AText und einer Mannschaften in AValues sind vom gleichen Verein
function MannschaftMatchVereinStr(AText: string;
  const AValues: array of string): boolean;
var
  S: String;
begin
  Result := False;
  for S in AValues do
  begin
    if RemoveTeamNumber(AText) = RemoveTeamNumber(S) then
    begin
      Result := True;
    end;
  end;
end;

function TFormPlanPanel.PaintOneTerminSisterGames(Canvas: TZoomCanvas;
  yPos: Integer; Game: TGame; OnlyShowGamesWithThatMannschaften
  : array of String): Integer;
var
  MannschaftsIndex: Integer;
  Mannschaft, SisterMannschaft: TMannschaft;
  SisterGames: TSisterGames;
  SisterGame: TSisterGame;
  S: String;
  NormalColor: TColor;
  SisterGame2: TGame;
begin
  Result := 0;
  // Inc(yPos, 5);

  NormalColor := Canvas.FontColor;

  for MannschaftsIndex := 0 to 1 do
  begin
    Mannschaft := nil;
    case MannschaftsIndex of
      0:
        Mannschaft := Game.MannschaftHeim;
      1:
        Mannschaft := Game.MannschaftGast;
    end;

    if Assigned(Mannschaft) then
    begin
      for SisterGames in Mannschaft.SisterGames.toSortedArray do
      begin
        for SisterGame in SisterGames.toSortedArray do
        begin
          if Trunc(SisterGame.Date) = Trunc(Game.Date) then
          begin
            if MannschaftMatchVereinStr(SisterGame.HomeMannschaftName,
              OnlyShowGamesWithThatMannschaften) or
              MannschaftMatchVereinStr(SisterGame.GuestMannschaftName,
              OnlyShowGamesWithThatMannschaften) then
            begin
              Inc(yPos, 20);
              Inc(Result, 20);

              if SisterGame.PreventParallelGame then
              begin
                // Echte Nachbarmannschaft
                Canvas.FontColor := RGB(0, 0, 255);
              end;

              S := MyFormatTime(SisterGame.Date);
              Canvas.TextOut(100, yPos, S);

              Canvas.TextOut(150, yPos, '(' + SisterGame.gender + ') ' +
                SisterGame.HomeMannschaftName);

              Canvas.TextOut(400, yPos, '-');

              Canvas.TextOut(420, yPos, SisterGame.GuestMannschaftName);

              S := '';
              if SisterGame.Location <> '' then
              begin
                S := 'Spiellokal: ' + SisterGame.Location;
              end;

              Canvas.TextOut(720, yPos, S);

              Canvas.FontColor := NormalColor;
            end;
          end;
        end;
      end;

      for SisterMannschaft in Mannschaft.SisterTeamsInPlan do
      begin
        for SisterGame2 in SisterMannschaft.gamesRef do
        begin
          // Ich spiele mit
          if (SisterGame2.MannschaftHeim <> Mannschaft) and
            (SisterGame2.MannschaftGast <> Mannschaft) then
          begin
            if Trunc(SisterGame2.Date) = Trunc(Game.Date) then
            begin
              if MannschaftMatchVereinStr(SisterGame2.MannschaftHeim.teamName,
                OnlyShowGamesWithThatMannschaften) or
                MannschaftMatchVereinStr(SisterGame2.MannschaftGast.teamName,
                OnlyShowGamesWithThatMannschaften) then
              begin
                Inc(yPos, 20);
                Inc(Result, 20);

                if 1 = Abs(Mannschaft.teamNumber - SisterMannschaft.teamNumber)
                then
                begin
                  // Echte Nachbarmannschaft
                  Canvas.FontColor := RGB(0, 0, 255);
                end;

                S := MyFormatTime(SisterGame2.Date);
                Canvas.TextOut(100, yPos, S);

                Canvas.TextOut(150, yPos, '(' + Plan.gender + ') ' +
                  SisterGame2.MannschaftHeim.teamName);

                Canvas.TextOut(400, yPos, '-');

                Canvas.TextOut(420, yPos, SisterGame2.MannschaftGast.teamName);

                S := SisterGame2.Location;
                if S <> '' then
                begin
                  S := 'Spiellokal: ' + S;
                end;

                Canvas.TextOut(720, yPos, S);

                Canvas.FontColor := NormalColor;
              end;
            end;
          end;
        end;
      end;
    end;
  end;

  Inc(Result, 10);

end;

function TFormPlanPanel.PaintSpielplan(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
var
  I: Integer;
  Game: TGame;
  games: TGameList;
  yPos: Integer;
  Height: Integer;
  LastGameWeek, currWeek: Integer;
begin

  yPos := 0;
  if Plan.NoGames() then
  begin
    Canvas.TextOut(0, 0, 'Keine Spiele terminiert');
  end
  else
  begin
    if not ForPrint then
    begin
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, 0, 'Spielplan:');
      Inc(yPos, 40);
      Canvas.FontHeight := cFontNormal;
    end;

    games := TGameList.Create(True);

    Plan.getCloneOfAllGames(games);
    games.SortByDate;

    LastGameWeek := -1;
    for I := 0 to games.Count - 1 do
    begin
      Game := games[I];

      if Game.NotNecessary then
      begin
        continue;
      end;

      if Game.Date = 0.0 then
      begin
        currWeek := 0;
      end
      else
      begin
        currWeek := KalenderWoche(Game.Date);
      end;

      if (currWeek <> LastGameWeek) and (LastGameWeek >= 0) then
      begin
        yPos := yPos + 10;
      end;
      LastGameWeek := currWeek;

      Height := PaintOneTermin(Canvas, yPos, Game, False, []);

      Inc(yPos, Height)
    end;
    games.Free;
  end;

  Result.cy := 40 + yPos;
  Result.cx := 1500;

end;

function TFormPlanPanel.PaintMannschaftsPlaene(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
begin
  Result := PaintMannschaftsPlaeneIntern(Canvas, ForPrint,
    'Mannschaftsplne:', False);
end;

function TFormPlanPanel.PaintMannschaftsPlaeneIntern(Canvas: TZoomCanvas;
  ForPrint: boolean; Title: String; WithSisterTeams: boolean): TSize;
var
  I: Integer;
  Game: TGame;
  games: TGameList;
  yPos, Height: Integer;
  Mannschaft: TMannschaft;
  LastRound, currentRound: Integer;
begin

  yPos := 0;
  if Plan.NoGames() then
  begin
    Canvas.TextOut(0, 0, 'Keine Spiele terminiert');
  end
  else
  begin
    if not ForPrint then
    begin
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, 0, Title);
      Inc(yPos, 40);
      Canvas.FontHeight := cFontNormal;
    end;

    games := TGameList.Create(True);

    Plan.getCloneOfAllGames(games);
    games.SortByDate;

    for Mannschaft in Plan.Mannschaften do
    begin
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, yPos, Mannschaft.teamName + ':');
      Canvas.FontHeight := cFontNormal;
      Inc(yPos, 30);

      LastRound := 0;

      for I := 0 to games.Count - 1 do
      begin
        Game := games[I];

        if Game.NotNecessary then
        begin
          continue;
        end;

        if (Game.MannschaftHeim = Mannschaft) or
          (Game.MannschaftGast = Mannschaft) then
        begin
          currentRound := Plan.GetRoundNumberByDate(Game.Date);

          if currentRound > LastRound then
          begin
            Inc(yPos, 30)
          end;
          LastRound := currentRound;

          Height := PaintOneTermin(Canvas, yPos, Game, WithSisterTeams,
            [Mannschaft.teamName]);
          Inc(yPos, Height)
        end;
      end;
      Inc(yPos, 30)

    end;
    games.Free;
  end;

  Result.cy := 40 + yPos;
  Result.cx := 1500;

end;

function TFormPlanPanel.PaintMannschaftsPlaeneWithSisterTeams
  (Canvas: TZoomCanvas; ForPrint: boolean): TSize;
begin
  Result := PaintMannschaftsPlaeneIntern(Canvas, ForPrint,
    'Mannschaftsplne mit Nachbarmannschaften:', True);
end;

procedure TFormPlanPanel.SetPanel(ParentPanel: TPanel);
begin
  if Assigned(ParentPanel) then
  begin
    PanelMain.Parent := ParentPanel;
  end
  else
  begin
    PanelMain.Parent := Self;
  end;
  InvalidateAll();
end;

procedure TFormPlanPanel.InvalidateAll();
begin
  ScrollBoxTerminMeldung.Invalidate;
  ScrollBoxSisterTeams.Invalidate;
  ScrollBoxVerteilung.Invalidate;
  ScrollBoxKosten.Invalidate;
  ScrollBoxTermine.Invalidate;
  PaintBoxTermine.Invalidate;
  PaintBoxKosten.Invalidate;
  PaintBoxVerteilung.Invalidate;
  PaintBoxTerminMeldung.Invalidate;
  PaintBoxSisterTeams.Invalidate;
end;

procedure TFormPlanPanel.SetPlan(Plan: TPlan);
begin
  if Plan <> nil then
  begin
    Self.Plan.Assign(Plan);
    Self.Plan.CalculateKosten(-1);

    InvalidateAll();
  end;
end;

procedure TFormPlanPanel.SetZoom(Value: MyDouble);
begin
  FZoomValue := Value;
  InvalidateAll();
end;

procedure TFormPlanPanel.ToolButtonMannschaftsplaeneClick(Sender: TObject);
begin
  ToolButtonMannschaftsplaeneMitNachbarn.Down := False;
  ToolButtonSpielplan.Down := False;
  ToolButtonMannschaftsplaene.Down := True;
  InvalidateAll();
end;

procedure TFormPlanPanel.ToolButtonMannschaftsplaeneMitNachbarnClick
  (Sender: TObject);
begin
  ToolButtonMannschaftsplaene.Down := False;
  ToolButtonSpielplan.Down := False;
  ToolButtonMannschaftsplaeneMitNachbarn.Down := True;
  InvalidateAll();
end;

procedure TFormPlanPanel.ToolButtonSpielplanClick(Sender: TObject);
begin
  ToolButtonMannschaftsplaene.Down := False;
  ToolButtonMannschaftsplaeneMitNachbarn.Down := False;
  ToolButtonSpielplan.Down := True;
  InvalidateAll();
end;

procedure TFormPlanPanel.PaintKostenPaintWithBackColor(Canvas: TZoomCanvas;
  X, Y, Length: Integer; Fehler: Integer; AllCount: Integer;
  Kosten, GesamtKosten: MyDouble; Gewichtung: Integer; ForPrint: boolean);
var
  S: string;
  OldColor: TColor;
  OldTextColor: TColor;
  R: Integer;
  Size: TSize;
  OldFontHeight: Integer;
  BoxLength: Integer;
begin

  BoxLength := Length;
  OldColor := Canvas.BrushColor;

  R := 255;
  if GesamtKosten > 0 then
  begin
    R := Min(255, Max(0, 255 - Trunc(255.0 * Kosten / GesamtKosten)));
  end;

  if Fehler >= 0 then
  begin
    (*
      if (Fehler = 0) and (AllCount <= 0) then
      begin
      S := '---';
      end
      else
    *)
    if AllCount >= 0 then
    begin
      S := IntToStr(Fehler) + ' /' + IntToStr(AllCount) + ' (' +
        MyFormatFloatShort(Kosten) + ')';
    end
    else
    begin
      S := IntToStr(Fehler) + ' (' + MyFormatFloatShort(Kosten) + ')';
    end;
  end
  else
  begin
    S := MyFormatFloatShort(Kosten);
  end;

  // if not ForPrint then
  begin
    if R < 240 then
    begin
      Size := Canvas.TextExtent(S);
      Length := Size.cx;

      Canvas.BrushColor := RGB(255, R, R);
      Canvas.FillRect(Rect(X - 2, Y - 2, X + Length + 4, Y + 15));
    end;
  end;

  Canvas.TextOut(X, Y, S);

  S := '';

  OldTextColor := Canvas.FontColor;

  if Gewichtung <= -1000 then
  begin
    S := 'X';
    Canvas.FontColor := clBlack;
  end
  else if Gewichtung < 0 then
  begin
    S := IntToStr(Gewichtung);
    Canvas.FontColor := clRed;
  end
  else if Gewichtung > 0 then
  begin
    S := '+' + IntToStr(Gewichtung);
    Canvas.FontColor := clGreen;
  end
  else
  begin
    S := '';
  end;

  OldFontHeight := Canvas.FontHeight;

  Canvas.FontHeight := 11;

  Size := Canvas.TextExtent(S);
  Canvas.TextOut(X + BoxLength + 20 - Size.cx, Y - 0, S);

  Canvas.FontHeight := OldFontHeight;
  Canvas.FontColor := OldTextColor;

  Canvas.BrushColor := OldColor;
end;

function TFormPlanPanel.PaintKosten(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
var
  yPos, xPos: Integer;
  I, j: Integer;
  Kosten: TMannschaftsKosten;
  AktType: TMannschaftsKostenType;
  Summen: TList<MyDouble>;
  GesamtKosten: MyDouble;
  KostenBreite, KostenOverlap, LastKostenBreite, LastKostenOverlap: MyDouble;
  VisibleMannschaftsKosten: Integer;
  GesamtBreite: Integer;
  Action: TActionRectKosten;
const
  cColWidth = 110;
  cRowHeight = 30;
  cActionHeight = 20;
begin
  yPos := 0;

  if Plan.NoGames() then
  begin
    Canvas.TextOut(0, 0, 'Keine Spiele terminiert');
    Inc(yPos, 40);
  end
  else
  begin

    if not ForPrint then
    begin
      yPos := 0;
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, 0, 'Kosten:');
      Inc(yPos, 30);
      Canvas.FontHeight := cFontNormal;
    end;

    GesamtKosten := Plan.CalculateKosten(-1);
    Canvas.TextOut(0, yPos, 'Gesamtkosten:');
    Canvas.TextOut(200, yPos, MyFormatFloatShort(GesamtKosten));

    if Plan.CalculateFehlterminKosten() > 0 then
    begin
      Inc(yPos, 20);
      Canvas.TextOut(0, yPos, 'Spiele nicht terminiert:');
      PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
        Plan.CalculateFehlterminKosten(), GesamtKosten, 0, ForPrint);
    end;

    if Plan.CalculateNotAllowedKosten() > 0 then
    begin
      Inc(yPos, 20);
      Canvas.TextOut(0, yPos, 'ungltige Spiele:');
      PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
        Plan.CalculateNotAllowedKosten(), GesamtKosten, 0, ForPrint);
    end;

    if Plan.CalculateFreeGameDateKosten() > 0 then
    begin
      Inc(yPos, 20);
      Canvas.TextOut(0, yPos, 'Spiele an spielfreien Tagen:');
      PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
        Plan.CalculateFreeGameDateKosten(), GesamtKosten, 0, ForPrint);
    end;

    Plan.CalculateSpielTagKosten(KostenBreite, KostenOverlap, LastKostenBreite,
      LastKostenOverlap);

    Inc(yPos, 20);

    Canvas.TextOut(0, yPos, 'berlappung Spieltage:');

    PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1, KostenOverlap,
      GesamtKosten, cCalculateOptionsDisplayInteger
      [Plan.getOptions.SpieltagOverlapp], ForPrint);
    Action := TActionRectKostenPlan.Create(pktGameDayOverlap);
    Action.setRect(200, yPos, 70, cActionHeight);
    Canvas.AddActionRect(Action);

    Inc(yPos, 20);

    Canvas.TextOut(0, yPos, 'Lnge Spieltage:');

    PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1, KostenBreite,
      GesamtKosten, cCalculateOptionsDisplayInteger
      [Plan.getOptions.SpieltagGewichtung], ForPrint);
    Action := TActionRectKostenPlan.Create(pktGameDayLength);
    Action.setRect(200, yPos, 70, cActionHeight);
    Canvas.AddActionRect(Action);

    Inc(yPos, 20);

    Canvas.TextOut(0, yPos, 'berlappung letzer Spieltag:');

    PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
      LastKostenOverlap, GesamtKosten, cCalculateOptionsDisplayInteger
      [Plan.getOptions.LastSpieltagOverlapp], ForPrint);
    Action := TActionRectKostenPlan.Create(pktLastGameDayOverlap);
    Action.setRect(200, yPos, 70, cActionHeight);
    Canvas.AddActionRect(Action);

    Inc(yPos, 20);

    Canvas.TextOut(0, yPos, 'Lnge letzer Spieltag:');

    PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
      LastKostenBreite, GesamtKosten, cCalculateOptionsDisplayInteger
      [Plan.getOptions.LastSpieltagGewichtung], ForPrint);
    Action := TActionRectKostenPlan.Create(pktLastGameDayLength);
    Action.setRect(200, yPos, 70, cActionHeight);
    Canvas.AddActionRect(Action);

    if Plan.CalculateSisterTeamsAmAnfang() > 0 then
    begin
      Inc(yPos, 20);
      Canvas.TextOut(0, yPos, 'Vereinsinterne Spiele am Anfang:');
      PaintKostenPaintWithBackColor(Canvas, 200, yPos, 70, -1, -1,
        Plan.CalculateSisterTeamsAmAnfang(), GesamtKosten,
        cCalculateOptionsDisplayInteger
        [Plan.getOptions.SisterTeamsAmAnfangGewichtung], ForPrint);
      Action := TActionRectKostenPlan.Create(pktSisterGamesAtBegin);
      Action.setRect(200, yPos, 70, cActionHeight);
      Canvas.AddActionRect(Action);

    end;

    Inc(yPos, 40);

    VisibleMannschaftsKosten := 0;
    for AktType := Low(TMannschaftsKostenType)
      to High(TMannschaftsKostenType) do
    begin
      if Plan.HasValuesForType(AktType) then
      begin
        Inc(VisibleMannschaftsKosten);
      end;
    end;

    GesamtBreite := 100 * VisibleMannschaftsKosten + 440;

    PaintTableBackground(Canvas, yPos + 52, Plan.Mannschaften.Count, cRowHeight,
      GesamtBreite);

    Canvas.TextOut(0, yPos, 'Mannschaft');
    xPos := 250;

    for AktType := Low(TMannschaftsKostenType)
      to High(TMannschaftsKostenType) do
    begin
      if Plan.HasValuesForType(AktType) then
      begin
        Canvas.TextOut(xPos, yPos, cMannschaftsKostenTypeNamen[AktType]);
        Action := TActionRectKostenTypeMannschaft.Create(AktType);
        Action.setRect(xPos, yPos, cColWidth, cActionHeight);
        Canvas.AddActionRect(Action);

        Inc(xPos, cColWidth);
      end;
    end;
    Canvas.TextOut(xPos, yPos, 'Gesamt');

    Summen := TList<MyDouble>.Create;
    for AktType := Low(TMannschaftsKostenType)
      to High(TMannschaftsKostenType) do
    begin
      Summen.Add(0.0);
    end;
    Summen.Add(0.0);

    Inc(yPos, cRowHeight);
    for I := 0 to Plan.Mannschaften.Count - 1 do
    begin
      Kosten := TMannschaftsKosten.Create;
      Plan.Mannschaften[I].CalculateKosten(Plan, Kosten, tmNormalMessage);
      Canvas.TextOut(0, yPos, Plan.Mannschaften[I].teamName);
      Action := TActionRectKostenMannschaftMain.Create
        (Plan.Mannschaften[I].teamId);
      Action.setRect(0, yPos, 250, cActionHeight);
      Canvas.AddActionRect(Action);

      xPos := 250;
      for AktType := Low(TMannschaftsKostenType)
        to High(TMannschaftsKostenType) do
      begin
        if Plan.HasValuesForType(AktType) then
        begin
          Summen[Ord(AktType)] := Summen[Ord(AktType)] + Kosten.Values[AktType]
            .GesamtKosten;
          PaintKostenPaintWithBackColor(Canvas, xPos, yPos, 70,
            Kosten.Values[AktType].Anzahl, Kosten.Values[AktType].AllCount,
            Kosten.Values[AktType].GesamtKosten, GesamtKosten,
            Plan.GetMannschaftKostenDisplayInteger(Plan.Mannschaften[I],
            AktType), ForPrint);
          Action := TActionRectKostenMannschaftDetail.Create
            (Plan.Mannschaften[I].teamId, AktType);
          Action.setRect(xPos, yPos, cColWidth, cActionHeight);
          Canvas.AddActionRect(Action);

          Inc(xPos, cColWidth);
        end;
      end;
      Summen[Ord(High(TMannschaftsKostenType)) + 1] :=
        Summen[Ord(High(TMannschaftsKostenType)) + 1] + Kosten.getGesamtKosten;
      PaintKostenPaintWithBackColor(Canvas, xPos, yPos, 70, -1, -1,
        Kosten.getGesamtKosten, GesamtKosten, cCalculateOptionsDisplayInteger
        [Plan.getOptions.GewichtungMannschaften.getMain(Plan.Mannschaften[I]
        .teamId)], ForPrint);
      Action := TActionRectKostenMannschaftMain.Create
        (Plan.Mannschaften[I].teamId);
      Action.setRect(xPos, yPos, cColWidth, cActionHeight);
      Canvas.AddActionRect(Action);

      Inc(yPos, cRowHeight);
      Kosten.Free;
    end;

    // Summen ausgeben
    Inc(yPos, 5);
    Canvas.TextOut(0, yPos, 'Summe:');
    xPos := 250;
    for AktType := Low(TMannschaftsKostenType)
      to High(TMannschaftsKostenType) do
    begin
      if Plan.HasValuesForType(AktType) then
      begin
        PaintKostenPaintWithBackColor(Canvas, xPos, yPos, 70, -1, -1,
          Summen[Ord(AktType)], GesamtKosten, cCalculateOptionsDisplayInteger
          [Plan.getOptions().GewichtungMainMannschaftsKosten[AktType]],
          ForPrint);
        Action := TActionRectKostenTypeMannschaft.Create(AktType);
        Action.setRect(xPos, yPos, cColWidth, cActionHeight);
        Canvas.AddActionRect(Action);

        Inc(xPos, cColWidth);
      end;
    end;
    PaintKostenPaintWithBackColor(Canvas, xPos, yPos, 70, -1, -1,
      Summen[Ord(High(TMannschaftsKostenType)) + 1], GesamtKosten, 0, ForPrint);

    (*
      Inc(yPos, 40);
      Canvas.TextOut(50, yPos, '2 (10.000) bedeutet: 2 mal wurde es nicht erfllt, in Klammern die dadurch entstehenden Kosten');

      Inc(yPos, 20);
      Canvas.TextOut(50, yPos, '2 /8 (10.000) bedeutet: 2 von 8 Wnschen wurden nicht erfllt, in Klammern die dadurch entstehenden Kosten');
    *)

    Inc(yPos, 40);

    Summen.Free;

    // Messages ausgeben
    Inc(yPos, 20);
    for I := 0 to Plan.Mannschaften.Count - 1 do
    begin
      Kosten := TMannschaftsKosten.Create;
      Plan.Mannschaften[I].CalculateKosten(Plan, Kosten, tmNormalMessage);

      if Kosten.Messages.Count > 0 then
      begin
        Canvas.FontHeight := cFontBig;
        Canvas.TextOut(0, yPos, Plan.Mannschaften[I].teamName + ':');
        Canvas.FontHeight := cFontNormal;
        Inc(yPos, 30);
        for j := 0 to Kosten.Messages.Count - 1 do
        begin
          Canvas.TextOut(0, yPos, Kosten.Messages[j]);
          Inc(yPos, 20);
        end;
        Inc(yPos, 20);
      end;

      Kosten.Free;
    end;
  end;

  Result.cy := yPos;
  Result.cx := GesamtBreite + 10;

end;

function TFormPlanPanel.xPosByDate(Date: TDateTime;
  FirstDate: TDateTime): Integer;
begin
  Result := 300 + Trunc((Date - FirstDate) * 4);
end;

function TFormPlanPanel.PaintVerteilungMannschaft(Canvas: TZoomCanvas;
  PosY, PosYNext: Integer; Mannschaft: TMannschaft; NextMannschaft: TMannschaft;
  PaintOverlay: boolean): Integer;
var
  I: Integer;
  xPos, xPosNext: Integer;
  OffsetGameList, OffsetGameListNext: Integer;
  OldColor, OldBrushColor: TColor;
  Game: TGame;
  FirstDate: TDateTime;
  gameList, gameListNext: TGameList;
begin

  OldBrushColor := Canvas.BrushColor;
  Canvas.BrushColor := clWhite;
  Result := 0;
  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  gameList := TGameList.Create(False);
  gameListNext := TGameList.Create(False);

  Mannschaft.gamesRef.GetRefListe(gameList);
  gameList.SortByDate;

  if Assigned(NextMannschaft) then
  begin
    NextMannschaft.gamesRef.GetRefListe(gameListNext);
    gameListNext.SortByDate;
  end;

  // Die nicht terminierten Spiele berspringen
  OffsetGameList := 0;
  for I := 0 to gameList.Count - 1 do
  begin
    if gameList[I].DateValid then
      break
    else
      Inc(OffsetGameList);
  end;

  OffsetGameListNext := 0;
  for I := 0 to gameListNext.Count - 1 do
  begin
    if gameListNext[I].DateValid then
      break
    else
      Inc(OffsetGameListNext);
  end;

  FirstDate := Plan.GetFirstDate();

  OldColor := Canvas.PenColor;
  for I := 0 to gameList.Count - 1 do
  begin
    if I + OffsetGameList < gameList.Count then
    begin
      Game := gameList[I + OffsetGameList];

      xPos := xPosByDate(Game.Date, FirstDate);
      Result := Max(Result, xPos);

      if PaintOverlay then
      begin
        // Linie fr den Spieltag
        if gameListNext.Count > I + OffsetGameListNext then
        begin
          if I mod 2 = 0 then
          begin
            Canvas.PenColor := cColorGameDayLine1;
          end
          else
          begin
            Canvas.PenColor := cColorGameDayLine2;
          end;

          xPosNext := xPosByDate(gameListNext[I + OffsetGameListNext].Date,
            FirstDate);

          Canvas.MoveTo(xPos, PosY + 5);
          Canvas.LineTo(xPosNext, PosYNext + 5);
        end;
      end
      else
      begin
        // Die wichtigen Strich zeichnen -> ich bin nicht im Overlay
        Canvas.BrushColor := OldBrushColor;
        if Game.MannschaftHeim = Mannschaft then
          Canvas.BrushColor := cColorHomeGame
        else
          Canvas.BrushColor := cColorAwayGame;

        Canvas.FillRect(Rect(xPos - 1, PosY, xPos + 1, PosY + 10));
      end;
    end;
  end;
  Canvas.PenColor := OldColor;
  Canvas.BrushColor := OldBrushColor;

  gameList.Free;
  gameListNext.Free;
end;

procedure TFormPlanPanel.Print;
var
  Printer: TSimplePrinter;
  SelectDialog: TFormPrintSelect;
  NeedPage: boolean;
begin

  SelectDialog := TFormPrintSelect.Create(Self);

  if mrOK = SelectDialog.ShowModal then
  Begin

    Printer := TSimplePrinter.Create(Self);

    Printer.Font.Name := 'Tahoma';
    Printer.Font.Height := -12;

    if SelectDialog.ComboBoxFormat.ItemIndex = 1 then
    begin
      Printer.Zoom := 0.60;
    end
    else
    begin
      Printer.Zoom := 0.45;
    end;

    if Printer.BeginPrint(Plan.PlanName,
      (SelectDialog.ComboBoxFormat.ItemIndex = 1)) then
    begin
      NeedPage := False;
      if SelectDialog.CheckBoxTerminWuensche.Checked then
      begin
        Printer.Print(PaintTerminMeldung, NeedPage, 'Terminwnsche');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxSisterTeams.Checked then
      begin
        Printer.Print(PaintSisterTeams, NeedPage,
          'Termine der Nachbarmannschaften');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxKosten.Checked then
      begin
        Printer.Print(PaintKosten, NeedPage, 'Kosten');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxTerminPlan.Checked then
      begin
        Printer.Print(PaintSpielplan, NeedPage, 'Spielplan');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxMannschaftsplaene.Checked then
      begin
        Printer.Print(PaintMannschaftsPlaene, NeedPage, 'Mannschaftsplne');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxMannschaftsplaeneWithSisterTeams.Checked then
      begin
        Printer.Print(PaintMannschaftsPlaeneWithSisterTeams, NeedPage,
          'Mannschaftsplne mit Nachbarmanschaften');
        NeedPage := True;
      end;

      if SelectDialog.CheckBoxVerteilung.Checked then
      begin
        Printer.Print(PaintVerteilung, NeedPage, 'Diagramme');
      end;

      Printer.EndPrint;
    end;
    Printer.Free;
  End;
  SelectDialog.Free;
end;

procedure TFormPlanPanel.PaintDateLine(Canvas: TZoomCanvas; PosY: Integer);
var
  xPos1, xPos2: Integer;
  FirstDate, LastDate, ActDate: TDateTime;
  OldColor: TColor;
begin
  Canvas.TextOut(0, PosY, 'KW');

  FirstDate := Plan.GetFirstDate();
  LastDate := Plan.GetLastDate();

  ActDate := Trunc(FirstDate);

  while (DayOfWeek(ActDate) <> 2) do
  begin
    ActDate := ActDate - 1;
  end;

  while ActDate < LastDate do
  begin
    xPos1 := xPosByDate(ActDate, FirstDate);
    xPos2 := xPosByDate(ActDate + 7, FirstDate);

    OldColor := Canvas.PenColor;
    Canvas.PenColor := cColorLineGray;
    Canvas.MoveTo(xPos1, PosY + 0);
    Canvas.LineTo(xPos1, PosY + 15);
    Canvas.MoveTo(xPos2, PosY + 0);
    Canvas.LineTo(xPos2, PosY + 15);
    Canvas.PenColor := OldColor;

    Canvas.TextOut(xPos1 + 10, PosY, IntToStr(WeekOf(ActDate)));
    ActDate := ActDate + 7;
  end;
end;

procedure TFormPlanPanel.PaintTableBackground(Canvas: TZoomCanvas;
  PosY: Integer; Count: Integer; Height: Integer; width: Integer);
var
  OldColor: TColor;
  line, Y: Integer;
begin
  OldColor := Canvas.BrushColor;
  Canvas.BrushColor := cColorBrushGray;
  for line := 0 to Count - 1 do
  begin
    Y := PosY + line * Height;
    if line mod 2 = 0 then
    begin
      Canvas.BrushColor := cColorBrushGray;
      Canvas.FillRect(Rect(0, Y, width, Y + Height));
    end;
  end;
  Canvas.BrushColor := OldColor;
end;

procedure TFormPlanPanel.PaintDateRaster(Canvas: TZoomCanvas; PosY: Integer;
  Count: Integer; Height: Integer);
var
  xPos1, xPos2: Integer;
  FirstDate, LastDate, ActDate: TDateTime;
  OldColor: TColor;
  line, Y: Integer;
begin
  FirstDate := Plan.GetFirstDate();
  LastDate := Plan.GetLastDate();

  ActDate := Trunc(FirstDate);

  while (DayOfWeek(ActDate) <> 2) do
  begin
    ActDate := ActDate - 1;
  end;

  xPos2 := 0;

  while ActDate < LastDate do
  begin
    xPos2 := xPosByDate(ActDate + 7, FirstDate);
    ActDate := ActDate + 7;
  end;

  PaintTableBackground(Canvas, PosY, Count, Height, xPos2);

  for line := 0 to Count - 1 do
  begin

    ActDate := Trunc(FirstDate);

    while (DayOfWeek(ActDate) <> 2) do
    begin
      ActDate := ActDate - 1;
    end;

    while ActDate < LastDate do
    begin
      xPos1 := xPosByDate(ActDate, FirstDate);
      xPos2 := xPosByDate(ActDate + 7, FirstDate);

      Y := PosY + line * Height;

      OldColor := Canvas.PenColor;
      Canvas.PenColor := cColorLineGray;
      Canvas.MoveTo(xPos2, Y - 10);
      Canvas.LineTo(xPos2, Y + Height);
      Canvas.MoveTo(xPos1, Y - 10);
      Canvas.LineTo(xPos1, Y + Height);
      Canvas.PenColor := OldColor;
      ActDate := ActDate + 7;
    end;
  end;
end;

function TFormPlanPanel.PaintSpielPlanOverlap(Canvas: TZoomCanvas;
  Top, Height: Integer): TSize;
var
  I, j: Integer;
  SpielTageCount: Integer;
  MaxDateValues: TList<TDateTime>;
  MinDateValues: TList<TDateTime>;
  FirstDate, MaxDate, MinDate, ActDate: TDateTime;
  Abstand: Integer;
  Darkness: Integer;
  NewColor, OldColor: TColor;
  x1, x2: Integer;
  DrawRect: TRect;
begin

  FirstDate := Plan.GetFirstDate();
  SpielTageCount := 0;
  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    SpielTageCount := Max(SpielTageCount, Plan.Mannschaften[I].gamesRef.Count);
  end;

  MaxDateValues := TList<TDateTime>.Create;
  MinDateValues := TList<TDateTime>.Create;

  for I := 0 to SpielTageCount - 1 do
  begin
    MaxDate := 0.0;
    MinDate := 0.0;
    for j := 0 to Plan.Mannschaften.Count - 1 do
    begin
      ActDate := 0.0;
      if Plan.Mannschaften[j].gamesRef.Count > I then
      begin
        ActDate := Plan.Mannschaften[j].gamesRef[I].Date;
      end;

      if ActDate <> 0.0 then
      begin
        MaxDate := Max(MaxDate, ActDate);

        if MinDate = 0.0 then
        begin
          MinDate := ActDate;
        end
        else
        begin
          MinDate := Min(MinDate, ActDate);
        end;
      end;
    end;

    if (MaxDate > 0) and (MinDate > 0) then
    begin
      MaxDateValues.Add(MaxDate);
      MinDateValues.Add(MinDate);
    end;
  end;

  // Erst die nahen berlappungen malen und dann die starken drber
  for Abstand := 1 to MaxDateValues.Count - 1 do
  begin
    for I := 0 to MaxDateValues.Count - 1 do
    begin
      j := I + Abstand;
      if MinDateValues.Count > j then
      begin
        if Trunc(MaxDateValues[I]) >= Trunc(MinDateValues[j]) then
        begin
          // hier berlappt was
          Darkness := 255 - Min(128, Abstand * (128 div 5));
          NewColor := RGB(Darkness, Darkness, Darkness);
          OldColor := Canvas.BrushColor;
          Canvas.BrushColor := NewColor;

          x1 := xPosByDate(MinDateValues[j], FirstDate);
          x2 := xPosByDate(MaxDateValues[I], FirstDate);

          if x2 <= x1 then
          begin
            x2 := x1 + 1;
          end;

          DrawRect := Rect(x1, Top, x2, Top + Height);

          Canvas.FillRect(DrawRect);

          Canvas.BrushColor := OldColor;
        end;
      end;
    end;
  end;

  MaxDateValues.Free;
  MinDateValues.Free;

end;

function TFormPlanPanel.PaintSpielPlanVerteilung(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;
  NextMannschaft: TMannschaft;
  OldColor: TColor;
  yMannschaft: Integer;
const
  cYSizeMannschaft = 30;
begin

  xMax := 0;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Spieltage');
  Canvas.FontHeight := cFontNormal;

  Y := Y + 30;
  PaintDateLine(Canvas, Y);
  Y := Y + 5;

  PaintSpielPlanOverlap(Canvas, Y + cYSizeMannschaft - 5,
    ((Plan.Mannschaften.Count - 1) * cYSizeMannschaft) + 20);

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    yMannschaft := Y + ((I + 1) * cYSizeMannschaft);

    NextMannschaft := nil;

    if I + 1 < Plan.Mannschaften.Count then
    begin
      NextMannschaft := Plan.Mannschaften[I + 1]
    end;

    xMax := Max(xMax, PaintVerteilungMannschaft(Canvas, yMannschaft,
      yMannschaft + cYSizeMannschaft, Plan.Mannschaften[I],
      NextMannschaft, True));
  end;

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    yMannschaft := Y + ((I + 1) * cYSizeMannschaft);

    NextMannschaft := nil;

    if I + 1 < Plan.Mannschaften.Count then
    begin
      NextMannschaft := Plan.Mannschaften[I + 1]
    end;

    xMax := Max(xMax, PaintVerteilungMannschaft(Canvas, yMannschaft,
      yMannschaft + cYSizeMannschaft, Plan.Mannschaften[I],
      NextMannschaft, False));
  end;

  Y := Y + ((Plan.Mannschaften.Count) * cYSizeMannschaft);

  // Legende
  Y := Y + 35;
  OldColor := Canvas.BrushColor;
  Canvas.BrushColor := cColorHomeGame;
  Canvas.FillRect(Rect(5 - 1, Y, 5 + 1, Y + 10));
  Canvas.TextOut(10, Y, 'Heimspiel');

  Canvas.BrushColor := cColorAwayGame;
  Canvas.FillRect(Rect(105 - 1, Y, 105 + 1, Y + 10));
  Canvas.TextOut(110, Y, 'Auswrtsspiel');

  Canvas.TextOut(210, Y, 'berlappungen der Spieltage sind grau hinterlegt');

  Canvas.BrushColor := OldColor;

  Result.cy := Y + 20;
  Result.cx := xMax + 100;
end;

const
  AmplitudeY = 10;
  AmplitudeX = 10;

function TFormPlanPanel.PaintWechselHeimAuswaertsMannschaft(Canvas: TZoomCanvas;
  PosY: Integer; Mannschaft: TMannschaft): Integer;
var
  I: Integer;
  xPos: Integer;
  Game: TGame;
  gameList: TGameList;
  OldColor: TColor;
  Y, OldY: Integer;
begin
  OldY := -1;
  Result := 0;

  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  gameList := TGameList.Create(False);

  Mannschaft.gamesRef.GetRefListe(gameList);
  gameList.SortByDate;

  // Die nicht terminierten Spiele berspringen
  for I := gameList.Count - 1 downto 0 do
  begin
    if not gameList[I].DateValid then
    begin
      gameList.Delete(I);
    end;
  end;

  xPos := 300;

  OldColor := Canvas.PenColor;

  for I := 0 to gameList.Count - 1 do
  begin
    Game := gameList[I];

    if Game.MannschaftHeim = Mannschaft then
    begin
      Y := PosY + AmplitudeY;
    end
    else
    begin
      Y := PosY;
    end;

    if OldY > 0 then
    begin
      Canvas.PenColor := OldColor;

      Canvas.MoveTo(xPos, OldY);
      Canvas.LineTo(xPos + AmplitudeX, Y);
    end;

    OldY := Y;

    xPos := xPos + AmplitudeX;
    Result := Max(Result, xPos);
  end;
  gameList.Free;

end;

function TFormPlanPanel.PaintSpielRankingMannschaft(Canvas: TZoomCanvas;
  PosY: Integer; Mannschaft: TMannschaft): Integer;
var
  I: Integer;
  xPos: Integer;
  Game: TGame;
  gameList: TGameList;
  OldColor: TColor;
  Y: Integer;
  Values: TList<Integer>;
  currRound: Integer;
  ZielValue: Integer;
  OldZielValue: Integer;
begin
  Result := 0;

  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  gameList := TGameList.Create(False);
  Values := TList<Integer>.Create;

  Mannschaft.gamesRef.GetRefListe(gameList);
  gameList.SortByDate;

  Mannschaft.getRankingDiffs(Plan, Values);

  OldZielValue := 0;
  ZielValue := Plan.Mannschaften.Count - 1;
  xPos := 300;
  currRound := 0;
  for I := 0 to gameList.Count - 1 do
  begin
    Game := gameList[I];

    if currRound < Plan.GetRoundNumberByDate(Game.Date) then
    begin
      currRound := Plan.GetRoundNumberByDate(Game.Date);
      ZielValue := Plan.Mannschaften.Count - 1;
      Inc(xPos, 50);
    end;

    if OldZielValue > ZielValue then
    begin
      OldColor := Canvas.PenColor;
      Canvas.PenColor := clSilver;
      Canvas.MoveTo(xPos - AmplitudeX - 3, PosY + 15 - (OldZielValue * 5));
      Canvas.LineTo(xPos + AmplitudeX - 3,
        PosY + 15 - ((OldZielValue - 2) * 5));
      Canvas.PenColor := OldColor;
    end;
    OldZielValue := ZielValue;

    xPos := xPos + AmplitudeX;
    Result := Max(Result, xPos);
    Dec(ZielValue);
  end;

  ZielValue := Plan.Mannschaften.Count - 1;
  xPos := 300;
  currRound := 0;
  for I := 0 to gameList.Count - 1 do
  begin
    Game := gameList[I];

    if currRound < Plan.GetRoundNumberByDate(Game.Date) then
    begin
      currRound := Plan.GetRoundNumberByDate(Game.Date);
      ZielValue := Plan.Mannschaften.Count - 1;
      Inc(xPos, 50);
    end;

    Y := Values[I] * 5;

    OldColor := Canvas.BrushColor;
    Canvas.BrushColor := clGreen;
    if Values[I] > ZielValue then
    begin
      Canvas.BrushColor := clYellow;
    end;

    if Values[I] > ZielValue * 2 then
    begin
      Canvas.BrushColor := RGB(255, 153, 102);
    end;

    if Values[I] > ZielValue * 5 then
    begin
      Canvas.BrushColor := clRed;
    end;

    Canvas.FillRect(Rect(xPos, PosY + 20 - Y, xPos + (AmplitudeX div 2),
      PosY + 20));

    Canvas.BrushColor := OldColor;

    xPos := xPos + AmplitudeX;
    Result := Max(Result, xPos);
    Dec(ZielValue);
  end;

  gameList.Free;
  Values.Free;

end;

function TFormPlanPanel.PaintSpielRanking(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;

begin

  xMax := 0;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Setzliste');
  Canvas.FontHeight := cFontNormal;

  Y := Y + 10;

  width := 300 + (AmplitudeX * Plan.getMaxGamePerMannschaft()) + 50 *
    Plan.Rounds.Count;

  PaintTableBackground(Canvas, Y + 22, Plan.Mannschaften.Count, 60, width + 10);

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    Y := Y + 60;

    xMax := Max(xMax, PaintSpielRankingMannschaft(Canvas, Y,
      Plan.Mannschaften[I]));
  end;

  Result.cy := Y + 20;
  Result.cx := xMax + 100;
end;

function TFormPlanPanel.PaintSpielWechselHeimAuswaerts(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;

begin

  xMax := 0;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Wechsel Heim/Auswrts');
  Canvas.FontHeight := cFontNormal;

  Y := Y + 10;

  width := 300 + (AmplitudeX * Plan.getMaxGamePerMannschaft());

  PaintTableBackground(Canvas, Y + 22, Plan.Mannschaften.Count, 30, width + 10);

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    Y := Y + 30;

    xMax := Max(xMax, PaintWechselHeimAuswaertsMannschaft(Canvas, Y,
      Plan.Mannschaften[I]));
  end;

  Result.cy := Y + 20;
  Result.cx := xMax + 100;
end;

procedure TFormPlanPanel.PaintGameBubble(Canvas: TZoomCanvas;
  PosX, PosY: Integer; Game: TGame; HeimSpiel: boolean; KoppelTermin: boolean);
var
  OldBrushColor: TColor;
  OldPenColor: TColor;
  Color: TColor;
  TextPos: Integer;
  LinePos: Integer;
  S: String;
  Size: TSize;
begin

  S := MyFormatDateShort(Game.Date);
  Size := Canvas.TextExtent(S);

  TextPos := PosY - 12;
  LinePos := PosY - 10;

  Canvas.MoveTo(PosX, PosY);
  Canvas.LineTo(PosX, LinePos);

  Canvas.FontRotation := -90;
  Canvas.TextOut(PosX - (Size.cy div 2) - 2, TextPos, S);
  Canvas.FontRotation := 0;

  OldPenColor := Canvas.PenColor;
  OldBrushColor := Canvas.BrushColor;

  if HeimSpiel then
  begin
    Color := cColorHomeGameBubble;
  end
  else
  begin
    Color := cColorAwayGameBubble;
  end;

  // Canvas.PenColor := Color;
  Canvas.BrushColor := Color;

  if not KoppelTermin then
  begin
    Canvas.DrawArc(PosX, PosY, 5);
  end
  else
  begin
    Canvas.DrawQuadrat(PosX, PosY, 5);
  end;

  Canvas.PenColor := OldPenColor;
  Canvas.BrushColor := OldBrushColor;
end;

function TFormPlanPanel.PaintSpielAbstaendeMannschaft(Canvas: TZoomCanvas;
  PosY: Integer; Mannschaft: TMannschaft): Integer;
var
  I: Integer;
  xPos: Integer;
  Game: TGame;
  FirstDate: TDateTime;
  isKoppel: boolean;
begin

  Result := 0;
  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  FirstDate := Plan.GetFirstDate();

  for I := 0 to Mannschaft.gamesRef.Count - 1 do
  begin
    Game := Mannschaft.gamesRef[I];

    if Game.DateValid then
    begin
      xPos := xPosByDate(Game.Date, FirstDate);
      Result := Max(Result, xPos);

      isKoppel := Assigned(Mannschaft.getKoppeledGame(I));

      PaintGameBubble(Canvas, xPos, PosY, Game,
        Game.MannschaftHeim = Mannschaft, isKoppel);
    end;
  end;
end;

function TFormPlanPanel.PaintSpielAbstaende(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;
begin

  xMax := 0;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Spielverteilung');
  Canvas.FontHeight := cFontNormal;

  Y := Y + 30;

  PaintDateLine(Canvas, Y);

  PaintDateRaster(Canvas, Y + 20, Plan.Mannschaften.Count, 80);

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    Y := Y + 80;

    xMax := Max(xMax, PaintSpielAbstaendeMannschaft(Canvas, Y,
      Plan.Mannschaften[I]));
  end;

  if Plan.HasKoppelTermine or Plan.HasAuswaertsKoppelTermine then
  begin
    Y := Y + 30;
    Canvas.TextOut(0, Y,
      'Koppeltermine/Doppelspieltage sind als Quadrate dargestellt  rot=Heimspiel schwarz=Auswrtsspiel');
  end;

  Result.cy := Y + 20;
  Result.cx := xMax + 100;
end;

function TFormPlanPanel.PaintMeldungMannschaft(Canvas: TZoomCanvas;
  PosY: Integer; Mannschaft: TMannschaft): Integer;
var
  I: Integer;
  xPos: Integer;
  OldColor: TColor;
  FirstDate: TDateTime;
begin
  Result := 0;
  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  FirstDate := Plan.GetFirstDate();

  OldColor := Canvas.BrushColor;
  for I := 0 to Mannschaft.WunschTermine.Count - 1 do
  begin
    if not(woAuswaertsKoppelSecondTime in Mannschaft.WunschTermine[I].Options)
    then
    begin
      xPos := xPosByDate(Trunc(Mannschaft.WunschTermine[I].Date) +
        EncodeTime(12, 0, 0, 0), FirstDate);
      Result := Max(Result, xPos);

      if OptionIsKoppel(Mannschaft.WunschTermine[I].Options) then
        Canvas.BrushColor := cColorKoppelWunsch
      else
        Canvas.BrushColor := cColorNormalWunsch;

      Canvas.FillRect(Rect(xPos - 1, PosY, xPos + 1, PosY + 10));
      // Canvas.MoveTo(xPos, PosY);
      // Canvas.LineTo(xPos, PosY+10);

      Result := Max(Result, xPos);
    end;
  end;

  for I := 0 to Mannschaft.SperrTermine.Count - 1 do
  begin
    xPos := xPosByDate(Mannschaft.SperrTermine[I], FirstDate);
    Result := Max(Result, xPos);

    Canvas.BrushColor := cColorSperrWunsch;

    Canvas.FillRect(Rect(xPos - 1, PosY, xPos + 1, PosY + 10));
    // Canvas.MoveTo(xPos, PosY);
    // Canvas.LineTo(xPos, PosY+10);

    Result := Max(Result, xPos);
  end;

  Canvas.BrushColor := OldColor;

end;

function TFormPlanPanel.PaintSpielPlanMeldungen(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;
  OldColor: TColor;
begin

  xMax := 0;

  // Canvas.TextOut(0, y, 'Terminwnsche');
  // y := y + 20;

  PaintDateLine(Canvas, Y);
  Y := Y + 5;

  PaintDateRaster(Canvas, Y + 20, Plan.Mannschaften.Count, 30);
  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    Y := Y + 30;

    xMax := Max(xMax, PaintMeldungMannschaft(Canvas, Y, Plan.Mannschaften[I]));
  end;

  // Legende
  Y := Y + 35;
  OldColor := Canvas.BrushColor;
  Canvas.BrushColor := cColorNormalWunsch;
  Canvas.FillRect(Rect(5 - 1, Y, 5 + 1, Y + 10));
  Canvas.TextOut(10, Y, 'Terminwunsch');

  Canvas.BrushColor := cColorSperrWunsch;
  Canvas.FillRect(Rect(105 - 1, Y, 105 + 1, Y + 10));
  Canvas.TextOut(110, Y, 'Sperrtermin');

  if Plan.HasKoppelTermine then
  begin
    Canvas.BrushColor := cColorKoppelWunsch;
    Canvas.FillRect(Rect(205 - 1, Y, 205 + 1, Y + 10));
    Canvas.TextOut(210, Y, 'Koppeltermin/Doppelspieltag');
  end;

  Canvas.BrushColor := OldColor;

  Inc(Y, 10);
  if not Plan.Has60KilometerValues then
  begin
    Inc(Y, 15);
    Canvas.TextOut(0, Y, 'Keine Wnsche fr die 60km-Regel');
  end;
  if not Plan.HasKoppelTermine then
  begin
    Inc(Y, 15);
    Canvas.TextOut(0, Y, 'Keine Koppeltermine/Doppelspieltage gemeldet');
  end;
  if not Plan.HasAuswaertsKoppelTermine then
  begin
    Inc(Y, 15);
    Canvas.TextOut(0, Y, 'Keine Auswrtskoppelwnsche gemeldet');
  end;

  Result.cy := Y;
  Result.cx := xMax + 100;

end;

function TFormPlanPanel.PaintSpielAbstandMannschaft(Canvas: TZoomCanvas;
  PosY: Integer; Mannschaft: TMannschaft): TSize;
var
  index, k, j, ColorValueR, ColorValueG: Integer;
  Abstand: Integer;
  Value: MyDouble;
  xPos, xPosNext: Integer;
  OldColor: TColor;
  Game, gameNext: TGame;
  FirstDate: TDateTime;
  gameList: TGameList;
  TeamsDone: TList<String>;
  currTeamId, nextTeamId: String;
begin
  Result.cx := 0;
  Result.cy := 20;
  Canvas.TextOut(0, PosY, Mannschaft.teamName);

  gameList := TGameList.Create(False);
  Mannschaft.gamesRef.GetRefListe(gameList);
  gameList.SortByDate;

  FirstDate := Plan.GetFirstDate();

  TeamsDone := TList<String>.Create;

  OldColor := Canvas.PenColor;
  for index := 0 to gameList.Count - 1 do
  begin
    Game := gameList[index];

    if Game.MannschaftHeim = Mannschaft then
    begin
      currTeamId := Game.MannschaftGast.teamId;
    end
    else
    begin
      currTeamId := Game.MannschaftHeim.teamId;
    end;

    if TeamsDone.Contains(currTeamId) then
    begin
      continue;
    end;

    if Game.DateValid then
    begin
      TeamsDone.Add(currTeamId);

      k := Index;

      while True do
      begin
        gameNext := nil;

        Abstand := 0;

        for j := k + 1 to gameList.Count - 1 do
        begin
          Inc(Abstand);
          gameNext := gameList[j];

          if gameNext.MannschaftHeim = Mannschaft then
          begin
            nextTeamId := gameNext.MannschaftGast.teamId;
          end
          else
          begin
            nextTeamId := gameNext.MannschaftHeim.teamId;
          end;

          if nextTeamId = currTeamId then
          begin
            break;
          end;
          gameNext := nil;
        end;

        if gameNext <> nil then
        begin
          // Farbe berechnen, Standardabstand grn, sonst rot
          // value = 1.0 -> maximal schlecht, value = 0.0 ideal
          Value := Abs(((Abstand * 1.0) - (gameList.Count / Plan.Rounds.Count))
            / (gameList.Count / Plan.Rounds.Count));

          if (Abstand * 1.0) > (gameList.Count / Plan.Rounds.Count) then
          begin
            // Spiele sind weiter auseinander als gefordert, ist nicht so schlimm
            // aus 0.6 wird 0.48
            Value := Value / 1.25;
          end;

          ColorValueG := Max(0, 255 - Trunc(Value * 255));
          ColorValueR := 255 - ColorValueG;

          if ColorValueR < ColorValueG then
            ColorValueG := 255
          else
            ColorValueR := 255;

          // ColorValueG := Min(255, ColorValueG + 128);
          // ColorValueR := Min(255, ColorValueR + 128);

          Canvas.PenColor := RGB(ColorValueR, ColorValueG, 0);
          xPos := xPosByDate(Game.Date, FirstDate);
          xPosNext := xPosByDate(gameNext.Date, FirstDate);

          Result.cx := Max(Result.cx, xPosNext + 100);

          Canvas.MoveTo(xPos, PosY);
          Canvas.LineTo(xPosNext, PosY);

          Canvas.PenColor := OldColor;
          (*
            Canvas.MoveTo(xPos, PosY-1);
            Canvas.LineTo(xPos, PosY+2);

            Canvas.MoveTo(xPosNext, PosY-1);
            Canvas.LineTo(xPosNext, PosY+2);
          *)
          Canvas.MoveTo(xPos, PosY - 2);
          Canvas.LineTo(xPos, PosY + 3);

          Canvas.MoveTo(xPosNext, PosY - 2);
          Canvas.LineTo(xPosNext, PosY + 3);

        end;

        if gameNext = nil then
        begin
          break;
        end;

        Game := gameNext;
        k := j;

      end;

      // Nchste Mannschaft
      PosY := PosY + 5;
    end;
  end;
  Canvas.PenColor := OldColor;

  gameList.Free;

  TeamsDone.Free;

  Result.cy := Max(Result.cy, PosY - 5);

end;

function TFormPlanPanel.PaintSpielAbstandHinRueck(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  I, xMax: Integer;
  Size: TSize;
begin

  xMax := 0;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Abstand Heimspiel zu Auswrtsspiel');
  Canvas.FontHeight := cFontNormal;

  Y := Y + 30;
  PaintDateLine(Canvas, Y);
  Y := Y + 25;

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin

    Size := PaintSpielAbstandMannschaft(Canvas, Y, Plan.Mannschaften[I]);

    Y := Size.cy + 10;
    xMax := Max(xMax, Size.cx);
  end;

  Result.cy := Y - 10;
  Result.cx := xMax + 100;

end;

function TFormPlanPanel.getXBySpieltageMondays(SpieltageMondays
  : TList<TDateTime>; index: Integer): Integer;
begin
  Result := 300;
  Result := Result + index * 70;
end;

function TFormPlanPanel.PaintWeekValues(Canvas: TZoomCanvas; Y: Integer;
  SpieltageMondays: TList<TDateTime>; Title: String; Values: TList<String>;
  ValuesPrintRed: TList<boolean>): TSize;
var
  I, xMax, X: Integer;
  OldColor: TColor;
begin
  Result.cx := 0;
  Result.cy := 20;
  Canvas.TextOut(0, Y, Title);

  xMax := 100;

  for I := 0 to Values.Count - 1 do
  begin
    X := getXBySpieltageMondays(SpieltageMondays, I);
    OldColor := Canvas.FontColor;
    if (Assigned(ValuesPrintRed)) and (ValuesPrintRed.Count > I) and
      (ValuesPrintRed[I]) then
    begin
      Canvas.FontColor := clRed;
    end;
    Canvas.TextOut(X, Y, Values[I]);
    Canvas.FontColor := OldColor;
    xMax := Max(xMax, X);
  end;

  Result.cx := xMax + 100;

end;

procedure TFormPlanPanel.PaintGamesPerWeekHeader(Canvas: TZoomCanvas;
  Y: Integer; SpieltageMondays: TList<TDateTime>);
var
  I, X: Integer;
  Year, Month, Day: Word;
begin
  Canvas.TextOut(0, Y + 5, 'Spieltag');
  Canvas.TextOut(0, Y + 45, '(Spiele die Woche / Spiele Gesamt)');
  for I := 0 to SpieltageMondays.Count - 2 do
  begin
    X := getXBySpieltageMondays(SpieltageMondays, I);
    DecodeDate(SpieltageMondays[I], Year, Month, Day);
    Canvas.FontHeight := cFontBig;
    Canvas.TextOut(X, Y, IntToStr(I + 1));
    Canvas.FontHeight := cFontNormal;
    Canvas.TextOut(X, Y + 25, MyFormatDateShort(SpieltageMondays[I]));
    Canvas.TextOut(X, Y + 40,
      '- ' + MyFormatDateShort(SpieltageMondays[I] + 6));
  end;
end;

function TFormPlanPanel.PaintGamesPerWeekOneRound(Canvas: TZoomCanvas;
  Y: Integer; SpieltageMondays: TList<TDateTime>): TSize;
var
  I, j, xMax: Integer;
  Value, minValue, maxValue: Integer;
  Size: TSize;
  Values: TList<String>;
  ValuesRedColor: TList<boolean>;
  FromDate, ToDate: TDateTime;
  width: Integer;
  AllCount, currentCount: Integer;
  OldColor: TColor;
begin

  xMax := 0;

  Y := Y + 30;
  PaintGamesPerWeekHeader(Canvas, Y, SpieltageMondays);
  Y := Y + 75;

  width := getXBySpieltageMondays(SpieltageMondays, SpieltageMondays.Count - 1);
  PaintTableBackground(Canvas, Y - 8, Plan.Mannschaften.Count, 30, width);

  for I := 0 to Plan.Mannschaften.Count - 1 do
  begin
    Values := TList<String>.Create;
    AllCount := 0;

    for j := 0 to SpieltageMondays.Count - 2 do
    begin
      FromDate := SpieltageMondays[j];
      ToDate := SpieltageMondays[j + 1] - EncodeTime(0, 0, 1, 0);
      currentCount := Plan.Mannschaften[I].getGameCount(FromDate, ToDate);
      AllCount := AllCount + currentCount;
      Values.Add(IntToStr(currentCount) + ' / ' + IntToStr(AllCount));
    end;

    Size := PaintWeekValues(Canvas, Y, SpieltageMondays,
      Plan.Mannschaften[I].teamName, Values, nil);

    Values.Free;
    Y := Y + 30;
    xMax := Max(xMax, Size.cx);
  end;

  Values := TList<String>.Create;
  ValuesRedColor := TList<boolean>.Create;
  for j := 0 to SpieltageMondays.Count - 2 do
  begin
    FromDate := SpieltageMondays[0];
    ToDate := SpieltageMondays[j + 1] - EncodeTime(0, 0, 1, 0);
    minValue := -1;
    maxValue := -1;
    for I := 0 to Plan.Mannschaften.Count - 1 do
    begin
      Value := Plan.Mannschaften[I].getGameCount(FromDate, ToDate);
      if I = 0 then
      begin
        maxValue := Value;
        minValue := Value;
      end;
      maxValue := Max(maxValue, Value);
      minValue := Min(minValue, Value);
    end;
    currentCount := Abs(maxValue - minValue);
    Values.Add(IntToStr(currentCount));
    ValuesRedColor.Add(currentCount >= 3);
  end;

  OldColor := Canvas.PenColor;
  Canvas.PenColor := cColorLineGray;
  Canvas.MoveTo(0, Y);
  Canvas.LineTo(width, Y);
  Canvas.PenColor := OldColor;

  Y := Y + 10;
  Size := PaintWeekValues(Canvas, Y, SpieltageMondays,
    'maximale Differenz absolvierter Spiele', Values, ValuesRedColor);

  Values.Free;
  ValuesRedColor.Free;

  Y := Y + 30;
  xMax := Max(xMax, Size.cx);

  Result.cx := xMax + 100;
  Result.cy := Y;

end;

function TFormPlanPanel.PaintGamesPerWeek(Canvas: TZoomCanvas;
  Y: Integer): TSize;
var
  xMax: Integer;
  Size: TSize;
  SpieltageMondays: TList<TDateTime>;
  DateTo: TDateTime;
begin
  xMax := 0;

  Y := Y + 50;

  Canvas.FontHeight := cFontBig;
  Canvas.TextOut(0, Y, 'Anzahl Spiele pro Woche');

  Y := Y + 50;
  Canvas.TextOut(0, Y, 'Hinrunde');
  Canvas.FontHeight := cFontNormal;

  SpieltageMondays := TList<TDateTime>.Create;
  if Plan.getOptions.RoundPlaning = rpHalfRound then
  begin
    DateTo := Plan.DateEnd;
  end
  else
  begin
    DateTo := Plan.DateBeginRueckrundeInXML;
  end;

  Plan.getSpieltagMondays(SpieltageMondays, 0, DateTo -
    EncodeTime(0, 0, 1, 0));
  SpieltageMondays.Add(DateTo - EncodeTime(0, 0, 1, 0));

  Size := PaintGamesPerWeekOneRound(Canvas, Y, SpieltageMondays);

  Y := Size.cy;
  xMax := Max(xMax, Size.cx);

  if not(Plan.getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpCorona]) then
  begin
    Y := Y + 30;
    Canvas.FontHeight := cFontBig;
    Canvas.TextOut(0, Y, 'Rckrunde');
    Canvas.FontHeight := cFontNormal;

    Plan.getSpieltagMondays(SpieltageMondays, Plan.DateBeginRueckrundeInXML,
      EncodeDate(2500, 1, 1));
    SpieltageMondays.Add(EncodeDate(2500, 1, 1));

    Size := PaintGamesPerWeekOneRound(Canvas, Y, SpieltageMondays);

    Y := Size.cy;
    xMax := Max(xMax, Size.cx);
  end;

  Result.cx := xMax + 100;
  Result.cy := Y + 40;

  SpieltageMondays.Free;

end;

function TFormPlanPanel.PaintVerteilung(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
var
  yPos: Integer;
  xMax: Integer;
  Size: TSize;
begin
  xMax := 500;

  yPos := 0;

  if Plan.NoGames() then
  begin
    Canvas.TextOut(0, 0, 'Keine Spiele terminiert');
  end
  else
  begin

    yPos := 0;

    Size := PaintSpielPlanVerteilung(Canvas, yPos);

    yPos := Size.cy + 40;
    xMax := Max(xMax, Size.cx);

    Size := PaintSpielWechselHeimAuswaerts(Canvas, yPos);

    yPos := Size.cy + 40;
    xMax := Max(xMax, Size.cx);

    Size := PaintSpielAbstaende(Canvas, yPos);

    if not(Plan.getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpCorona]) then
    begin
      yPos := Size.cy + 40;
      xMax := Max(xMax, Size.cx);

      Size := PaintSpielAbstandHinRueck(Canvas, yPos);
    end;

    yPos := Size.cy + 40;
    xMax := Max(xMax, Size.cx);

    Size := PaintGamesPerWeek(Canvas, yPos);

    if Plan.HasRanking then
    begin
      yPos := Size.cy + 40;
      xMax := Max(xMax, Size.cx);

      Size := PaintSpielRanking(Canvas, yPos);
    end;

    yPos := Size.cy + 40;
    xMax := Max(xMax, Size.cx);

  end;

  Result.cy := yPos - 40;
  Result.cx := xMax + 100;
end;

procedure TFormPlanPanel.ClearActionRects(PaintBox: TObject);
var
  I: Integer;
begin
  for I := ActionRects.Count - 1 downto 0 do
  begin
    if ActionRects[I].getPaintBoxOwner() = PaintBox then
    begin
      ActionRects.Delete(I);
    end;
  end;
end;

procedure TFormPlanPanel.AddActionRects(PaintBox: TObject;
  Values: TList<TActionRect>; Canvas: TZoomCanvas);
var
  ActionRect: TActionRect;
  Rect: TRect;
begin
  for ActionRect in Values do
  begin
    ActionRect.setPaintBoxOwner(PaintBox);
    ActionRects.Add(ActionRect);
    Rect := ActionRect.getRect();
    Rect.Left := Canvas.ConvertXCoord(Rect.Left);
    Rect.Right := Canvas.ConvertXCoord(Rect.Right);
    Rect.Top := Canvas.ConvertYCoord(Rect.Top);
    Rect.Bottom := Canvas.ConvertYCoord(Rect.Bottom);
    ActionRect.setRect(Rect.Left, Rect.Top, Rect.width, Rect.Height);
  end;
end;

procedure TFormPlanPanel.PaintBoxPaint(Sender: TObject);
var
  Canvas: TZoomCanvas;
  PaintBox: TPaintBox;
  RetSize: TSize;
begin
  PaintBox := Sender as TPaintBox;
  ClearActionRects(PaintBox);
  Canvas := TZoomCanvas.Create(PaintBox.Canvas, FZoomValue);
  Canvas.BrushColor := clWhite;
  Canvas.FillRect(Rect(0, 0, PaintBox.width, PaintBox.Height));
  RetSize.cx := 0;
  RetSize.cy := 0;

  if PaintBox = PaintBoxTerminMeldung then
    RetSize := PaintTerminMeldung(Canvas, False);
  if PaintBox = PaintBoxSisterTeams then
    RetSize := PaintSisterTeams(Canvas, False);
  if PaintBox = PaintBoxKosten then
    RetSize := PaintKosten(Canvas, False);
  if PaintBox = PaintBoxTermine then
    RetSize := PaintTermineMain(Canvas, False);
  if PaintBox = PaintBoxVerteilung then
    RetSize := PaintVerteilung(Canvas, False);

  PaintBox.Height := Max(100, Canvas.ConvertYCoord(RetSize.cy) + 40);
  PaintBox.width := Max(500, Canvas.ConvertXCoord(RetSize.cx) + 40);

  AddActionRects(PaintBox, Canvas.ActionRects, Canvas);

  Canvas.Free;
end;

procedure TFormPlanPanel.PaintBoxMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  ActionRect: TActionRect;
begin
  for ActionRect in ActionRects do
  begin
    if ActionRect.getPaintBoxOwner() = Sender then
    begin
      if PtInRect(ActionRect.getRect, Point(X, Y)) then
      begin
        ActionRect.DoAction(Point(X, Y));
      end;
    end;
  end;
end;

procedure TFormPlanPanel.PaintBoxMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  Cursor: TCursor;
  ActionRect: TActionRect;
begin
  Cursor := crDefault;
  for ActionRect in ActionRects do
  begin

    if ActionRect.getPaintBoxOwner() = Sender then
    begin
      if PtInRect(ActionRect.getRect, Point(X, Y)) then
      begin
        Cursor := crHandPoint;
      end;
    end;
  end;

  if Sender is TPaintBox then
  begin
    (Sender as TPaintBox).Cursor := Cursor;
  end;
end;

function TFormPlanPanel.PaintTermineMain(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
begin
  if ToolButtonSpielplan.Down then
    Result := PaintSpielplan(Canvas, ForPrint)
  else if ToolButtonMannschaftsplaene.Down then
    Result := PaintMannschaftsPlaene(Canvas, ForPrint)
  else
    Result := PaintMannschaftsPlaeneWithSisterTeams(Canvas, ForPrint);

end;

function TFormPlanPanel.PaintTerminMeldung(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
var
  I, j: Integer;
  yPos: Integer;
  Mannschaft: TMannschaft;
  S: String;
  SperrTerminCount: Integer;
  BelegungString: String;
  BelegungNum: Integer;
  NormalColor: TColor;
  RedColor, YellowColor, GreenColor: TColor;
  TermineVorrunde: Integer;
  TermineVorrundeBelegt: Integer;
  TermineVorrundeParallel: Integer;
  TermineRueckrunde: Integer;
  TermineRueckrundeBelegt: Integer;
  TermineRueckrundeParallel: Integer;
  TermineSpielFreiVorrunde: Integer;
  TermineSpielFreiRueckRunde: Integer;

  IsVorrunde: boolean;
  EffectTerminValue: Integer;
  MinTerminePerRound: MyDouble;
  MessageList: TList<String>;
  xMax: Integer;
  Size: TSize;
begin

  yPos := 0;
  xMax := 0;

  if Plan.Mannschaften.Count = 0 then
  begin
    Canvas.TextOut(0, 0, 'Kein Plan geladen');
    Inc(yPos, 40);
  end
  else
  begin
    if not ForPrint then
    begin
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, 0, 'Terminwnsche:');
      Inc(yPos, 30);
      Canvas.FontHeight := cFontNormal;
    end;

    Size := PaintSpielPlanMeldungen(Canvas, yPos);

    yPos := Size.cy + 40;
    xMax := Size.cx;

    NormalColor := Canvas.FontColor;
    RedColor := RGB(255, 0, 0);
    YellowColor := RGB(255, 165, 0);
    GreenColor := RGB(60, 179, 113);

    for I := 0 to Plan.Mannschaften.Count - 1 do
    begin
      TermineVorrunde := 0;
      TermineVorrundeBelegt := 0;
      TermineVorrundeParallel := 0;
      TermineRueckrunde := 0;
      TermineRueckrundeBelegt := 0;
      TermineRueckrundeParallel := 0;
      TermineSpielFreiVorrunde := 0;
      TermineSpielFreiRueckRunde := 0;

      Mannschaft := Plan.Mannschaften[I];

      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, yPos, Mannschaft.teamName + ':');
      Canvas.FontHeight := cFontNormal;
      Inc(yPos, 10);

      for j := 0 to Mannschaft.WunschTermine.Count - 1 do
      begin
        if not(woAuswaertsKoppelSecondTime in Mannschaft.WunschTermine[j]
          .Options) then
        begin

          Inc(yPos, 20);

          S := MyFormatDateTime(Mannschaft.WunschTermine[j].Date);

          if Plan.getOptions.RoundPlaning = rpHalfRound then
          begin
            IsVorrunde := true;
          end
          else
          begin
            IsVorrunde := Plan.DateBeginRueckrundeInXML >
              Mannschaft.WunschTermine[j].Date;
          end;

          if IsVorrunde then
            Inc(TermineVorrunde)
          else
            Inc(TermineRueckrunde);

          if OptionIsKoppel(Mannschaft.WunschTermine[j].Options) then
          begin
            S := S + ', ' + cWunschterminOptionName
              [KoppelValueFromOptions(Mannschaft.WunschTermine[j].Options)];
          end;

          if Mannschaft.WunschTermine[j].ParallelGames > 0 then
          begin
            S := S + ', maximale Heimspiele: ' +
              IntToStr(Mannschaft.WunschTermine[j].ParallelGames);
          end;

          if Mannschaft.isAusWeichTermin(Mannschaft.WunschTermine[j].Date) then
          begin
            S := S + ', Ausweichtermin';
          end;

          if Mannschaft.WunschTermine[j].Location <> '' then
          begin
            S := S + ', Spiellokal: ' + Mannschaft.WunschTermine[j].Location;
          end;

          Mannschaft.GetHallenBelegung(Mannschaft.WunschTermine[j].Date,
            Mannschaft.WunschTermine[j].Location, Plan, BelegungNum,
            BelegungString, True, True);

          if BelegungNum > 0 then
          begin
            if (Mannschaft.WunschTermine[j].ParallelGames > 0) and
              (Mannschaft.WunschTermine[j].ParallelGames <= BelegungNum) then
            begin
              S := S + ', Halle ist belegt von: ' + BelegungString;
              Canvas.FontColor := YellowColor;

              if not Plan.FreeGameDays.ContainsKey
                (Trunc(Mannschaft.WunschTermine[j].Date)) then
              begin
                if IsVorrunde then
                  Inc(TermineVorrundeBelegt)
                else
                  Inc(TermineRueckrundeBelegt);
              end;

            end;
          end;

          Mannschaft.GetParallelGames(Mannschaft.WunschTermine[j].Date, Plan,
            BelegungNum, BelegungString, True, True);

          if BelegungNum > 0 then
          begin
            S := S + ', parallele Spiele (' + IntToStr(BelegungNum) + '): ' +
              BelegungString;

            if IsVorrunde then
              Inc(TermineVorrundeParallel)
            else
              Inc(TermineRueckrundeParallel);

          end;

          if Plan.FreeGameDays.ContainsKey
            (Trunc(Mannschaft.WunschTermine[j].Date)) then
          begin
            S := S + ', spielfreier Tag';
            Canvas.FontColor := RedColor;

            if IsVorrunde then
              Inc(TermineSpielFreiVorrunde)
            else
              Inc(TermineSpielFreiRueckRunde);

          end;

          Canvas.TextOut(20, yPos, S);
          Canvas.FontColor := NormalColor;
        end;
      end;

      SperrTerminCount := 0;
      for j := 0 to Mannschaft.SperrTermine.Count - 1 do
      begin
        if Plan.IsInRoundToGenerate(Mannschaft.SperrTermine[j]) then
        begin
          Inc(SperrTerminCount);
        end;
      end;

      if SperrTerminCount > 0 then
      begin
        Inc(yPos, 25);
        Canvas.TextOut(10, yPos, 'Sperrtermine (' + IntToStr(SperrTerminCount) +
          ' Stck):');
      end;

      for j := 0 to Mannschaft.SperrTermine.Count - 1 do
      begin
        if Plan.IsInRoundToGenerate(Mannschaft.SperrTermine[j]) then
        begin
          Inc(yPos, 20);
          S := MyFormatDate(Mannschaft.SperrTermine[j]);
          Canvas.TextOut(20, yPos, S);
        end;
      end;

      if Mannschaft.AuswaertsKoppel.Count > 0 then
      begin
        Inc(yPos, 25);
        Canvas.TextOut(10, yPos, 'Auswrtskoppelwnsche:');
      end;

      for j := 0 to Mannschaft.AuswaertsKoppel.Count - 1 do
      begin
        if Mannschaft.AuswaertsKoppel[j].Option = ktNone then
        begin
          continue;
        end;

        Inc(yPos, 20);

        S := Mannschaft.AuswaertsKoppel[j].TeamA + ' und ' +
          Mannschaft.AuswaertsKoppel[j].TeamB + ' ';
        if Mannschaft.AuswaertsKoppel[j].Option = ktSameDay then
        begin
          S := S + 'am gleichen Tag';
        end
        else if Mannschaft.AuswaertsKoppel[j].Option = ktDiffDay then
        begin
          S := S + 'mit bernachtung';
        end
        else
        begin
          S := S + 'mit optionaler bernachtung';
        end;

        Canvas.TextOut(20, yPos, S);
      end;

      if Mannschaft.noWeekGame.Count > 0 then
      begin
        Inc(yPos, 25);
        Canvas.TextOut(10, yPos,
          'Spiele gegen diese Mannschaften nur am Wochenende (60km Regel):');
      end;

      for j := 0 to Mannschaft.noWeekGame.Count - 1 do
      begin
        Inc(yPos, 20);
        S := Mannschaft.noWeekGame[j];
        Canvas.TextOut(20, yPos, S);
      end;

      if (Mannschaft.HomeRightCountRoundOne >= 0) or
        (Mannschaft.HomeRights.Count > 0) then
      begin
        Inc(yPos, 25);
        Canvas.TextOut(10, yPos, 'Heimrecht:');

        if Mannschaft.HomeRightCountRoundOne >= 0 then
        begin
          Inc(yPos, 20);
          S := 'Vorrunde: ' + IntToStr(Mannschaft.HomeRightCountRoundOne) +
            ' Rckrunde: ' + IntToStr(Plan.Mannschaften.Count -
            Mannschaft.HomeRightCountRoundOne - 1);
          Canvas.TextOut(20, yPos, S);
        end;

        for j := 0 to Mannschaft.HomeRights.Count - 1 do
        begin
          Inc(yPos, 20);
          S := Mannschaft.HomeRights[j].AsString;
          Canvas.TextOut(20, yPos, S);
        end;
      end;

      // Meldungsstatistik
      Inc(yPos, 25);
      Canvas.TextOut(10, yPos, 'Auswertung:');
      Inc(yPos, 20);

      // Doppelrunde wird hier so dargestellt, dass die beiden Hauptrunden als Vor und Rckrunde gesehen werden
      MinTerminePerRound := ((Plan.Mannschaften.Count - 1) * 1.0) / 2;

      if Plan.getOptions.RoundPlaning <> rpCorona then
      begin
        EffectTerminValue := TermineVorrunde - TermineVorrundeBelegt -
          TermineSpielFreiVorrunde;

        if Plan.getOptions().DoubleRound then
        begin
          // Doppelrunde, brauchen wir doppelt so viele Termine
          MinTerminePerRound := MinTerminePerRound * 2;
        end;

        MinTerminePerRound := Trunc(MinTerminePerRound + 0.75) + 2;

        if EffectTerminValue < MinTerminePerRound then
        begin
          Canvas.FontColor := RedColor;
        end
        else
        begin
          Canvas.FontColor := GreenColor;
        end;

        if Plan.getOptions().RoundPlaning = rpHalfRound then
        begin
          S := 'Terminwnsche: ' + IntToStr(TermineVorrunde);
        end
        else
        begin
          S := 'Terminwnsche Vorrunde: ' + IntToStr(TermineVorrunde);
        end;

        if TermineVorrundeBelegt > 0 then
        begin
          S := S + ', davon belegt: ' + IntToStr(TermineVorrundeBelegt);
        end;

        if TermineSpielFreiVorrunde > 0 then
        begin
          S := S + ', davon an spielfreien Tagen: ' +
            IntToStr(TermineSpielFreiVorrunde);
        end;

        if TermineVorrundeParallel > 0 then
        begin
          S := S + ', davon mit parallelen Spielen: ' +
            IntToStr(TermineVorrundeParallel);
        end;
        S := S + ', effektiv: ' + IntToStr(EffectTerminValue);

        Canvas.TextOut(20, yPos, S);

        Canvas.FontColor := NormalColor;
        Inc(yPos, 20);
      end
      else if Plan.getOptions.RoundPlaning <> rpSecondOnly then
      begin
        EffectTerminValue := TermineVorrunde - TermineVorrundeBelegt -
          TermineSpielFreiVorrunde;

        if Plan.getOptions().DoubleRound then
        begin
          // Doppelrunde, brauchen wir doppelt so viele Termine
          MinTerminePerRound := MinTerminePerRound * 2;
        end;

        MinTerminePerRound := Trunc(MinTerminePerRound + 0.75) + 2;

        if EffectTerminValue < MinTerminePerRound then
        begin
          Canvas.FontColor := RedColor;
        end
        else
        begin
          Canvas.FontColor := GreenColor;
        end;

        if Plan.getOptions().RoundPlaning = rpHalfRound then
        begin
          S := 'Terminwnsche: ' + IntToStr(TermineVorrunde);
        end
        else
        begin
          S := 'Terminwnsche Vorrunde: ' + IntToStr(TermineVorrunde);
        end;

        if TermineVorrundeBelegt > 0 then
        begin
          S := S + ', davon belegt: ' + IntToStr(TermineVorrundeBelegt);
        end;

        if TermineSpielFreiVorrunde > 0 then
        begin
          S := S + ', davon an spielfreien Tagen: ' +
            IntToStr(TermineSpielFreiVorrunde);
        end;

        if TermineVorrundeParallel > 0 then
        begin
          S := S + ', davon mit parallelen Spielen: ' +
            IntToStr(TermineVorrundeParallel);
        end;
        S := S + ', effektiv: ' + IntToStr(EffectTerminValue);

        Canvas.TextOut(20, yPos, S);

        Canvas.FontColor := NormalColor;
        Inc(yPos, 20);
      end;

      if not(Plan.getOptions.RoundPlaning in [rpHalfRound, rpFirstOnly, rpCorona]) then
      begin

        EffectTerminValue := TermineRueckrunde - TermineRueckrundeBelegt -
          TermineSpielFreiRueckRunde;

        if EffectTerminValue < MinTerminePerRound then
        begin
          Canvas.FontColor := RedColor;
        end
        else
        begin
          Canvas.FontColor := GreenColor;
        end;

        S := 'Terminwnsche Rckrunde: ' + IntToStr(TermineRueckrunde);

        if TermineRueckrundeBelegt > 0 then
        begin
          S := S + ', davon belegt: ' + IntToStr(TermineRueckrundeBelegt);
        end;

        if TermineSpielFreiRueckRunde > 0 then
        begin
          S := S + ', davon an spielfreien Tagen: ' +
            IntToStr(TermineSpielFreiRueckRunde);
        end;

        if TermineRueckrundeParallel > 0 then
        begin
          S := S + ', davon mit parallelen Spielen: ' +
            IntToStr(TermineRueckrundeParallel);
        end;
        S := S + ', effektiv: ' + IntToStr(EffectTerminValue);

        Canvas.TextOut(20, yPos, S);

        // Gesperrte Termine
        Canvas.FontColor := RedColor;
        MessageList := TList<String>.Create;
        Mannschaft.getTermineNotPossible(Plan, MessageList);

        for j := 0 to MessageList.Count - 1 do
        begin
          Inc(yPos, 20);
          Canvas.TextOut(20, yPos, MessageList[j]);
        end;

        MessageList.Free;
        Canvas.FontColor := NormalColor;

        Inc(yPos, 20);
      end;
      Inc(yPos, 20);
    end;
  end;

  Result.cy := yPos - 40;
  Result.cx := Max(xMax, 1000);
end;

function TFormPlanPanel.PaintSisterTeams(Canvas: TZoomCanvas;
  ForPrint: boolean): TSize;
var
  yPos: Integer;
  Mannschaft: TMannschaft;
  S: String;
  SisterGames: TSisterGames;
  SisterGame: TSisterGame;
  isFirst: boolean;
  NormalColor: TColor;

begin

  yPos := 0;

  if Plan.Mannschaften.Count = 0 then
  begin
    Canvas.TextOut(0, 0, 'Kein Plan geladen');
    Inc(yPos, 40);
  end
  else
  begin
    if not ForPrint then
    begin
      Inc(yPos, 20);
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, 0, 'Termine der Nachbarmannschaften:');
      Canvas.FontHeight := cFontNormal;
    end;

    NormalColor := Canvas.FontColor;

    Inc(yPos, 20);

    Canvas.FontColor := RGB(0, 0, 255);

    Canvas.TextOut(0, yPos, 'Blau: Spiele direkter Nachbarmannschaften');

    Canvas.FontColor := NormalColor;
    for Mannschaft in Plan.Mannschaften do
    begin

      Inc(yPos, 30);
      Canvas.FontHeight := cFontBig;
      Canvas.TextOut(0, yPos, Mannschaft.teamName + ':');
      Canvas.FontHeight := cFontNormal;
      Inc(yPos, 10);

      for SisterGames in Mannschaft.SisterGames.toSortedArray do
      begin
        isFirst := True;
        Inc(yPos, 5);
        for SisterGame in SisterGames.toSortedArray do
        begin

          Inc(yPos, 20);

          if isFirst then
          begin
            S := MyFormatDate(SisterGame.Date);
            Canvas.TextOut(20, yPos, S);
            isFirst := False;
          end;

          S := MyFormatTime(SisterGame.Date);
          Canvas.TextOut(150, yPos, S);

          if SisterGame.PreventParallelGame then
          begin
            // Echte Nachbarmannschaft
            Canvas.FontColor := RGB(0, 0, 255);
          end;

          S := '(' + SisterGame.gender + ') ' + SisterGame.HomeMannschaftName +
            ' - ' + SisterGame.GuestMannschaftName;
          if SisterGame.Location <> '' then
          begin
            S := S + ' Spiellokal: ' + SisterGame.Location;
          end;

          Canvas.TextOut(200, yPos, S);

          Canvas.FontColor := NormalColor;

        end;
      end;
      Inc(yPos, 20);
    end;
  end;

  Result.cy := yPos;
  Result.cx := 1000;
end;

{ TActionRectKostenPlan }

constructor TActionRectKostenPlan.Create(KostenType: TPlanKostenType);
begin
  inherited Create();
  Self.KostenType := KostenType;
end;

procedure TActionRectKostenPlan.DoAction(ScreenPosition: TPoint);
var
  Dlg: TDialogForSingleGewichtungsOptions;
  Values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Plan: TPlan;
begin
  Plan := getMainPlan();
  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  Values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;
  Dlg.SetTitle('Gewichtung ndern');

  titles.Add(cPlanKostenTypeNamen[KostenType]);
  Values.Add(Plan.getOptions.getGewichtungPlan(KostenType));

  Dlg.Init(titles);
  Dlg.SetValues(Values);
  Dlg.SmallDlg();

  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(Values);

    Plan := getMainPlan();
    Plan.getOptions.setGewichtungPlan(KostenType, Values[0]);

    DoAfterMainPlanChanged(True);
  end;
  Values.Free;
  titles.Free;
  Dlg.Free();
end;

{ TActionRectKostenTypeMannschaft }

constructor TActionRectKostenTypeMannschaft.Create
  (KostenType: TMannschaftsKostenType);
begin
  inherited Create();
  Self.KostenType := KostenType;
end;

procedure TActionRectKostenTypeMannschaft.DoAction(ScreenPosition: TPoint);
var
  Dlg: TDialogForSingleGewichtungsOptions;
  Values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Plan: TPlan;
begin
  Plan := getMainPlan();
  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  Values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;
  Dlg.SetTitle('Gewichtung ndern');

  titles.Add(cMannschaftsKostenTypeNamen[KostenType]);
  Values.Add(Plan.getOptions.GewichtungMainMannschaftsKosten[KostenType]);

  Dlg.Init(titles);
  Dlg.SetValues(Values);
  Dlg.SmallDlg();

  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(Values);

    Plan := getMainPlan();
    Plan.getOptions.GewichtungMainMannschaftsKosten[KostenType] := Values[0];

    DoAfterMainPlanChanged(True);
  end;
  Values.Free;
  titles.Free;
  Dlg.Free();
end;

{ TActionRectKostenMannschaftMain }

constructor TActionRectKostenMannschaftMain.Create(teamId: String);
begin
  inherited Create();
  Self.teamId := teamId;
end;

procedure TActionRectKostenMannschaftMain.DoAction(ScreenPosition: TPoint);
var
  Dlg: TDialogForSingleGewichtungsOptions;
  Values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Plan: TPlan;
  Mannschaft: TMannschaft;
begin
  Plan := getMainPlan();
  Mannschaft := Plan.Mannschaften.FindByTeamId(teamId);

  if not Assigned(Mannschaft) then
  begin
    Exit;
  end;

  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  Values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;
  Dlg.SetTitle('Gewichtung der Mannschaft ndern');

  titles.Add(Mannschaft.teamName);
  Values.Add(Plan.getOptions.GewichtungMannschaften.getMain(teamId));

  Dlg.Init(titles);
  Dlg.SetValues(Values);
  Dlg.SmallDlg();

  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(Values);

    Plan := getMainPlan();
    Plan.getOptions.GewichtungMannschaften.addMain(teamId, Values[0]);

    DoAfterMainPlanChanged(True);
  end;
  Values.Free;
  titles.Free;
  Dlg.Free();
end;

{ TActionRectKostenMannschaftMainDetail }

constructor TActionRectKostenMannschaftDetail.Create(teamId: String;
  KostenType: TMannschaftsKostenType);
begin
  inherited Create();
  Self.teamId := teamId;
  Self.KostenType := KostenType;
end;

procedure TActionRectKostenMannschaftDetail.DoAction(ScreenPosition: TPoint);
var
  Dlg: TDialogForSingleGewichtungsOptions;
  Values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Plan: TPlan;
  Mannschaft: TMannschaft;
  MannschaftsKosten: TMannschaftsKosten;
begin
  Plan := getMainPlan();
  Mannschaft := Plan.Mannschaften.FindByTeamId(teamId);

  if not Assigned(Mannschaft) then
  begin
    Exit;
  end;

  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  Values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;
  Dlg.SetTitle('Gewichtung der Mannschaft ' + Mannschaft.teamName + ' ndern');

  titles.Add(cMannschaftsKostenTypeNamen[KostenType]);
  Values.Add(Plan.getOptions.GewichtungMannschaften.getDetail(teamId,
    KostenType));

  Dlg.Init(titles);
  Dlg.SetValues(Values);

  // Die Kostenberechnung muss ber den ViewdPlan gehen und nicht ber den MainPlan
  Plan := getCurrentViewedPlan();
  if Assigned(Plan) then
  begin
    Mannschaft := Plan.Mannschaften.FindByTeamId(teamId);
    if Assigned(Mannschaft) then
    begin
      MannschaftsKosten := TMannschaftsKosten.Create;
      Mannschaft.CalculateKostenForType(Plan, KostenType, MannschaftsKosten,
        tmDetailMessage);

      if MannschaftsKosten.Messages.Count = 0 then
      begin
        MannschaftsKosten.Messages.Add
          ('Zu dieser Kostenart gibt es bei dieser Mannschaft keine Meldung')
      end;

      Dlg.SetMessages(MannschaftsKosten.Messages);
      MannschaftsKosten.Free;
    end;
  end;

  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(Values);

    Plan := getMainPlan();
    Plan.getOptions.GewichtungMannschaften.addDetail(teamId, KostenType,
      Values[0]);

    DoAfterMainPlanChanged(True);
  end;
  Values.Free;
  titles.Free;
  Dlg.Free();
end;

initialization

cColorBrushGray := RGB(240, 240, 240);
cColorLineGray := RGB(200, 200, 200);
cColorHomeGame := RGB(255, 0, 0);
cColorAwayGame := RGB(0, 0, 0);
cColorHomeGameBubble := RGB(255, 128, 128);
cColorAwayGameBubble := RGB(128, 128, 128);
// cColorGameDayLine1 := RGB(255, 255, 0);
// cColorGameDayLine2 := RGB(0, 255, 255);
cColorGameDayLine1 := RGB(255, 215, 0);
cColorGameDayLine2 := RGB(0, 205, 205);
cColorNormalWunsch := RGB(0, 0, 0);
cColorKoppelWunsch := RGB(0, 0, 255);
cColorSperrWunsch := RGB(255, 0, 0);

end.
