unit PlanOptimizer;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, PlanTypes, PlanUtils, StdCtrls, math, ExtCtrls, PlanMainThread, SyncObjs;




type
  TPlanOptimizer = class(TThread)
  public
    { Private-Deklarationen }
    Durchlaufe: Int64;
    LastResetDurchlauf: Int64;
    StartTick: DWORD;
    Threads: TList;
    SpecialThread: TPlanMainThread;
    SpecialThreadMaxGewichtung: TMannschaftsKostenType;
    procedure StartThreads();
    procedure KillThreads();
  public
    { Public-Deklarationen }
    Plan: TPlan;
    CriticalSectionOptimizer: TCriticalSection;
  private
    ReInitPlanData: boolean;
    FPaused: boolean;
    procedure SetPaused(const Value: boolean);
    procedure WaitForThreadInit;
    function  getNextMaxGewichtung(Plan: TPlan): TMannschaftsKostenType;
    procedure StartSpecialThread;
  public
    constructor Create();
    destructor Destroy(); override;

    procedure copyPlanDates(Plan: TPlan);

    procedure setNewPlanData(Plan: TPlan);

    procedure Execute; override;

    function PlanPerSec: MyDouble;

    property Paused: boolean read FPaused write SetPaused;
  end;


implementation

const
  //cDurchLaufBeforeReinit = 100000;
{$ifopt D+}
  cDurchLaufBeforeReinit = 1000;
  cDurchLaufBeforeStartSpecialThread = 300000;
  cMinDurchLaufSpecialThread = 20000;
{$else}
  cDurchLaufBeforeReinit = 100000;
  cDurchLaufBeforeStartSpecialThread = 3000000;
  cMinDurchLaufSpecialThread = 200000;
  //cDurchLaufBeforeReinit = 1000;
{$endif}


procedure TPlanOptimizer.Execute;
var
  Kosten: MyDouble;
  LastMessungTick: DWORD;
  Thread: TPlanMainThread;
  i: integer;
  ActPlanKosten: MyDouble;
  ThreadDurchLauf, ActDurchlaufe: Int64;
  BadestThread: TPlanMainThread;
  BadestKosten: MyDouble;
  ReinitString: String;
  BadestThreadDurchLaufe, ThreadDurchLaufAfterReInit, ThreadDurchLaufOhneVerbesserung: Int64;
  TempPlan: TPlan;
  Dummy1, Dummy2, Dummy3: Int64;
  FloatDummy: MyDouble;
  IsInitWithSpecialPlan: boolean;
begin
  Durchlaufe := 0;
  LastResetDurchlauf := 0;
  StartTick := GetTickCount;
  LastMessungTick := GetTickCount;
  StartThreads();
  try
    while not Terminated do
    begin
      if Paused then
      begin
        Sleep(100);
        Continue;
      end;

      if GetTickCount - LastMessungTick > 1000 then
      begin
        ActDurchlaufe := 0;
        CriticalSectionOptimizer.Enter;
        ActPlanKosten := Plan.CalculateKosten(-1);
        BadestKosten := -1;
        BadestThread := nil;
        BadestThreadDurchLaufe := 0;
        for i:=0 to Threads.Count-1 do
        begin
          Thread := Threads[i];
          Thread.GetActResult(nil, ThreadDurchLauf, ThreadDurchLaufAfterReInit, ThreadDurchLaufOhneVerbesserung, Kosten);
          Inc(ActDurchlaufe, ThreadDurchLauf);

          if (Kosten >= 0) and ((Kosten > BadestKosten) or (BadestKosten < 0)) then
          begin
            BadestThread := Thread;
            BadestKosten := Kosten;
            BadestThreadDurchLaufe := ThreadDurchLaufOhneVerbesserung;
          end;

          (*
          if (Kosten >= 0) and ((Kosten > ActPlanKosten) and (ThreadDurchLaufOhneVerbesserung > 100000)) then
          begin
            BadestThread := Thread;
          end;
          *)


          if (Kosten >= 0) and ((Kosten < ActPlanKosten) or (ActPlanKosten < 0)) and (not ReInitPlanData) then
          begin
            Thread.GetActResult(Plan, ThreadDurchLauf, ThreadDurchLaufAfterReInit, ThreadDurchLaufOhneVerbesserung, Kosten);
            ActPlanKosten := Plan.CalculateKosten(-1);
          end;
        end;

        Durchlaufe := ActDurchlaufe;

        if Assigned(SpecialThread) then
        begin
          SpecialThread.GetActResult(nil, ThreadDurchLauf, ThreadDurchLaufAfterReInit, ThreadDurchLaufOhneVerbesserung, Kosten);
          Durchlaufe := Durchlaufe + ThreadDurchLauf;
        end;

        if Durchlaufe > cDurchLaufBeforeStartSpecialThread then
        begin
          StartSpecialThread();
        end;


        if (BadestThreadDurchLaufe > cDurchLaufBeforeReinit)  and (not ReInitPlanData) then
        begin
          if BadestThread <> nil then
          begin
            if cMaxThreads > 1 then
            begin

              IsInitWithSpecialPlan := False;

              ReinitString := ' Reinit: ' + MyFormatFloat(Durchlaufe / 1000000.0);

              if Assigned(SpecialThread) then
              begin
                if SpecialThread.Durchlaufe > cMinDurchLaufSpecialThread then
                begin
                  TempPlan := TPlan.Create;
                  TempPlan.Assign(Plan);
                  SpecialThread.GetActResult(TempPlan, Dummy1, Dummy2, Dummy3, FloatDummy);

                  IsInitWithSpecialPlan := True;

{$ifdef TRACE}
                  TraceString('#############################################Thread ' + cMannschaftsKostenTypeNamen[SpecialThreadMaxGewichtung] + ' reinit!!');
{$endif}

                  BadestThread.ReInit(TempPlan, 'Von Spezial' + cMannschaftsKostenTypeNamen[SpecialThreadMaxGewichtung]);

                  TempPlan.Free;

                  SpecialThread.Durchlaufe := 0;
                  SpecialThreadMaxGewichtung := getNextMaxGewichtung(Plan);
                  SpecialThread.ReInit(Plan, '');
                  SpecialThread.SetMaxOption(SpecialThreadMaxGewichtung);
                end;
              end;

              if not IsInitWithSpecialPlan then
              begin
                BadestThread.ReInitAll(ReinitString);
              end;

{$ifdef TRACE}
              TraceString('Thread ' + BadestThread.ThreadName + ' reinit!!');
{$endif}
            end;
          end;
        end;

        (*
        if not ReInitOptions then
        begin
          if BadestThread <> nil then
          begin
            BadestThread.ReInitAll();
            TraceString('Thread ' + BadestThread.ThreadName + ' reinit!!');
          end;
        end;
        *)


        if ReInitPlanData then
        begin
          CriticalSectionOptimizer.Enter;
          for i:=0 to Threads.Count-1 do
          begin
            Thread := Threads[i];
            Thread.ReInit(Plan, Thread.ReinitString);
          end;
          if Assigned(SpecialThread) then
          begin
            SpecialThread.ReInit(Plan, '');
          end;
          ReInitPlanData := False;
{$ifdef TRACE}
          TraceString('Thread ' + BadestThread.ThreadName + ' reinit Options!!' + Plan.getOptions().ToString);
{$endif}
          StartTick := GetTickCount;
          LastResetDurchlauf := Durchlaufe;
          CriticalSectionOptimizer.Leave;
        end;

        CriticalSectionOptimizer.Leave;

        LastMessungTick := GetTickCount;
      end;

      Sleep(100)

    end;
  finally
    KillThreads();
  end;

end;





function TPlanOptimizer.getNextMaxGewichtung(Plan: TPlan): TMannschaftsKostenType;
var
  KostenTypes: set of TMannschaftsKostenType;
begin
  KostenTypes := [];
  Result := mktAbstandHeimAuswaerts;
  if Plan.HasAuswaertsKoppelTermine and (Plan.CalculateMannschaftsKosten(mktAuswaertsKoppelTermine) > 0) then
  begin
    KostenTypes := KostenTypes + [mktAuswaertsKoppelTermine];
  end;

  if Plan.HasKoppelTermine and (Plan.CalculateMannschaftsKosten(mktKoppelTermine) > 0) then
  begin
    KostenTypes := KostenTypes + [mktKoppelTermine];
  end;

  if Plan.HasRanking and (Plan.CalculateMannschaftsKosten(mktRanking) > 0) then
  begin
    KostenTypes := KostenTypes + [mktRanking];
  end;

  if Plan.CalculateMannschaftsKosten(mktSisterGames) > 0 then
  begin
    KostenTypes := KostenTypes + [mktSisterGames];
  end;

  if Plan.Has60KilometerValues and (Plan.CalculateMannschaftsKosten(mkt60Kilometer) > 0) then
  begin
    KostenTypes := KostenTypes + [mkt60Kilometer];
  end;

  if Plan.CalculateMannschaftsKosten(mktAbstandHeimAuswaerts) > 0 then
  begin
    KostenTypes := KostenTypes + [mktAbstandHeimAuswaerts];
  end;

  if Plan.CalculateMannschaftsKosten(mkt2SpieleProWoche) > 0 then
  begin
    KostenTypes := KostenTypes + [mkt2SpieleProWoche];
  end;


  if KostenTypes <> [] then
  begin
    while True do
    begin
      Result := TMannschaftsKostenType(Ord(Low(TMannschaftsKostenType)) + Random(Ord(High(TMannschaftsKostenType))));

      if Result in KostenTypes then
      begin
        break;
      end;
    end;
  end;

end;

procedure TPlanOptimizer.KillThreads;
var
  i: integer;
  Thread: TPlanMainThread;
begin
  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];
    if Thread.Suspended then
      Thread.Suspended := False;
    Thread.Terminate;

    while(not Thread.Finished) do
    begin
      Sleep(2);
    end;

    Thread.Free;
  end;

  Threads.Clear;

  if Assigned(SpecialThread) then
  begin
    if SpecialThread.Suspended then
      SpecialThread.Suspended := False;
    SpecialThread.Terminate;

    while(not SpecialThread.Finished) do
    begin
      Sleep(2);
    end;

    FreeAndNil(SpecialThread);
  end;

end;


procedure TPlanOptimizer.SetPaused(const Value: boolean);
var
  i: integer;
  Thread: TPlanMainThread;
begin
  FPaused := Value;

  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];
    Thread.Paused := Value;
  end;

  if Assigned(SpecialThread) then
  begin
    SpecialThread.Paused := Value;
  end;

  StartTick := GetTickCount;
  LastResetDurchlauf := Durchlaufe;

end;

procedure TPlanOptimizer.WaitForThreadInit();
var
  i: integer;
  Thread: TPlanMainThread;
begin
  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];

    while(not Thread.IsInit) do
    begin
      Sleep(10);
    end;
  end;

  if Assigned(SpecialThread) then
  begin
    while(not SpecialThread.IsInit) do
    begin
      Sleep(10);
    end;

  end;
end;

procedure TPlanOptimizer.StartSpecialThread;
begin
  if not Assigned(SpecialThread) then
  begin
  {$ifdef SPECIALTHREAD}
    SpecialThread := TPlanMainThread.Create();
    SpecialThread.InitWithPlan('Special', Plan, CriticalSectionOptimizer);
    SpecialThreadMaxGewichtung := getNextMaxGewichtung(Plan);
    SpecialThread.ReInit(Plan, '');
    SpecialThread.SetMaxOption(SpecialThreadMaxGewichtung);
    SpecialThread.Paused := true;
    SpecialThread.Start;
    SpecialThread.Paused := FPaused;
    while(not SpecialThread.IsInit) do
    begin
      Sleep(10);
    end;

  {$else}
    SpecialThread := nil;
  {$endif}
  end;
end;

procedure TPlanOptimizer.StartThreads;
var
  i: integer;
  Thread: TPlanMainThread;
begin
{$ifdef TRACE}
  TraceString('Beginn TPlanOptimizer.StartThreads');
{$endif}

  CriticalSectionOptimizer.Enter;
  for i:=1 to cMaxThreads do
  begin
    Thread := TPlanMainThread.Create();
    Thread.InitWithPlan(IntToStr(i), Plan, CriticalSectionOptimizer);
    Thread.Paused := true;
    Thread.Start;
    Threads.Add(Thread);
  end;

  SpecialThread := nil;


  CriticalSectionOptimizer.Leave;
  WaitForThreadInit();
  SetPaused(FPaused);
{$ifdef TRACE}
  TraceString('Ende TPlanOptimizer.StartThreads');
{$endif}

end;

constructor TPlanOptimizer.Create;
begin
  inherited Create(true);
  Plan := TPlan.Create;

  Threads := TList.Create;

  SpecialThreadMaxGewichtung := mktAbstandHeimAuswaerts;

  CriticalSectionOptimizer := TCriticalSection.Create;
end;

destructor TPlanOptimizer.Destroy;
begin
  KillThreads();
  Threads.Free;
  Plan.Free;
  CriticalSectionOptimizer.free;
  inherited;
end;


procedure TPlanOptimizer.copyPlanDates(Plan: TPlan);
begin
  CriticalSectionOptimizer.Enter;
  Plan.AssignOptimizedDates(Self.Plan);
  CriticalSectionOptimizer.Leave;
end;



procedure TPlanOptimizer.setNewPlanData(Plan: TPlan);
begin
  CriticalSectionOptimizer.Enter;
  ReInitPlanData := true;
  Self.Plan.AssignPlanData(Plan);
  Self.Plan.RemoveNotValidDates();
  CriticalSectionOptimizer.Leave;
end;


function TPlanOptimizer.PlanPerSec: MyDouble;
var
  Ticks: DWORD;
begin
  Ticks := GetTickCount - StartTick;

  if(Ticks / 1000.0) > 0 then
  begin
    Result := ((Durchlaufe - LastResetDurchlauf) * 1.0) / (Ticks / 1000.0);
  end
  else
  begin
    Result := 0;
  end;

end;




initialization
  Randomize();
end.
