unit PlanMainThread;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms, System.Types,
  Dialogs, PlanTypes, PlanUtils, StdCtrls, math, ExtCtrls, SyncObjs, DialogProfile;


type
  TPlanMainThread = class(TThread)
  public
    { Private-Deklarationen }
    Durchlaufe: Int64;
    ReInitByDurchLauf: Int64;
    VerbesserungByDurchLauf: Int64;
    ReinitString: String;
    ReInitPlanData: TPlan;
    FlagReInitAllDates: Boolean;
    FlagReInitOptions: Boolean;
    Threads: TList;
    DynamicThreads: TList;
    CriticalSectionMainThread: TCriticalSection;
    procedure StartThreads();
    procedure KillThreads();
  public
    { Public-Deklarationen }
    Plan: TPlan;
    RetKosten: MyDouble;
    ThreadName: String;
    ParentCriticalSection: TCriticalSection;
    IsInit: boolean;
  private
    FPaused: boolean;
    procedure SetPaused(const Value: boolean);
    procedure WaitForThreadInit;
  public
    constructor Create();
    destructor Destroy(); override;

    procedure InitWithPlan(ThreadName: String; Plan: TPlan; ParentCriticalSection: TCriticalSection);

    procedure ReInitAll(ReinitString: String);

    procedure ReInit(Plan: TPlan; ReinitString: String);

    procedure SetMaxOption(KostenType: TMannschaftsKostenType);

    procedure GetActResult(Plan: TPlan; var DurchLauf, DurchLaufAfterReInit, DurchLaufOhneVerbesserung: Int64; var Kosten: MyDouble);

    procedure Execute; override;

    property Paused: boolean read FPaused write SetPaused;
  end;


implementation


procedure TPlanMainThread.Execute;
var
  Kosten: MyDouble;
  LastMessungTick: DWORD;
  Thread: TPlanCalcThread;
  i: integer;
  ThreadDurchLauf: Int64;
  MustReinit: boolean;
  ThreadBestType: String;
  NumThreads: Integer;
  TempPlan: TPlan;
{$ifdef PROFILE}
  S: String;
{$endif}

begin
  Durchlaufe := 0;
  ReInitByDurchLauf := 0;
  VerbesserungByDurchLauf := 0;
  LastMessungTick := GetTickCount;
  RetKosten := -1;
  StartThreads();
  IsInit := True;
  try
    while not Terminated do
    begin
      if Paused then
      begin
        Sleep(100);
        Continue;
      end;

      if GetTickCount - LastMessungTick > 100 then
      begin

        ThreadBestType := '';

        RetKosten := Plan.CalculateKosten(-1);

        CriticalSectionMainThread.Enter;

        if FlagReInitAllDates then
        begin
          RetKosten := -1;
          ReInitByDurchLauf := Durchlaufe;
          VerbesserungByDurchLauf := Durchlaufe;
          Plan.clearAllDates();
          for i:=0 to Threads.Count-1 do
          begin
            Thread := Threads[i];
            Thread.DoReInitPlan(Plan, ReinitString);
          end;
          FlagReInitAllDates := False;
        end;

        if FlagReInitOptions then
        begin
          ReInitByDurchLauf := Durchlaufe;
          VerbesserungByDurchLauf := Durchlaufe;
          Plan.AssignPlanData(ReInitPlanData);
          Plan.RemoveNotValidDates();
          FreeAndNil(ReInitPlanData);
          RetKosten := Plan.CalculateKosten(-1);
          for i:=0 to Threads.Count-1 do
          begin
            Thread := Threads[i];
            Thread.DoReInitPlanData(Plan, ReinitString);
          end;
          FlagReInitOptions := False;
        end;


        Durchlaufe := 0;
        MustReinit := False;
        for i:=0 to Threads.Count-1 do
        begin
          Thread := Threads[i];
          Thread.GetActResult(nil, ThreadDurchLauf, Kosten);
          Inc(Durchlaufe, ThreadDurchLauf);

          if (Kosten >= 0) and ((Kosten < RetKosten) or (RetKosten < 0)) then
          begin
            TempPlan := TPlan.Create;
            TempPlan.Assign(Plan);
            Thread.GetActResult(TempPlan, ThreadDurchLauf, Kosten);
{$ifdef PROFILE}
            AddThreadTypeToStatistics(Thread.ThreadType);
{$endif}

            if 1 < Abs(Kosten - TempPlan.CalculateKosten(-1)) then
            begin
              // Hier ist was faul
              {$ifdef TRACE}
              TraceString('Hier ist was oberfaul ' + ThreadName);
              {$endif}
              //Kosten := TempPlan.CalculateKosten(-1);
            end;

            Plan.AssignOptimizedDates(TempPlan);
            if 1 < Abs(Kosten - Plan.CalculateKosten(-1)) then
            begin
              // Hier ist was faul
              {$ifdef TRACE}
              TraceString('Jetzt geht das immer noch nicht ' + ThreadName);
              {$endif}
              //Kosten := Plan.CalculateKosten(-1);
            end;

            RetKosten := Kosten;
            MustReinit := True;
            ThreadBestType := Thread.ThreadType;
            TempPlan.Free;
          end;
        end;

        if MustReinit then
        begin
          // Eine Verbesserung wurde gefunden, mit diesem Plan weitermachen
          VerbesserungByDurchLauf := Durchlaufe;
          for i:=0 to Threads.Count-1 do
          begin
            Thread := Threads[i];
            Thread.DoReInitPlan(Plan, '');
          end;

          if ThreadBestType <> '' then
          begin
            // Nur, wenn nicht schon 30% der Threads mit dem Typ laufen einen neuen dynaischen Thread mit diesem Typ setzen
            NumThreads := 0;
            for i := 0 to DynamicThreads.Count-1 do
            begin
              Thread := DynamicThreads[0];
              if Thread.ThreadType = ThreadBestType then
              begin
                Inc(NumThreads);
              end;
            end;

            if NumThreads < (DynamicThreads.Count div 3) then
            begin
              if DynamicThreads.Count > 1 then
              begin
                Thread := DynamicThreads[0];
                Thread.SetThreadType(ThreadBestType);
                DynamicThreads.Remove(Thread);
                DynamicThreads.Add(Thread);
              end;
            end;
          end;

{$ifdef PROFILE}
          S := '';
          for i:=0 to DynamicThreads.Count-1 do
          begin
            Thread := DynamicThreads[i];
            if i <> 0 then
            begin
              S := S + ',';
            end;
            S := S + Thread.ThreadType;
          end;

          AddThreadInfoToStatistics(ThreadName, S);
{$endif}



        end;
        CriticalSectionMainThread.Leave;
        LastMessungTick := GetTickCount;
      end;

      Sleep(100)

    end;
  finally
    KillThreads();
  end;

end;





procedure TPlanMainThread.KillThreads;
var
  i: integer;
  Thread: TPlanCalcThread;
begin
  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];
    if Thread.Suspended then
      Thread.Suspended := False;
    Thread.Terminate;
    while(not Thread.Finished) do
    begin
      Sleep(2);
    end;
    Thread.free;
  end;

  Threads.Clear;
  DynamicThreads.Clear;
end;

procedure TPlanMainThread.SetPaused(const Value: boolean);
var
  i: integer;
  Thread: TPlanCalcThread;
begin
  FPaused := Value;

  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];
    Thread.Paused := Value;
  end;
end;

procedure TPlanMainThread.StartThreads;
var
  i: integer;
  Thread: TPlanCalcThread;
begin
{$ifdef TRACE}
  TraceString('Beginn TPlanMainThread.StartThreads ' + ThreadName);
{$endif}
  CriticalSectionMainThread.Enter;
  for i := Low(cThreadTauschPercent) to High(cThreadTauschPercent) do
  begin

    Thread := TPlanCalcThread.Create();
    Thread.StartWithLists(ThreadName, Plan, cThreadTauschPercent[i], CriticalSectionMainThread);
    Thread.Paused := True;
    //Thread.StartWithLists(TrainerList, Termine, OptTermine, OptTrainer, True);
    Thread.Start;
    Threads.Add(Thread);
  end;

  // 10 Dynamische Thread
  for i := 0 to NUM_DYN_THREADS do
  begin
    Thread := TPlanCalcThread.Create();
    Thread.StartWithLists(ThreadName + '|Dyn', Plan, '2', CriticalSectionMainThread);
    Thread.Paused := True;
    Thread.Start;
    Threads.Add(Thread);
    DynamicThreads.Add(Thread);
  end;


  CriticalSectionMainThread.Leave;

  WaitForThreadInit();

  SetPaused(FPaused);

{$ifdef TRACE}
  TraceString('Ende TPlanMainThread.StartThreads ' + ThreadName);
{$endif}

end;

constructor TPlanMainThread.Create;
begin
  inherited Create(true);
  IsInit := False;
  CriticalSectionMainThread := TCriticalSection.Create;
  Plan := TPlan.Create;

  Threads := TList.Create;
  DynamicThreads := TList.Create;
end;

destructor TPlanMainThread.Destroy;
begin
  KillThreads();
  Threads.Free;
  DynamicThreads.Free;
  Plan.Free;
  CriticalSectionMainThread.Free;
  ReInitPlanData.Free;
  inherited;
end;


procedure TPlanMainThread.InitWithPlan(ThreadName: String; Plan: TPlan; ParentCriticalSection: TCriticalSection);
begin
  Self.ParentCriticalSection := ParentCriticalSection;
  CriticalSectionMainThread.Enter;
  Self.ThreadName := ThreadName;
  Self.Plan.Assign(Plan);
  CriticalSectionMainThread.Leave;
end;

procedure TPlanMainThread.WaitForThreadInit();
var
  i: integer;
  Thread: TPlanCalcThread;
begin
  for i:=0 to Threads.Count-1 do
  begin
    Thread := Threads[i];

    while(not Thread.IsInit) do
    begin
      Sleep(10);
    end;
  end;
end;



procedure TPlanMainThread.GetActResult(Plan: TPlan; var DurchLauf, DurchLaufAfterReInit, DurchLaufOhneVerbesserung: Int64; var Kosten: MyDouble);
begin
  ParentCriticalSection.Enter;
  CriticalSectionMainThread.Enter;

  if Assigned(Plan) then
  begin
    Plan.AssignOptimizedDates(Self.Plan);
  end;
  DurchLauf := Self.Durchlaufe;
  DurchLaufAfterReInit := DurchLauf - Self.ReInitByDurchLauf;
  DurchLaufOhneVerbesserung := DurchLauf - Self.VerbesserungByDurchLauf;
  Kosten :=  RetKosten;

  if FlagReInitAllDates or FlagReInitOptions then
  begin
    // Ich soll mich neu machen -> also habe ich noch keine Kosten
    Kosten := -1;
  end;


  CriticalSectionMainThread.Leave;
  ParentCriticalSection.Leave;
end;


procedure TPlanMainThread.ReInitAll(ReinitString: String);
begin
  FlagReInitAllDates := True;
  Self.ReinitString := ReinitString;
end;

procedure TPlanMainThread.ReInit(Plan: TPlan; ReinitString: String);
begin
  CriticalSectionMainThread.Enter;
  FlagReInitOptions := True;
  FreeAndNil(ReInitPlanData);
  ReInitPlanData := TPlan.Create;
  ReInitPlanData.Assign(Plan);
  Self.ReinitString := ReinitString;
  CriticalSectionMainThread.Leave;
end;


procedure TPlanMainThread.SetMaxOption(KostenType: TMannschaftsKostenType);
begin
  CriticalSectionMainThread.Enter;
  FlagReInitOptions := True;
  if Assigned(ReInitPlanData) then
  begin
    ReInitPlanData.getOptions().GewichtungMainMannschaftsKosten[KostenType] := coExtremHoch;
  end;
  CriticalSectionMainThread.Leave;
end;


initialization
  Randomize();
end.
