unit PlanDataObjects;

(*
  Copyright 2017 Andreas Hofmann
  Lizenz:
  Der Sourcecode steht unter der GPL V3
  siehe <http://www.gnu.org/licenses/>.
*)

interface

uses
  Forms, Windows, Messages, SysUtils, Variants, Classes, Graphics, Contnrs,
  Math, System.Generics.Collections, Xml.XMLDoc, Xml.XMLIntf, PlanUtils,
  System.TypInfo,
  System.Generics.Defaults, Dialogs;

const
  aState = 'state';

  nPlan = 'plan';
  aGender = 'gender';
  aMid = 'mid';
  aFrom = 'from';
  aUntil = 'until';
  aName = 'name';
  aID = 'id';

  nTeam = 'team';
  aClubID = 'clubid';
  aTeamId = 'teamid';
  aTeamNumber = 'teamnumber';
  aTeamName = 'teamname';
  aOrgTeamName = 'orgteamname';
  aLocation = 'location';
  aHomeRights = 'homerights';

  nHomeGameDay = 'homegameday';
  aDateTime = 'datetime';
  aParallelGames = 'parallelgames';
  aAusweichTermin = 'ausweichtermin';
  aCoupleGameDay = 'couplegameday';
  aDoubleGameDay = 'doublegameday';
  aCoupleAuswaertsSecondTime = 'coupleauswaertssecondtime';
  aCouplePrio = 'coupleprio';
  aCoupleSecondTime = 'couplesecondtime';

  nNoGameDay = 'nogameday';
  aDate = 'date';

  nRoadCouple = 'roadcouple';
  aTeamNameA = 'teamnamea';
  aTeamNameB = 'teamnameb';
  aSameDay = 'sameday';

  nNoWeekGames = 'noweekgames';
  // aTeamName = 'teamname';

  nSisterTeam = 'sisterteam';
  // aTeamName = 'teamname';
  // aGender = 'gender';
  // aTeamNumber = 'teamnumber';
  aNoParallelGames = 'noparallelgames';
  aParallelHomeGames = 'parallelhomegames';

  nSisterGame = 'sistergame';
  // aDateTime = 'datetime';
  aHomeTeamName = 'hometeamname';
  aGuestTeamName = 'guestteamname';

  nPredefinedGames = 'predefinedgames';

  nGame = 'game';
  // aDateTime = 'datetime';
  // aHomeTeamName = 'hometeamname';
  // aGuestTeamName = 'guestteamname';
  // aLocation = 'location';

  nExistingSchedule = 'existingschedule';

  nSchedule = 'schedule';

  nMandatoryGames = 'mandatorygames';
  aDateFrom = 'datefrom';
  aDateTo = 'dateto';
  aNumberGames = 'numbergames';

  nRanking = 'ranking';
  aActive = 'active';

  // nTeam
  aRankingIndex = 'rankingindex';

  nHomeRights = 'homerights';
  aHomeRight = 'homeright';

type
  TDataObjectState = (dosNormal, dosNew, dosModified, dosDelete);

  TDataObjectKey = class(TObject)
  public
    Name: String;
    attributeValues: TList<TPair<String, String>>;
  public
    constructor Create();
    destructor Destroy(); override;

    procedure setAttribute(Key, Value: String);
    procedure setAttributeAsDate(Key: String; Date: TDateTime);
    procedure setAttributeAsDateTime(Key: String; Date: TDateTime);
  end;

  TDataObject = class;

  IDataObjectList = interface
    function GetEnumerator: TEnumerator<TDataObject>;
  end;

  TInterfacedDataObjectList = class(TInterfacedObject, IDataObjectList)
    ListValue: TList<TDataObject>;
    function GetEnumerator: TEnumerator<TDataObject>;
    constructor Create;
    destructor Destroy; override;
  end;

  TDataObject = class(TObject)
  public
    State: TDataObjectState;
    Name: String;
    attributes: TDictionary<String, String>;
    childs: TObjectList<TDataObject>;
    parent: TDataObject;
  private
    function isTheSame(Node: TDataObject): boolean;
    function isTheSameOneDirection(Node: TDataObject): boolean;

  public
    constructor Create(Name: String; parent: TDataObject);
    destructor Destroy(); override;

    procedure Clear();

    procedure assignAttributes(Source: TDataObject);
    procedure assign(Source: TDataObject);
    procedure Merge(Source: TDataObject);
    procedure Diff(PlanDataAfter, PlanDataBefore: TDataObject);

    procedure SetAsString(attName: String; Value: String);
    procedure SetAsDate(attName: String; Value: TDateTime);
    procedure SetAsBool(attName: String; Value: boolean);
    procedure SetAsDateTime(attName: String; Value: TDateTime);
    procedure SetAsTime(attName: String; Value: TDateTime);
    procedure SetAsInt(attName: String; Value: integer);

    function GetAsString(attName: String): String;
    function GetAsBool(attName: String): boolean;
    function GetAsDateTime(attName: String): TDateTime;
    function GetAsTime(attName: String): TDateTime;
    function GetAsInt(attName: String): integer;
    function GetAsDate(attName: String): TDateTime;

    function NamedChilds(ChildName: String): IDataObjectList;

    procedure deleteChilds(ChildName: String);
    procedure RemoveChild(Child: TDataObject);

    function FindChild(ChildName: String): TDataObject;
    function getOrCreateChild(ChildName: String): TDataObject;

    procedure getKey(DataKey: TDataObjectKey);
    function findChildIndexByKey(DataKey: TDataObjectKey): integer;

    procedure Save(parentElem: IXMLNode);
    procedure Load(Elem: IXMLNode);

    class function NodesAreTheSame(Node1: TDataObject;
      Node2: TDataObject): boolean;
    class function ChildNodesAreTheSame(Node1: TDataObject; Node2: TDataObject;
      ChildNodeName: String): boolean;

  end;

  THomeDay = class(TObject)
  public
    NoDate: boolean;
    Date: TDateTime;
    KoppelDate: boolean;
    DoubleDate: boolean;
    KoppelPrio: integer;
    KoppelAuswaertsSecondTime: boolean;
    KoppelSecondTime: TDateTime;
    MaxParallelGame: integer;
    SperrTermin: boolean;
    AusweichTermin: boolean;
    Location: String;
  private
    procedure SplitString(Values: TStrings; Value: String);
    function TimeFromString(const dateString: String): TDateTime;
    procedure Clear;
    function DecodeLocation(Location: String): String;
    function EncodeLocation(Location: String): String;
  public
    function isTheSame(Value: THomeDay): boolean;

    function AsShortString(): String;
    function FromShortString(Date: TDateTime; Value: String): boolean;
    function IsHomeDay(): boolean;
    procedure assign(Source: THomeDay);
  end;

  THomeDayList = class(TObjectList<THomeDay>)
  public
    constructor Create();
  end;

  TPlanData = class(TObject)
  private
    rootElem: TDataObject;
    function findChildNode(Node: IXMLNode; const searchTag: String): IXMLNode;
    procedure loadTeamFromClickTT(teamNode: IXMLNode);
    procedure loadGameDaysFromClickTT(TeamElem: TDataObject; gameNode: IXMLNode;
      DoubleGameDaysFavoured: boolean);
    procedure loadHomeGameDayFromClickTT(TeamElem: TDataObject;
      dateNode: IXMLNode);
    procedure loadNoGameDayFromClickTT(TeamElem: TDataObject;
      dateNode: IXMLNode);
    procedure loadAuswaertsKoppelFromClickTT(TeamElem: TDataObject;
      Node: IXMLNode);
    procedure loadNoWeekGameFromClickTT(TeamElem: TDataObject; Node: IXMLNode);
    procedure loadSisterGameDaysFromClickTT(TeamElem: TDataObject;
      gameNode: IXMLNode);
    procedure loadSisterTeamFromClickTT(TeamElem: TDataObject;
      sisterTeamNode: IXMLNode);
    procedure LoadGameDatesFromClickTT(parentElem: TDataObject;
      gameNode: IXMLNode);
    procedure LoadNoGameDaysFromClickTT(noGameDaysNode: IXMLNode);
    procedure LoadMandatoryGameDaysFromClickTT(subNode: IXMLNode);
  public
    constructor Create();
    destructor Destroy(); override;
    procedure Clear();
    procedure assign(Source: TPlanData);
    procedure LoadFromClickTTFile(XMLFile: String);

    procedure Merge(Source: TPlanData);

    procedure Diff(PlanDataAfter, PlanDataBefore: TPlanData);

    function root(): TDataObject;

    procedure SaveToXML(XMLFile: String);
    procedure LoadFromXML(XMLFile: String);

    // Hilfsmethoden
    // Alle Teamnamen
    procedure getTeamNames(teamNames: TStrings);
    // Alle Tage, an denen es Wunschtermine gibt
    procedure getAllPossibleDates(DateList: TList<TDateTime>);

    // Alle Spiellokale einer Mannschaft
    procedure getTeamLocationsInt(ResultLocations: TStrings);
    function isValidLocation(Value: String): boolean;
    function FixLocation(Value: String): String;

    procedure GetHomeDays(TeamName: String; HomeDays: THomeDayList);
    procedure SetHomeDays(TeamName: String; HomeDays: THomeDayList);

  end;

function DateFromString(const dateString: String): TDateTime;
function TimeFromString(const dateString: String): TDateTime;
function IsClickTTFile(XMLFile: String): boolean;
function DateToString(Date: TDateTime): string;
function TimeToString(Date: TDateTime): string;
function FormatTimeForExport(Date: TDateTime): string;
procedure GetGenderList(Strs: TStrings);

implementation

procedure GetGenderList(Strs: TStrings);
begin
  Strs.Clear;
  Strs.Add('Herren');
  Strs.Add('Damen');
  Strs.Add('Jungen');
  Strs.Add('Jungen U18');
  Strs.Add('Mdchen');
  Strs.Add('Mdchen U18');
  Strs.Add('Schler');
  Strs.Add('Schlerinnen');
  Strs.Add('Bambini');
  Strs.Add('Senioren 40');
  Strs.Add('Senioren 50');
  Strs.Add('Senioren 60');
  Strs.Add('Senioren 70');
  Strs.Add('Seniorinnen 40');
  Strs.Add('Seniorinnen 50');
  Strs.Add('Seniorinnen 60');
  Strs.Add('Seniorinnen 70');
end;

function FormatTimeForExport(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'hh:mm', Date);
end;

function getKeysForDataType(Name: String; ResultValues: TStrings): boolean;
begin
  Result := False;

  ResultValues.Clear;

  if Name = nPlan then
  begin
    // Leer
    Result := True;
  end;

  if Name = nTeam then
  begin
    ResultValues.Add(aTeamName);
    Result := True;
  end;

  if Name = nHomeGameDay then
  begin
    ResultValues.Add(aDateTime);
    Result := True;
  end;

  if Name = nNoGameDay then
  begin
    ResultValues.Add(aDate);
    Result := True;
  end;

  if Name = nRoadCouple then
  begin
    ResultValues.Add(aTeamNameA);
    ResultValues.Add(aTeamNameB);
    Result := True;
  end;

  if Name = nNoWeekGames then
  begin
    ResultValues.Add(aTeamName);
    Result := True;
  end;

  if Name = nHomeRights then
  begin
    ResultValues.Add(aTeamName);
    Result := True;
  end;

  if Name = nSisterTeam then
  begin
    ResultValues.Add(aTeamName);
    ResultValues.Add(aGender);
    Result := True;
  end;

  if Name = nSisterGame then
  begin
    ResultValues.Add(aDateTime);
    Result := True;
  end;

  if Name = nPredefinedGames then
  begin
    // Nur ein Knoten
    Result := True;
  end;

  if Name = nGame then
  begin
    ResultValues.Add(aDateTime);
    ResultValues.Add(aHomeTeamName);
    ResultValues.Add(aGuestTeamName);
    Result := True;
  end;

  if Name = nExistingSchedule then
  begin
    Result := True;
  end;

  if Name = nSchedule then
  begin
    Result := True;
  end;

  if Name = nMandatoryGames then
  begin
    ResultValues.Add(aDateFrom);
    ResultValues.Add(aDateTo);
    Result := True;
  end;

  if Name = nRanking then
  begin
    Result := True;
  end;
end;

function getAttributValue(Node: IXMLNode; const searchName: String;
  const default: String): String;
begin
  Result := default;
  if Node.HasAttribute(searchName) then
  begin
    Result := Trim(VarToStr(Node.attributes[searchName]));
  end
end;

procedure setAttributValue(Node: IXMLNode; const Name: String;
  const Value: String);
begin
  Node.attributes[name] := Value;
end;

{ TDataObject }

procedure TDataObject.assign(Source: TDataObject);
var
  NewChild, Child: TDataObject;
begin
  attributes.Clear;
  childs.Clear;
  assignAttributes(Source);

  for Child in Source.childs do
  begin
    NewChild := TDataObject.Create(Child.Name, Self);
    NewChild.assign(Child);
  end;
end;

procedure TDataObject.assignAttributes(Source: TDataObject);
var
  Key: String;
begin
  attributes.Clear;
  for Key in Source.attributes.Keys do
  begin
    SetAsString(Key, Source.attributes[Key]);
  end;
end;

procedure TDataObject.Clear;
begin
  attributes.Clear;
  childs.Clear;
end;

function TDataObject.getOrCreateChild(ChildName: String): TDataObject;
begin
  Result := FindChild(ChildName);

  if not Assigned(Result) then
  begin
    Result := TDataObject.Create(ChildName, Self);
  end;
end;

function TDataObject.FindChild(ChildName: String): TDataObject;
var
  Child: TDataObject;
begin
  Result := nil;
  for Child in childs do
  begin
    if Child.Name = ChildName then
    begin
      Result := Child;
      break;
    end;
  end;
end;

function TDataObject.findChildIndexByKey(DataKey: TDataObjectKey): integer;
var
  i: integer;
  Child: TDataObject;
  AttributeValue: TPair<String, String>;
  Found: boolean;
begin
  Result := -1;
  for i := 0 to childs.Count - 1 do
  begin
    Child := childs[i];

    Found := False;

    if Child.Name = DataKey.Name then
    begin
      Found := True;
      for AttributeValue in DataKey.attributeValues do
      begin
        if AttributeValue.Value <> Child.GetAsString(AttributeValue.Key) then
        begin
          Found := False;
        end;
      end;
    end;

    if Found then
    begin
      if Result >= 0 then
      begin
        raise Exception.Create('Doppelter Schlssel in Datei: ' + DataKey.Name);
      end;
      Result := i;
    end;
  end;
end;

procedure TDataObject.getKey(DataKey: TDataObjectKey);
var
  Key: String;
  KeyValue: TPair<String, String>;
  Keys: TStrings;
begin
  DataKey.Name := Name;
  DataKey.attributeValues.Clear;

  Keys := TStringList.Create;

  getKeysForDataType(Name, Keys);

  for Key in Keys do
  begin
    KeyValue.Key := Key;
    KeyValue.Value := GetAsString(Key);
    DataKey.attributeValues.Add(KeyValue);
  end;

  Keys.Free;
end;

constructor TDataObject.Create(Name: String; parent: TDataObject);
var
  Dummy: TStrings;
begin
  inherited Create();
  State := dosNormal;
  Self.Name := name;

  Dummy := TStringList.Create;
  if not getKeysForDataType(name, Dummy) then
  begin
    raise Exception.Create('Kein Key fr den Typ: ' + name + ' definiert');
  end;
  Dummy.Free;

  attributes := TDictionary<String, String>.Create;
  childs := TObjectList<TDataObject>.Create;
  Self.parent := parent;

  if Assigned(parent) then
  begin
    parent.childs.Add(Self);
  end;

end;

procedure TDataObject.deleteChilds(ChildName: String);
var
  i: integer;
begin
  for i := childs.Count - 1 downto 0 do
  begin
    if childs[i].Name = ChildName then
    begin
      childs.Delete(i);
    end;
  end;
end;

destructor TDataObject.Destroy;
begin
  attributes.Free;
  childs.Free;
  inherited;
end;

function TDataObject.GetAsBool(attName: String): boolean;
begin
  Result := GetAsString(attName) = 'true';
end;

function TDataObject.GetAsInt(attName: String): integer;
begin
  Result := StrToIntDef(GetAsString(attName), 0);
end;

function DateFromString(const dateString: String): TDateTime;
var
  year, month, day: Word;
begin
  day := StrToInt(Copy(dateString, 1, 2));
  month := StrToInt(Copy(dateString, 4, 2));
  year := StrToInt(Copy(dateString, 7, 4));
  Result := EncodeDate(year, month, day);
end;

function TimeFromString(const dateString: String): TDateTime;
var
  hour, min: Word;
begin
  Result := 0;
  if Copy(dateString, 3, 1) = ':' then
  begin
    try
      hour := StrToInt(Copy(dateString, 1, 2));
      min := StrToInt(Copy(dateString, 4, 2));
      Result := EncodeTime(hour, min, 0, 0);
    Except
      Result := 0;
    end;
  end;
end;

function TDataObject.GetAsDate(attName: String): TDateTime;
var
  Value: String;
begin
  Result := 0;
  Value := GetAsString(attName);
  if Value <> '' then
  begin
    Result := DateFromString(Value);
  end;
end;

function TDataObject.GetAsDateTime(attName: String): TDateTime;
var
  Value: String;
  OnlyTime, DateTime: TDateTime;
begin
  Result := 0;
  Value := GetAsString(attName);
  if Value <> '' then
  begin
    OnlyTime := TimeFromString(Copy(Value, 12, 5));
    DateTime := DateFromString(Copy(Value, 1, 10));
    Result := DateTime + OnlyTime;
  end;
end;

function TDataObject.GetAsTime(attName: String): TDateTime;
var
  Value: String;
  OnlyTime: TDateTime;
begin
  Result := 0;
  Value := GetAsString(attName);
  if Value <> '' then
  begin
    OnlyTime := TimeFromString(Value);
    Result := OnlyTime;
  end;
end;

function TDataObject.GetAsString(attName: String): String;
begin
  Result := '';
  if attributes.ContainsKey(attName) then
  begin
    Result := attributes[attName];
  end;
end;

procedure TDataObject.Load(Elem: IXMLNode);
var
  i: integer;
  Key: String;
  NewElem: TDataObject;
  Value: String;
  valueInt: integer;
begin

  for i := 0 to Elem.AttributeNodes.Count - 1 do
  begin
    Key := Elem.AttributeNodes[i].NodeName;

    if Key = aState then
    begin
      Value := Elem.attributes[Key];
      valueInt := GetEnumValue(System.TypeInfo(TDataObjectState), Value);
      if valueInt >= 0 then
      begin
        State := TDataObjectState(valueInt);
      end;
    end
    else
    begin
      SetAsString(Key, Elem.attributes[Key]);
    end;
  end;

  for i := 0 to Elem.ChildNodes.Count - 1 do
  begin
    NewElem := TDataObject.Create(Elem.ChildNodes[i].NodeName, Self);
    NewElem.Load(Elem.ChildNodes[i]);
  end;
end;

procedure TDataObject.Merge(Source: TDataObject);
var
  Key: String;
  Child: TDataObject;
  DataKey: TDataObjectKey;
  DestIndex: integer;
  DestChild: TDataObject;
begin
  if Source.State = dosModified then
  begin
    for Key in Source.attributes.Keys do
    begin
      SetAsString(Key, Source.attributes[Key]);
    end;
  end;

  for Child in Source.childs do
  begin
    DataKey := TDataObjectKey.Create;
    Child.getKey(DataKey);
    DestIndex := findChildIndexByKey(DataKey);

    if Child.State = dosNew then
    begin
      if DestIndex >= 0 then
      begin
        childs.Delete(DestIndex);
      end;
      DestChild := TDataObject.Create(Child.Name, Self);
      DestChild.assign(Child);
    end
    else if Child.State = dosDelete then
    begin
      if DestIndex >= 0 then
      begin
        childs.Delete(DestIndex);
      end;
    end
    else
    begin
      if DestIndex >= 0 then
      begin
        DestChild := childs[DestIndex];
        DestChild.Merge(Child);
      end;
    end;
    DataKey.Free;
  end;
end;

function TDataObject.NamedChilds(ChildName: String): IDataObjectList;
var
  ResultObject: TInterfacedDataObjectList;
  Child: TDataObject;
begin
  ResultObject := TInterfacedDataObjectList.Create;

  for Child in childs do
  begin
    if Child.Name = ChildName then
    begin
      ResultObject.ListValue.Add(Child);
    end;
  end;

  Result := ResultObject;
end;

class function TDataObject.NodesAreTheSame(Node1, Node2: TDataObject): boolean;
begin
  if Node1 = Node2 then
  begin
    Result := True;
  end
  else if (Node1 = nil) or (Node2 = nil) then
  begin
    Result := False;
  end
  else
  begin
    Result := Node1.isTheSame(Node2);
  end;
end;

class function TDataObject.ChildNodesAreTheSame(Node1, Node2: TDataObject;
  ChildNodeName: String): boolean;
var
  childNode: TDataObject;
  DataKey: TDataObjectKey;
  Index: integer;
begin
  Result := True;
  if Node1 = Node2 then
  begin
    Result := True;
  end
  else if (Node1 = nil) or (Node2 = nil) then
  begin
    Result := False;
  end
  else
  begin
    for childNode in Node1.NamedChilds(ChildNodeName) do
    begin
      DataKey := TDataObjectKey.Create;
      childNode.getKey(DataKey);

      Index := Node2.findChildIndexByKey(DataKey);

      if Index >= 0 then
      begin
        if not childNode.isTheSame(Node2.childs[Index]) then
        begin
          Result := False;
        end;
      end
      else
      begin
        Result := False;
      end;
      DataKey.Free;
    end;

    for childNode in Node2.NamedChilds(ChildNodeName) do
    begin
      DataKey := TDataObjectKey.Create;
      childNode.getKey(DataKey);

      Index := Node1.findChildIndexByKey(DataKey);

      if Index >= 0 then
      begin
        if not childNode.isTheSame(Node1.childs[Index]) then
        begin
          Result := False;
        end;
      end
      else
      begin
        Result := False;
      end;
      DataKey.Free;
    end;

  end;

end;

procedure TDataObject.RemoveChild(Child: TDataObject);
begin
  childs.Remove(Child);
end;

function TDataObject.isTheSame(Node: TDataObject): boolean;
begin
  Result := isTheSameOneDirection(Node);
  if Result then
  begin
    Result := Node.isTheSameOneDirection(Self);
  end;
end;

function TDataObject.isTheSameOneDirection(Node: TDataObject): boolean;
var
  KeyValue: TPair<String, String>;
  Child: TDataObject;
  DataKey: TDataObjectKey;
  DestIndex: integer;
begin
  Result := True;

  for KeyValue in attributes do
  begin
    if GetAsString(KeyValue.Key) <> Node.GetAsString(KeyValue.Key) then
    begin
      Result := False;
    end;
  end;

  for Child in childs do
  begin
    DataKey := TDataObjectKey.Create;
    Child.getKey(DataKey);
    DestIndex := Node.findChildIndexByKey(DataKey);

    if DestIndex < 0 then
    begin
      Result := False;
    end
    else
    begin
      if not Child.isTheSame(Node.childs[DestIndex]) then
      begin
        Result := False;
      end;
    end;

    DataKey.Free;
  end;
end;

procedure TDataObject.Diff(PlanDataAfter, PlanDataBefore: TDataObject);
var
  Key: String;
  Child: TDataObject;
  DataKey: TDataObjectKey;
  DestIndex: integer;
  NewElem: TDataObject;
  KeyValue: TPair<String, String>;
begin
  attributes.Clear;
  childs.Clear;

  DataKey := TDataObjectKey.Create;
  PlanDataAfter.getKey(DataKey);
  for KeyValue in DataKey.attributeValues do
  begin
    SetAsString(KeyValue.Key, KeyValue.Value);
  end;
  DataKey.Free;

  for Key in PlanDataAfter.attributes.Keys do
  begin
    if PlanDataAfter.GetAsString(Key) <> PlanDataBefore.GetAsString(Key) then
    begin
      SetAsString(Key, PlanDataAfter.GetAsString(Key));
      Self.State := dosModified;
    end;
  end;

  for Key in PlanDataBefore.attributes.Keys do
  begin
    if PlanDataAfter.GetAsString(Key) <> PlanDataBefore.GetAsString(Key) then
    begin
      SetAsString(Key, PlanDataAfter.GetAsString(Key));
      Self.State := dosModified;
    end;
  end;

  // Alle gelschten weg
  for Child in PlanDataBefore.childs do
  begin
    DataKey := TDataObjectKey.Create;
    Child.getKey(DataKey);
    DestIndex := PlanDataAfter.findChildIndexByKey(DataKey);

    if DestIndex < 0 then
    begin
      NewElem := TDataObject.Create(Child.Name, Self);
      NewElem.assignAttributes(Child);
      NewElem.State := dosDelete;
    end;
    DataKey.Free;
  end;

  for Child in PlanDataAfter.childs do
  begin
    DataKey := TDataObjectKey.Create;
    Child.getKey(DataKey);
    DestIndex := PlanDataBefore.findChildIndexByKey(DataKey);

    if DestIndex < 0 then
    begin
      // Ein neues Element
      NewElem := TDataObject.Create(Child.Name, Self);
      NewElem.assign(Child);
      NewElem.State := dosNew;
    end
    else
    begin
      // War schon vorher bekannt -> Diff
      NewElem := TDataObject.Create(Child.Name, Self);
      NewElem.Diff(Child, PlanDataBefore.childs[DestIndex]);
    end;

    DataKey.Free;
  end;
end;

function DateToString(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'dd.mm.yyyy', Date);
end;

function DateWithTimeToString(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'dd.mm.yyyy hh:mm', Date);
end;

function TimeToString(Date: TDateTime): string;
begin
  DateTimeToString(Result, 'hh:mm', Date);
end;

function BoolToString(Value: boolean): string;
begin
  if Value then
  begin
    Result := 'true';
  end
  else
  begin
    Result := 'false';
  end;
end;

procedure TDataObject.Save(parentElem: IXMLNode);
var
  Node: IXMLNode;
  Key: String;
  Child: TDataObject;
begin
  Node := parentElem.AddChild(Name);

  if State <> dosNormal then
  begin
    setAttributValue(Node, aState,
      GetEnumName(System.TypeInfo(TDataObjectState), Ord(State)));
  end;

  for Key in attributes.Keys do
  begin
    setAttributValue(Node, Key, attributes[Key]);
  end;

  for Child in childs do
  begin
    Child.Save(Node);
  end;
end;

procedure TDataObject.SetAsBool(attName: String; Value: boolean);
begin
  SetAsString(attName, BoolToString(Value));
end;

procedure TDataObject.SetAsDate(attName: String; Value: TDateTime);
begin
  SetAsString(attName, DateToString(Value));
end;

procedure TDataObject.SetAsDateTime(attName: String; Value: TDateTime);
begin
  SetAsString(attName, DateWithTimeToString(Value));
end;

procedure TDataObject.SetAsTime(attName: String; Value: TDateTime);
begin
  SetAsString(attName, TimeToString(Value));
end;

procedure TDataObject.SetAsInt(attName: String; Value: integer);
begin
  SetAsString(attName, IntToStr(Value));
end;

procedure TDataObject.SetAsString(attName, Value: String);
begin
  Value := Trim(Value);
  if attributes.ContainsKey(attName) then
  begin
    attributes[attName] := Value;
  end
  else
  begin
    attributes.Add(attName, Value);
  end;

end;

{ TPlanData }

procedure TPlanData.assign(Source: TPlanData);
begin
  Clear();
  rootElem.assign(Source.rootElem);
end;

procedure TPlanData.Clear;
begin
  rootElem.Free;
  rootElem := TDataObject.Create(nPlan, nil);
end;

constructor TPlanData.Create;
begin
  inherited Create();

  rootElem := TDataObject.Create(nPlan, nil);
end;

destructor TPlanData.Destroy;
begin
  rootElem.Free;
  inherited;
end;

function TPlanData.findChildNode(Node: IXMLNode; const searchTag: String)
  : IXMLNode;
var
  i: integer;
begin
  Result := nil;
  for i := 0 to Node.ChildNodes.Count - 1 do
  begin
    if Node.ChildNodes[i].NodeName = searchTag then
    begin
      Result := Node.ChildNodes[i];
      break;
    end;
  end;
end;

procedure TPlanData.getAllPossibleDates(DateList: TList<TDateTime>);
var
  teamNode: TDataObject;
  WunschterminNode: TDataObject;
  Date: TDateTime;
begin
  DateList.Clear;
  for teamNode in rootElem.NamedChilds(nTeam) do
  begin
    for WunschterminNode in teamNode.NamedChilds(nHomeGameDay) do
    begin
      Date := Trunc(WunschterminNode.GetAsDateTime(aDateTime));
      if 0 > DateList.IndexOf(Date) then
      begin
        DateList.Add(Date);
      end;
    end;
  end;

  DateList.Sort;

end;

function TPlanData.isValidLocation(Value: String): boolean;
var
  TempString: TStrings;
begin
  TempString := TStringList.Create;
  getTeamLocationsInt(TempString);
  Result := TempString.IndexOf(Value) >= 0;
  TempString.Free;
end;

function TPlanData.FixLocation(Value: String): String;
begin
  Result := Trim(Value);
  if not isValidLocation(Result) then
  begin
    Result := '';
  end;
end;

procedure TPlanData.getTeamLocationsInt(ResultLocations: TStrings);
var
  TempString: TStringList;
  i: integer;
begin
  TempString := TStringList.Create;
  TempString.Duplicates := dupIgnore;
  TempString.Sorted := True;
  TempString.Add('');

  // Click-tt kann nur Numerische Spiellokale, deshalb einfach die Liste mit 1-5 fllen
  for i := 1 to 5 do
  begin
    TempString.Add(IntToStr(i));
  end;

  (*
    for TeamNode in root.NamedChilds(nTeam) do
    begin
    if teamName = TeamNode.GetAsString(aTeamName) then
    begin
    TempString.Add(TeamNode.GetAsString(aLocationInt));

    for HomeGameNode in TeamNode.NamedChilds(nHomeGameDay) do
    begin
    TempString.Add(HomeGameNode.GetAsString(aLocationInt));
    end;

    for SisterTeamNode in TeamNode.NamedChilds(nSisterTeam) do
    begin
    TempString.Add(SisterTeamNode.GetAsString(aLocationInt));

    for SisterGameNode in SisterTeamNode.NamedChilds(nSisterGame) do
    begin
    TempString.Add(SisterGameNode.GetAsString(aLocationInt));
    end;
    end;
    end;
    end;
  *)

  ResultLocations.assign(TempString);
  TempString.Free;
end;

procedure TPlanData.GetHomeDays(TeamName: String; HomeDays: THomeDayList);
var
  teamNode: TDataObject;
  HomeDayNode: TDataObject;
  HomeDay: THomeDay;
  SperrTermine: TList<TDateTime>;
begin
  HomeDays.Clear;

  SperrTermine := TList<TDateTime>.Create;

  for teamNode in root.NamedChilds(nTeam) do
  begin
    if teamNode.GetAsString(aTeamName) = TeamName then
    begin
      for HomeDayNode in teamNode.NamedChilds(nNoGameDay) do
      begin
        HomeDay := THomeDay.Create;
        HomeDays.Add(HomeDay);
        HomeDay.Date := HomeDayNode.GetAsDate(aDate);
        HomeDay.SperrTermin := True;
        SperrTermine.Add(HomeDay.Date);
      end;

      for HomeDayNode in teamNode.NamedChilds(nHomeGameDay) do
      begin
        HomeDay := THomeDay.Create;

        HomeDay.Date := HomeDayNode.GetAsDateTime(aDateTime);
        HomeDay.KoppelDate := HomeDayNode.GetAsBool(aCoupleGameDay);

        if not HomeDay.KoppelDate then
        begin
          HomeDay.DoubleDate := HomeDayNode.GetAsBool(aDoubleGameDay);
        end
        else
        begin
          HomeDay.KoppelSecondTime := HomeDayNode.GetAsTime(aCoupleSecondTime);
        end;

        if not HomeDay.KoppelDate then
        begin
          HomeDay.KoppelAuswaertsSecondTime :=
            HomeDayNode.GetAsBool(aCoupleAuswaertsSecondTime);
          if HomeDay.KoppelAuswaertsSecondTime then
          begin
            HomeDay.KoppelSecondTime :=
              HomeDayNode.GetAsTime(aCoupleSecondTime);
          end;
        end;

        if HomeDay.KoppelDate or HomeDay.DoubleDate then
        begin
          HomeDay.KoppelPrio := HomeDayNode.GetAsInt(aCouplePrio);
        end;

        HomeDay.MaxParallelGame := HomeDayNode.GetAsInt(aParallelGames);
        HomeDay.AusweichTermin := HomeDayNode.GetAsBool(aAusweichTermin);
        HomeDay.Location := FixLocation(HomeDayNode.GetAsString(aLocation));

        if SperrTermine.Contains(Trunc(HomeDay.Date)) then
        begin
          // An einem Sperrtermin kann kein normales Spiel sein
          HomeDay.Free;
        end
        else
        begin
          HomeDays.Add(HomeDay);
        end;

      end;
    end;
  end;
  SperrTermine.Free;

  HomeDays.Sort;
end;

procedure TPlanData.getTeamNames(teamNames: TStrings);
var
  S: String;
  teamNode: TDataObject;
  teams: TList<String>;
begin
  teamNames.Clear;

  teams := TList<String>.Create;

  for teamNode in rootElem.NamedChilds(nTeam) do
  begin
    teams.Add(teamNode.GetAsString(aTeamName));
  end;

  teams.Sort;

  for S in teams do
  begin
    teamNames.Add(S);
  end;

  teams.Free;
end;

procedure TPlanData.loadHomeGameDayFromClickTT(TeamElem: TDataObject;
  dateNode: IXMLNode);
var
  Date, time: string;
  DateTime, OnlyTime: TDateTime;
  NewElem: TDataObject;
  coupleValue: String;
  Key: TDataObjectKey;
begin
  Date := getAttributValue(dateNode, 'day', '');
  time := getAttributValue(dateNode, 'time', '');

  if (Date <> '') and (time <> '') then
  begin
    OnlyTime := TimeFromString(time);
    DateTime := DateFromString(Date) + OnlyTime;

    Key := TDataObjectKey.Create;
    Key.Name := nHomeGameDay;
    Key.setAttributeAsDateTime(aDateTime, DateTime);

    if 0 > TeamElem.findChildIndexByKey(Key) then
    begin
      NewElem := TDataObject.Create(nHomeGameDay, TeamElem);
      NewElem.SetAsDateTime(aDateTime, DateTime);
      NewElem.SetAsBool(aCoupleGameDay, False);
      NewElem.SetAsBool(aCoupleAuswaertsSecondTime, False);
      NewElem.SetAsBool(aDoubleGameDay, False);
      NewElem.SetAsBool(aAusweichTermin, ('true' = getAttributValue(dateNode, 'secondary', 'false')));
      NewElem.SetAsString(aLocation, '');

      NewElem.SetAsInt(aParallelGames,
        StrToInt(getAttributValue(dateNode, 'maxParallelGames', '0')));

      coupleValue := getAttributValue(dateNode, 'coupleGame', '');
      if coupleValue <> '' then
      begin
        NewElem.SetAsBool(aCoupleGameDay, True);
        if coupleValue = 'soft' then
        begin
          NewElem.SetAsInt(aCouplePrio, 1);
        end
        else
        begin
          NewElem.SetAsInt(aCouplePrio, 2);
        end;

        // Zweite Zeit einfgen
        OnlyTime := DateTime - Trunc(DateTime);
        if OnlyTime > EncodeTime(16, 59, 0, 0) then
        begin
          // Der Koppeltermin ist spt, also muss das 2. Spiel davor sein
          OnlyTime := FixDateTime(OnlyTime - EncodeTime(4, 0, 0, 0));
        end
        else
        begin
          OnlyTime := FixDateTime(OnlyTime + EncodeTime(4, 0, 0, 0));
        end;
        NewElem.SetAsTime(aCoupleSecondTime, OnlyTime);
      end;
    end;
    Key.Free;
  end;
end;

procedure TPlanData.loadNoGameDayFromClickTT(TeamElem: TDataObject;
  dateNode: IXMLNode);
var
  fromDate, toDate: string;
  fromDateTime, toDateTime, i: TDateTime;
  NewElem: TDataObject;
  Key: TDataObjectKey;
begin
  fromDate := getAttributValue(dateNode, 'from', '');
  toDate := getAttributValue(dateNode, 'until', '');

  if (fromDate <> '') and (toDate <> '') then
  begin
    fromDateTime := DateFromString(fromDate);
    toDateTime := DateFromString(toDate);

    i := Trunc(fromDateTime);
    while i <= toDateTime do
    begin
      i := Trunc(i);

      Key := TDataObjectKey.Create;
      Key.Name := nNoGameDay;
      Key.setAttributeAsDate(aDate, Trunc(i));

      // Test, ob doppelt
      if 0 > TeamElem.findChildIndexByKey(Key) then
      begin
        NewElem := TDataObject.Create(nNoGameDay, TeamElem);
        NewElem.SetAsDate(aDate, Trunc(i));
      end;
      i := i + 1.0;
      Key.Free;
    end;
  end;
end;

procedure TPlanData.LoadNoGameDaysFromClickTT(noGameDaysNode: IXMLNode);
var
  i: integer;
  Node: IXMLNode;
  fromDateTime, toDateTime, j: TDateTime;
  NewElem: TDataObject;
begin
  for i := 0 to noGameDaysNode.ChildNodes.Count - 1 do
  begin
    Node := noGameDaysNode.ChildNodes[i];
    if Node.NodeName = 'noGame' then
    begin
      fromDateTime := DateFromString(getAttributValue(Node, 'from', ''));
      toDateTime := DateFromString(getAttributValue(Node, 'until', ''));

      j := fromDateTime;
      while j <= toDateTime do
      begin
        NewElem := TDataObject.Create(nNoGameDay, rootElem);
        NewElem.SetAsDate(aDate, Trunc(j));
        j := j + 1.0;
      end;
    end;
  end;
end;

procedure TPlanData.loadAuswaertsKoppelFromClickTT(TeamElem: TDataObject;
  Node: IXMLNode);
var
  NewValue: TDataObject;
  sameDayValue: integer;
begin
  NewValue := TDataObject.Create(nRoadCouple, TeamElem);
  NewValue.SetAsString(aTeamNameA, getAttributValue(Node, 'teamA', ''));
  NewValue.SetAsString(aTeamNameB, getAttributValue(Node, 'teamB', ''));
  sameDayValue := 1;
  if ('true' = getAttributValue(Node, 'onSameDay', 'false')) then
  begin
    sameDayValue := 0;
  end;

  NewValue.SetAsInt(aSameDay, sameDayValue);
end;

procedure TPlanData.loadNoWeekGameFromClickTT(TeamElem: TDataObject;
  Node: IXMLNode);
var
  NewValue: TDataObject;
begin
  var
    TeamList: TStrings := TStringList.Create;

  // Gegner sind manchmal doppelt drin.

  for var noWaekNode in TeamElem.NamedChilds(nNoWeekGames) do
  begin
    TeamList.Add(noWaekNode.GetAsString(aTeamName));
  end;

  var
    TeamName: string := getAttributValue(Node, 'team', '');

  if TeamList.IndexOf(TeamName) < 0 then
  begin
    NewValue := TDataObject.Create(nNoWeekGames, TeamElem);
    NewValue.SetAsString(aTeamName, TeamName);
  end;
  TeamList.Free;
end;

procedure TPlanData.loadGameDaysFromClickTT(TeamElem: TDataObject;
  gameNode: IXMLNode; DoubleGameDaysFavoured: boolean);
var
  Day1, Day2: TDataObject;
  i: integer;
begin
  for i := 0 to gameNode.ChildNodes.Count - 1 do
  begin
    if gameNode.ChildNodes[i].NodeName = 'homeGame' then
    begin
      loadHomeGameDayFromClickTT(TeamElem, gameNode.ChildNodes[i]);
    end;
    if gameNode.ChildNodes[i].NodeName = 'noGame' then
    begin
      loadNoGameDayFromClickTT(TeamElem, gameNode.ChildNodes[i]);
    end;
    if gameNode.ChildNodes[i].NodeName = 'roadCouple' then
    begin
      loadAuswaertsKoppelFromClickTT(TeamElem, gameNode.ChildNodes[i]);
    end;
    if gameNode.ChildNodes[i].NodeName = 'noWeekGame' then
    begin
      loadNoWeekGameFromClickTT(TeamElem, gameNode.ChildNodes[i]);
    end;
  end;

  if DoubleGameDaysFavoured then
  begin
    for Day1 in TeamElem.NamedChilds(nHomeGameDay) do
    begin
      if (not Day1.GetAsBool(aCoupleGameDay)) and
        (not Day1.GetAsBool(aDoubleGameDay)) then
      begin
        for Day2 in TeamElem.NamedChilds(nHomeGameDay) do
        begin
          if Day2 <> Day1 then
          begin
            if (not Day2.GetAsBool(aCoupleGameDay)) and
              (not Day2.GetAsBool(aDoubleGameDay)) then
            begin
              if 1 = Trunc(Day2.GetAsDateTime(aDateTime)) -
                Trunc(Day1.GetAsDateTime(aDateTime)) then
              begin
                // Zwei Spieltage hintereinander, die noch keine Koppeltermine sind -> mache die zu einem Doppelspieltag
                Day1.SetAsBool(aDoubleGameDay, True);
                Day2.SetAsBool(aDoubleGameDay, True);
                Day1.SetAsInt(aCouplePrio, 1);
                Day2.SetAsInt(aCouplePrio, 1);
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;

procedure TPlanData.loadSisterGameDaysFromClickTT(TeamElem: TDataObject;
  gameNode: IXMLNode);
var
  i: integer;
begin
  for i := 0 to gameNode.ChildNodes.Count - 1 do
  begin
    if gameNode.ChildNodes[i].NodeName = 'team' then
    begin
      loadSisterTeamFromClickTT(TeamElem, gameNode.ChildNodes[i]);
    end;
  end;
end;

procedure TPlanData.loadSisterTeamFromClickTT(TeamElem: TDataObject;
  sisterTeamNode: IXMLNode);
var
  nameNode: IXMLNode;
  gameNode: IXMLNode;
  i: integer;
  day: TDateTime;
  time: TDateTime;
  homeTeam, guestTeam: String;
  Name: string;
  SisterTeamElem: TDataObject;
  GameElem: TDataObject;
  gender: String;
  DataKey: TDataObjectKey;
  DestIndex: integer;
begin
  nameNode := findChildNode(sisterTeamNode, 'name');
  name := Trim(nameNode.text);
  gender := getAttributValue(sisterTeamNode, 'gender', '');

  DataKey := TDataObjectKey.Create;
  DataKey.Name := nSisterTeam;
  DataKey.setAttribute(aTeamName, name);
  DataKey.setAttribute(aGender, gender);

  DestIndex := TeamElem.findChildIndexByKey(DataKey);

  DataKey.Free;

  if DestIndex >= 0 then
  begin
    SisterTeamElem := TeamElem.childs[DestIndex];
  end
  else
  begin
    SisterTeamElem := TDataObject.Create(nSisterTeam, TeamElem);
  end;

  SisterTeamElem.SetAsString(aTeamName, name);
  SisterTeamElem.SetAsString(aGender, gender);
  SisterTeamElem.SetAsString(aLocation, '');
  SisterTeamElem.SetAsInt(aTeamNumber, StrToInt(getAttributValue(sisterTeamNode,
    'number', '0')));
  SisterTeamElem.SetAsBool(aParallelHomeGames, False);
  SisterTeamElem.SetAsBool(aNoParallelGames, False);

  if (gender = root.GetAsString(aGender)) and
    (1 = Abs(TeamElem.GetAsInt(aTeamNumber) - SisterTeamElem.GetAsInt
    (aTeamNumber))) then
  begin
    // Eine Nachbarmannschaft -> parallele Spiele vermeiden
    SisterTeamElem.SetAsBool(aNoParallelGames, True);
  end;

  for i := 0 to sisterTeamNode.ChildNodes.Count - 1 do
  begin
    if sisterTeamNode.ChildNodes[i].NodeName = 'game' then
    begin
      gameNode := sisterTeamNode.ChildNodes[i];
      day := DateFromString(getAttributValue(gameNode, 'day', ''));
      time := TimeFromString(getAttributValue(gameNode, 'time', ''));
      homeTeam := getAttributValue(gameNode, 'homeTeam', '');
      guestTeam := getAttributValue(gameNode, 'guestTeam', '');

      if (homeTeam <> 'spielfrei') and (guestTeam <> 'spielfrei') then
      begin
        DataKey := TDataObjectKey.Create;
        DataKey.Name := nSisterGame;
        DataKey.setAttribute(aDateTime, DateWithTimeToString(day + time));
        if 0 <= SisterTeamElem.findChildIndexByKey(DataKey) then
        begin
          // Fehler in der Click-TT XML
          // Passiert wegen dem Fehler mit den Bambinis
        end
        else
        begin
          GameElem := TDataObject.Create(nSisterGame, SisterTeamElem);
          GameElem.SetAsDateTime(aDateTime, day + time);
          GameElem.SetAsString(aHomeTeamName, homeTeam);
          GameElem.SetAsString(aGuestTeamName, guestTeam);
        end;
        DataKey.Free;
      end;
    end;
  end;
end;

procedure TPlanData.loadTeamFromClickTT(teamNode: IXMLNode);
var
  nameNode: IXMLNode;
  gameNode: IXMLNode;
  sisterNode: IXMLNode;
  TeamElem: TDataObject;
  noDoubleDays: boolean;
begin
  TeamElem := TDataObject.Create(nTeam, rootElem);
  TeamElem.SetAsString(aClubID, getAttributValue(teamNode, 'clubId', ''));
  TeamElem.SetAsString(aTeamId, getAttributValue(teamNode, 'id', ''));
  TeamElem.SetAsString(aLocation, '');
  TeamElem.SetAsInt(aHomeRights, -1);
  TeamElem.SetAsInt(aTeamNumber, StrToInt(getAttributValue(teamNode,
    'number', '1')));
  noDoubleDays := 'true' = getAttributValue(teamNode, 'noDoubleDays', 'true');

  nameNode := findChildNode(teamNode, 'name');
  if Assigned(nameNode) then
  begin
    TeamElem.SetAsString(aTeamName, nameNode.text);
  end;

  gameNode := findChildNode(teamNode, 'gameDays');
  if Assigned(gameNode) then
  begin
    loadGameDaysFromClickTT(TeamElem, gameNode, not noDoubleDays);
  end;

  sisterNode := findChildNode(teamNode, 'sisterTeams');
  if Assigned(sisterNode) then
  begin
    loadSisterGameDaysFromClickTT(TeamElem, sisterNode);
  end;
end;

procedure TPlanData.Merge(Source: TPlanData);
begin
  rootElem.Merge(Source.rootElem);
end;

procedure TPlanData.Diff(PlanDataAfter, PlanDataBefore: TPlanData);
begin
  rootElem.Diff(PlanDataAfter.rootElem, PlanDataBefore.rootElem);
end;

function TPlanData.root: TDataObject;
begin
  Result := rootElem;
end;

procedure TPlanData.LoadGameDatesFromClickTT(parentElem: TDataObject;
  gameNode: IXMLNode);
var
  i: integer;
  day, time, Date: TDateTime;
  homeTeam, guestTeam: String;
  Node: IXMLNode;
  GameElem: TDataObject;
begin
  for i := 0 to gameNode.ChildNodes.Count - 1 do
  begin
    Node := gameNode.ChildNodes[i];
    if Node.NodeName = 'game' then
    begin
      day := DateFromString(getAttributValue(Node, 'day', ''));
      time := TimeFromString(getAttributValue(Node, 'time', ''));
      homeTeam := getAttributValue(Node, 'homeTeam', '');
      guestTeam := getAttributValue(Node, 'guestTeam', '');

      Date := day + time;

      GameElem := TDataObject.Create(nGame, parentElem);
      GameElem.SetAsDateTime(aDateTime, Date);
      GameElem.SetAsString(aHomeTeamName, homeTeam);
      GameElem.SetAsString(aGuestTeamName, guestTeam);
      GameElem.SetAsString(aLocation, '');
    end;
  end;
end;

procedure TPlanData.LoadMandatoryGameDaysFromClickTT(subNode: IXMLNode);
var
  i: integer;
  Node: IXMLNode;
  Date: TDateTime;
  MandatoryElem: TDataObject;
  MandatoryGameDaysInXML: TList<TDateTime>;
  LastDate: TDateTime;

begin
  MandatoryGameDaysInXML := TList<TDateTime>.Create;
  for i := 0 to subNode.ChildNodes.Count - 1 do
  begin
    Node := subNode.ChildNodes[i];
    if Node.NodeName = 'mandatoryDay' then
    begin
      Date := DateFromString(getAttributValue(Node, 'day', ''));
      MandatoryGameDaysInXML.Add(Date);
    end;
  end;

  MandatoryGameDaysInXML.Sort;
  LastDate := 0;
  MandatoryElem := nil;
  for Date in MandatoryGameDaysInXML do
  begin
    if Assigned(MandatoryElem) and (Trunc(Date - LastDate) = 1) then
    begin
      MandatoryElem.SetAsDate(aDateTo, Date);
    end
    else
    begin
      MandatoryElem := TDataObject.Create(nMandatoryGames, rootElem);
      MandatoryElem.SetAsDate(aDateFrom, Date);
      MandatoryElem.SetAsDate(aDateTo, Date);
      MandatoryElem.SetAsInt(aNumberGames, 1);
    end;
    LastDate := Date;
  end;
  MandatoryGameDaysInXML.Free;
end;

procedure TPlanData.LoadFromClickTTFile(XMLFile: String);
var
  predefinedGamesNode, existingScheduleNode, mainNode, teamNode,
    subNode: IXMLNode;
  document: IXMLDocument;
  i: integer;
  GameElem: TDataObject;
begin
  Clear();
  document := LoadXMLDocument(XMLFile);

  mainNode := document.DocumentElement;

  rootElem.SetAsString(aGender, getAttributValue(mainNode, 'gender', ''));

  rootElem.SetAsDate(aMid, DateFromString(getAttributValue(mainNode,
    'mid', '')));
  rootElem.SetAsDate(aFrom, DateFromString(getAttributValue(mainNode,
    'from', '')));
  rootElem.SetAsDate(aUntil, DateFromString(getAttributValue(mainNode,
    'until', '')));
  rootElem.SetAsString(aName, getAttributValue(mainNode, 'name', ''));
  rootElem.SetAsString(aID, getAttributValue(mainNode, 'id', ''));

  if Assigned(mainNode) then
  begin
    teamNode := findChildNode(mainNode, 'teams');
    if Assigned(teamNode) then
    begin
      for i := 0 to teamNode.ChildNodes.Count - 1 do
      begin
        if teamNode.ChildNodes[i].NodeName = 'team' then
        begin
          loadTeamFromClickTT(teamNode.ChildNodes[i]);
        end;
      end;
    end;
  end;

  predefinedGamesNode := findChildNode(mainNode, 'predefinedGames');
  if Assigned(predefinedGamesNode) then
  begin
    if predefinedGamesNode.ChildNodes.Count > 0 then
    begin
      GameElem := TDataObject.Create(nPredefinedGames, rootElem);
      LoadGameDatesFromClickTT(GameElem, predefinedGamesNode);
    end;
  end;

  subNode := findChildNode(mainNode, 'noGameDays');
  if Assigned(subNode) then
  begin
    LoadNoGameDaysFromClickTT(subNode);
  end;

  subNode := findChildNode(mainNode, 'mandatoryGameDays');
  if Assigned(subNode) then
  begin
    LoadMandatoryGameDaysFromClickTT(subNode);
  end;

  existingScheduleNode := findChildNode(mainNode, 'existingSchedule');
  if Assigned(existingScheduleNode) then
  begin
    if existingScheduleNode.ChildNodes.Count > 0 then
    begin
      GameElem := TDataObject.Create(nExistingSchedule, rootElem);
      LoadGameDatesFromClickTT(GameElem, existingScheduleNode);
    end;
  end;

end;

procedure TPlanData.SaveToXML(XMLFile: String);
var
  XMLDoc: IXMLDocument;
  root: IXMLNode;
begin
  XMLDoc := TXMLDocument.Create(nil);
  XMLDoc.Active := True;
  XMLDoc.Version := '1.0';
  XMLDoc.Encoding := 'UTF-8';

  root := XMLDoc.AddChild('andigenerator');

  rootElem.Save(root);

  try
    XMLDoc.SaveToFile(XMLFile);
  except
    on E: Exception do
    begin
      ShowMessage('Datei ' + XMLFile + ' kann nicht geschrieben werden. Grund: '
        + E.Message);
    end;
  end;
end;

procedure TPlanData.SetHomeDays(TeamName: String; HomeDays: THomeDayList);
var
  HomeDay: THomeDay;
  NewElem: TDataObject;
  teamNode: TDataObject;
begin
  for teamNode in root.NamedChilds(nTeam) do
  begin
    if teamNode.GetAsString(aTeamName) = TeamName then
    begin
      teamNode.deleteChilds(nHomeGameDay);
      teamNode.deleteChilds(nNoGameDay);

      for HomeDay in HomeDays do
      begin
        if HomeDay.NoDate then
        begin
        end
        else if HomeDay.SperrTermin then
        begin
          NewElem := TDataObject.Create(nNoGameDay, teamNode);
          NewElem.SetAsDate(aDate, HomeDay.Date);
        end
        else
        begin
          NewElem := TDataObject.Create(nHomeGameDay, teamNode);
          NewElem.SetAsDateTime(aDateTime, HomeDay.Date);
          NewElem.SetAsInt(aParallelGames, HomeDay.MaxParallelGame);
          NewElem.SetAsBool(aAusweichTermin, HomeDay.AusweichTermin);
          NewElem.SetAsString(aLocation, FixLocation(HomeDay.Location));
          NewElem.SetAsBool(aDoubleGameDay, HomeDay.DoubleDate);
          NewElem.SetAsBool(aCoupleGameDay, HomeDay.KoppelDate);
          NewElem.SetAsBool(aCoupleAuswaertsSecondTime,
            HomeDay.KoppelAuswaertsSecondTime);

          if HomeDay.KoppelDate then
          begin
            NewElem.SetAsTime(aCoupleSecondTime, HomeDay.KoppelSecondTime);
          end;

          if HomeDay.KoppelDate or HomeDay.DoubleDate then
          begin
            NewElem.SetAsInt(aCouplePrio, HomeDay.KoppelPrio);
          end;

          if HomeDay.KoppelAuswaertsSecondTime then
          begin
            NewElem.SetAsTime(aCoupleSecondTime, HomeDay.KoppelSecondTime);
          end;

        end;
      end;
    end;
  end;
end;

procedure TPlanData.LoadFromXML(XMLFile: String);
var
  mainNode: IXMLNode;
  document: IXMLDocument;
  i: integer;
begin
  Clear();
  document := LoadXMLDocument(XMLFile);

  mainNode := document.DocumentElement;

  for i := 0 to mainNode.ChildNodes.Count - 1 do
  begin
    if mainNode.ChildNodes[i].NodeName = rootElem.Name then
    begin
      rootElem.Load(mainNode.ChildNodes[i]);
      break;
    end;
  end;
end;

function IsClickTTFile(XMLFile: String): boolean;
var
  mainNode: IXMLNode;
  document: IXMLDocument;
begin
  document := LoadXMLDocument(XMLFile);
  mainNode := document.DocumentElement;

  Result := False;
  if mainNode.NodeName = 'TT' then
  begin
    Result := True;
  end;
end;

{ TDataObjectKey }

constructor TDataObjectKey.Create;
begin
  inherited Create;
  attributeValues := TList < TPair < String, String >>.Create;
end;

destructor TDataObjectKey.Destroy;
begin
  attributeValues.Free;
  inherited;
end;

procedure TDataObjectKey.setAttribute(Key, Value: String);
var
  Values: TPair<String, String>;
begin
  Values.Key := Key;
  Values.Value := Value;
  attributeValues.Add(Values);
end;

procedure TDataObjectKey.setAttributeAsDate(Key: String; Date: TDateTime);
begin
  setAttribute(Key, DateToString(Date));

end;

procedure TDataObjectKey.setAttributeAsDateTime(Key: String; Date: TDateTime);
begin
  setAttribute(Key, DateWithTimeToString(Date));
end;

{ TInterfacedDataObjectList }

constructor TInterfacedDataObjectList.Create;
begin
  ListValue := TList<TDataObject>.Create;
end;

destructor TInterfacedDataObjectList.Destroy;
begin
  ListValue.Free;
  inherited;
end;

function TInterfacedDataObjectList.GetEnumerator: TEnumerator<TDataObject>;
begin
  Result := ListValue.GetEnumerator;
end;

{ THomeDay }

procedure THomeDay.SplitString(Values: TStrings; Value: String);
var
  i: integer;
  PartValue: String;
  InMarks: boolean;
begin
  Values.Clear;

  InMarks := False;

  i := 1;
  while i <= Length(Value) do
  begin
    if Value[i] = '"' then
    begin
      InMarks := not InMarks;
    end;

    if ((Value[i] = ' ') and (not InMarks)) or (i = Length(Value)) then
    begin
      PartValue := Trim(Copy(Value, 1, i));
      Value := Trim(Copy(Value, i + 1, Length(Value)));
      if PartValue <> '' then
      begin
        Values.Add(PartValue);
      end;
      i := 1;
    end
    else
    begin
      Inc(i);
    end;
  end;
end;

// Geschtzes Leerzeichen anstatt dem normalen
function THomeDay.EncodeLocation(Location: String): String;
begin
  Result := Location.Replace('"', '');

  if Result.Contains(' ') then
  begin
    Result := '"' + Result + '"';
  end;

end;

// Normales Leerzeichen anstatt dem geschtzen
function THomeDay.DecodeLocation(Location: String): String;
begin
  Result := Location;
  if Result.StartsWith('"') and Result.EndsWith('"') then
  begin
    Result := Copy(Result, 2, Length(Result) - 2);
  end;

  Result := Result.Replace('', '"');
end;

function THomeDay.AsShortString: String;
begin
  if NoDate then
  begin
    Result := '';
  end
  else if SperrTermin then
  begin
    Result := 'FREI';
  end
  else
  begin
    Result := FormatTimeForExport(Date);

    if AusweichTermin then
    begin
      Result := Result + ' A';
    end;

    if KoppelDate then
    begin
      Result := Result + ' K' + IntToStr(KoppelPrio) + ':' +
        FormatTimeForExport(KoppelSecondTime);
    end
    else if KoppelAuswaertsSecondTime then
    begin
      Result := Result + ' T2:' + FormatTimeForExport(KoppelSecondTime);
    end;

    if DoubleDate then
    begin
      Result := Result + ' D' + IntToStr(KoppelPrio);
    end;

    if (MaxParallelGame > 0) then
    begin
      Result := Result + ' Max:' + IntToStr(MaxParallelGame);
    end;

    if Location <> '' then
    begin
      Result := Result + ' L:' + EncodeLocation(Location);
    end;

  end;
end;

function THomeDay.TimeFromString(const dateString: String): TDateTime;
var
  hour, min: integer;
begin
  Result := -1;
  if Copy(dateString, 3, 1) = ':' then
  begin
    hour := StrToIntDef(Copy(dateString, 1, 2), -1);
    min := StrToIntDef(Copy(dateString, 4, 2), -1);

    if (hour >= 0) and (min >= 0) then
    begin
      Result := EncodeTime(hour, min, 0, 0);
    end;
  end;
end;

procedure THomeDay.assign(Source: THomeDay);
begin
  Date := Source.Date;
  KoppelDate := Source.KoppelDate;
  KoppelAuswaertsSecondTime := Source.KoppelAuswaertsSecondTime;
  DoubleDate := Source.DoubleDate;
  KoppelPrio := Source.KoppelPrio;
  KoppelSecondTime := Source.KoppelSecondTime;
  MaxParallelGame := Source.MaxParallelGame;
  AusweichTermin := Source.AusweichTermin;
  Location := Source.Location;
  SperrTermin := Source.SperrTermin;
  NoDate := Source.NoDate;
end;

procedure THomeDay.Clear();
begin
  Date := 0;
  KoppelDate := False;
  KoppelAuswaertsSecondTime := False;
  DoubleDate := False;
  KoppelPrio := 0;
  KoppelSecondTime := 0;
  MaxParallelGame := 0;
  AusweichTermin := False;
  SperrTermin := False;
  NoDate := False;
  Location := '';
end;

function THomeDay.FromShortString(Date: TDateTime; Value: String): boolean;
var
  time: TDateTime;
  Values: TStrings;
  S: String;
  Prio: integer;
  MaxGames: integer;
begin
  Self.Clear;
  Value := Trim(Value);

  Values := TStringList.Create;

  SplitString(Values, Value);

  Result := False;

  if Trim(Value) = '' then
  begin
    NoDate := True;
  end;

  for S in Values do
  begin
    if S <> '' then
    begin
      if UpperCase(S) = 'FREI' then
      begin
        SperrTermin := True;
        Result := True;
      end
      else
      begin
        time := TimeFromString(S);
        if time >= 0 then
        begin
          Result := True;
          Self.Date := FixDateTime(Date + time);
        end;

        if UpperCase(S) = 'A' then
        begin
          Self.AusweichTermin := True;
        end;

        if UpperCase(S).StartsWith('L:') then
        begin
          Location := DecodeLocation(Copy(S, 3, Length(S)));
        end;

        if (not Self.DoubleDate) and (not Self.KoppelAuswaertsSecondTime) then
        begin
          if UpperCase(S).StartsWith('K') and (Length(S) >= 8) then
          begin
            time := TimeFromString(Copy(S, 4, 5));
            Prio := StrToIntDef(Copy(S, 2, 1), -1);
            if (time >= 0) and (Prio >= 0) and (Prio <= 2) then
            begin
              Self.KoppelDate := True;
              Self.KoppelPrio := Prio;
              Self.KoppelSecondTime := time;
            end;
          end;
        end;

        if not Self.KoppelDate then
        begin
          if UpperCase(S).StartsWith('D') and (Length(S) >= 2) then
          begin
            Prio := StrToIntDef(Copy(S, 2, 1), -1);
            if (Prio >= 0) and (Prio <= 2) then
            begin
              Self.DoubleDate := True;
              Self.KoppelPrio := Prio;
            end;
          end;
        end;

        if not Self.KoppelDate then
        begin
          if UpperCase(S).StartsWith('T2') and (Length(S) >= 8) then
          begin
            time := TimeFromString(Copy(S, 4, 5));
            if (time >= 0) then
            begin
              Self.KoppelAuswaertsSecondTime := True;
              Self.KoppelSecondTime := time;
            end;
          end;
        end;

        if UpperCase(S).StartsWith('MAX:') and (Length(S) >= 5) then
        begin
          MaxGames := StrToIntDef(Copy(S, 5, Length(S)), -1);
          if (MaxGames >= 0) then
          begin
            Self.MaxParallelGame := MaxGames;
          end;
        end;
      end;
    end;
  end;

  if SperrTermin then
  begin
    Self.Clear();
    Self.Date := Trunc(Date);
    Self.SperrTermin := True;
  end;

  Values.Free;
end;

function THomeDay.IsHomeDay: boolean;
begin
  Result := (not NoDate) and (not SperrTermin);
end;

function THomeDay.isTheSame(Value: THomeDay): boolean;
begin
  if NoDate or Value.NoDate then
  begin
    Result := (Date = Value.Date) and (NoDate = Value.NoDate);
  end
  else if SperrTermin or Value.SperrTermin then
  begin
    Result := (Date = Value.Date) and (SperrTermin = Value.SperrTermin);
  end
  else
  begin
    Result := (Date = Value.Date) and (KoppelDate = Value.KoppelDate) and
      (KoppelAuswaertsSecondTime = Value.KoppelAuswaertsSecondTime) and
      (DoubleDate = Value.DoubleDate) and (KoppelPrio = Value.KoppelPrio) and
      (KoppelSecondTime = Value.KoppelSecondTime) and
      (SperrTermin = Value.SperrTermin) and (NoDate = Value.NoDate) and
      (AusweichTermin = Value.AusweichTermin) and (Location = Value.Location)
      and (MaxParallelGame = Value.MaxParallelGame);
  end;

end;

type
  THomeDayComparer = class(TComparer<THomeDay>)
    function Compare(const Left, Right: THomeDay): integer; override;
  end;

function THomeDayComparer.Compare(const Left, Right: THomeDay): integer;
begin
  Result := 0;
  if Left.Date > Right.Date then
    Result := 1;

  if Left.Date < Right.Date then
    Result := -1;

  if Left.Date = Right.Date then
    Result := 0;

end;

constructor THomeDayList.Create();
begin
  inherited Create(THomeDayComparer.Create, True);
end;

end.
