unit DialogProfile;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  System.UITypes, Winapi.Windows, Winapi.Messages, Vcl.StdCtrls, Vcl.Controls, Vcl.ExtCtrls, Vcl.Forms, System.SysUtils,
  Vcl.ComCtrls, System.Classes, PlanTypes, System.Generics.Collections, System.Generics.Defaults,
  Dialogs, Vcl.Menus, PlanUtils, Winapi.ShellAPI, System.TypInfo, SyncObjs;

type

  TMannschaftsKostenEvent = procedure(Sender: TObject; MannschaftsType: TMannschaftsKostenType) of object;

  TTestAction = class(TObject)
    Name: String;
    Millisecs: DWORD;
    MannschaftsType: TMannschaftsKostenType;
    EventCustom: TNotifyEvent;
    EventMannschaft: TMannschaftsKostenEvent;
  public
    constructor Create(Name: String; EventCustom: TNotifyEvent); overload;
    constructor Create(Name: String; MannschaftsType: TMannschaftsKostenType; EventMannschaft: TMannschaftsKostenEvent); overload;
  end;


  TFormProfile = class(TForm)
    ButtonOK: TButton;
    PanelBottom: TPanel;
    Memo: TMemo;
    ButtonStart: TButton;
    Button1: TButton;
    procedure ButtonOKClick(Sender: TObject);
    procedure ButtonStartClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure Button1Click(Sender: TObject);
  private
    procedure CreateActions();
    procedure CalcMannschaft(Sender: TObject; MannschaftsType: TMannschaftsKostenType);
    procedure FireAction(Action: TTestAction);

    procedure CalculateSpielTagKosten(Sender: TObject);
    procedure CalculateFehlterminKosten(Sender: TObject);
    procedure CalculateFreeGameDateKosten(Sender: TObject);
    procedure CalculateSisterTeamsAmAnfang(Sender: TObject);
    procedure FillTermine10(Sender: TObject);
    procedure FillTermine2(Sender: TObject);
    procedure ShowResults(OnlyStats: boolean);

  public
    { Public-Deklarationen }
    Plan: TPlan;
    PlanTemp: TPlan;
    Actions: TObjectList<TTestAction>;
  end;


procedure ShowDialogProfile(Plan: TPlan);

procedure AddThreadTypeToStatistics(ThreadType: String);
procedure GetThreadStatistics(Strs: TStrings);
procedure ClearStats();
procedure AddThreadInfoToStatistics(ThreadName, Value: String);


implementation

{$R *.dfm}


procedure ShowDialogProfile(Plan: TPlan);
var
  Dlg: TFormProfile;
begin
  Dlg := TFormProfile.Create(nil);
  Dlg.Plan := Plan;
  Dlg.PlanTemp := TPlan.Create;
  Dlg.PlanTemp.Assign(Plan);
  Dlg.PlanTemp.TransferDateToRefDate();
  Dlg.ShowModal();
  Dlg.Free;
end;

procedure TFormProfile.FillTermine10(Sender: TObject);
begin
  PlanTemp.ProfileFillTermine(10);
end;

procedure TFormProfile.FillTermine2(Sender: TObject);
begin
  PlanTemp.ProfileFillTermine(2);
end;


procedure TFormProfile.FireAction(Action: TTestAction);
var
  StartTick: DWORD;
  i: Integer;
begin
  StartTick := GetTickCount();
  for i := 0 to 10000 do
  begin
    if Assigned(Action.EventCustom) then
    begin
      Action.EventCustom(nil);
    end;
    if Assigned(Action.EventMannschaft) then
    begin
      Action.EventMannschaft(nil, Action.MannschaftsType);
    end;
  end;

  Action.Millisecs := GetTickCount() - StartTick;

end;

type
  TTestActionComparer = class(TComparer<TTestAction>)
    function Compare(const Left, Right: TTestAction): Integer; override;
  end;

function TTestActionComparer.Compare(const Left, Right: TTestAction): Integer;
begin
  Result := Right.Millisecs - Left.Millisecs;
end;




procedure TFormProfile.ButtonStartClick(Sender: TObject);
begin
  ShowResults(False);
end;


procedure TFormProfile.ShowResults(OnlyStats: boolean);
var
  Action: TTestAction;
  Gesamt: DWORD;
  Comparer: TTestActionComparer;
begin
  StartWaitCursor();
  GetThreadStatistics(Memo.Lines);
  Memo.Update();
  Memo.Lines.Add('');

  if not OnlyStats then
  begin

    for Action in Actions do
    begin
      Action.Millisecs := 0;
    end;


    for Action in Actions do
    begin
      FireAction(Action);
    end;

    Gesamt := 0;
    for Action in Actions do
    begin
      Inc(Gesamt, Action.Millisecs);
    end;

    Comparer := TTestActionComparer.Create;
    Actions.Sort(Comparer);
    Comparer.Free;

    for Action in Actions do
    begin
      Memo.Lines.Add(Action.Name + ': ' + MyFormatInt(Action.Millisecs) + 'ms   ' + MyFormatFloat(Action.Millisecs * 100.0 / Gesamt) + '%');
    end;
  end;


end;


procedure TFormProfile.Button1Click(Sender: TObject);
begin
  ClearStats();
  ModalResult := mrOK;
end;

procedure TFormProfile.ButtonOKClick(Sender: TObject);
begin
  ModalResult := mrOK;
end;


procedure TFormProfile.CalcMannschaft(Sender: TObject;
  MannschaftsType: TMannschaftsKostenType);
begin
  Plan.CalculateMannschaftsKosten(MannschaftsType);
end;

procedure TFormProfile.CalculateFehlterminKosten(Sender: TObject);
begin
  Plan.CalculateFehlterminKosten();
end;

procedure TFormProfile.CalculateFreeGameDateKosten(Sender: TObject);
begin
  Plan.CalculateFreeGameDateKosten();
end;

procedure TFormProfile.CalculateSisterTeamsAmAnfang(Sender: TObject);
begin
  Plan.CalculateSisterTeamsAmAnfang();
end;

procedure TFormProfile.CalculateSpielTagKosten(Sender: TObject);
var
  Dummy1, Dummy2, Dummy3, Dummy4: MyDouble;
begin
  Plan.CalculateSpielTagKosten(Dummy1, Dummy2, Dummy3, Dummy4);
end;

procedure TFormProfile.CreateActions;
var
  Typ: TMannschaftsKostenType;
  Action: TTestAction;
begin
  for Typ := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    Action := TTestAction.Create(GetEnumName(TypeInfo(TMannschaftsKostenType), Ord(Typ)), Typ, CalcMannschaft);
    Actions.Add(Action);
  end;

  Action := TTestAction.Create('Spieltagkosten', CalculateSpielTagKosten);
  Actions.Add(Action);

  Action := TTestAction.Create('Fehltermine', CalculateFehlterminKosten);
  Actions.Add(Action);

  Action := TTestAction.Create('Nicht erlaubte Spieltage', CalculateFreeGameDateKosten);
  Actions.Add(Action);

  Action := TTestAction.Create('Vereinsinterne Spiele am Anfang', CalculateSisterTeamsAmAnfang);
  Actions.Add(Action);

  Action := TTestAction.Create('Filltermine 10%', FillTermine10);
  Actions.Add(Action);

  Action := TTestAction.Create('Filltermine 2%', FillTermine2);
  Actions.Add(Action);

end;

procedure TFormProfile.FormCreate(Sender: TObject);
begin
  Actions := TObjectList<TTestAction>.Create(True);
  CreateActions;
  FixControls(Self);
end;

procedure TFormProfile.FormDestroy(Sender: TObject);
begin
  Actions.Free;
  PlanTemp.Free;
end;

procedure TFormProfile.FormShow(Sender: TObject);
begin
  ShowResults(True);
end;

{ TTestAction }

constructor TTestAction.Create(Name: String; EventCustom: TNotifyEvent);
begin
  inherited Create;
  Self.Name := Name;
  Self.EventCustom := EventCustom;
end;

constructor TTestAction.Create(Name: String;
  MannschaftsType: TMannschaftsKostenType;
  EventMannschaft: TMannschaftsKostenEvent);
begin
  inherited Create;
  Self.Name := Name;
  Self.MannschaftsType := MannschaftsType;
  Self.EventMannschaft := EventMannschaft;
end;

type
  TStatsComparer = class(TComparer<TPair<String, Integer>>)
  public
    function Compare(const Value1, Value2: TPair<String, Integer>): Integer; override;
  end;

function TStatsComparer.Compare(const Value1, Value2: TPair<String, Integer>): Integer;
begin
  Result := Value2.Value - Value1.Value;
end;


var
  StatisticsCriticalSection: TCriticalSection;
  ThreadStatistics: TDictionary<String, Integer>;
  ThreadInfo: TDictionary<String, String>;


procedure ClearStats();
begin
  StatisticsCriticalSection.Enter;

  ThreadStatistics.Clear;

  StatisticsCriticalSection.Leave;
end;


procedure GetThreadStatistics(Strs: TStrings);
var
  Value: TPair<String, Integer>;
  ValueList: TList<TPair<String, Integer>>;
  ValueInfo: TPair<String, String>;
  Comparer: TStatsComparer;
begin
  StatisticsCriticalSection.Enter;

  Strs.Clear;

  Strs.Add('Treffer der Threadtypen');

  ValueList := TList<TPair<String, Integer>>.Create;

  for Value in ThreadStatistics do
  begin
    ValueList.Add(Value);
  end;

  Comparer := TStatsComparer.Create;

  ValueList.Sort(Comparer);

  Comparer.Free;

  for Value in ValueList do
  begin
    Strs.Add(Value.Key + ': ' + MyFormatInt(Value.Value));
  end;

  ValueList.Free;

  Strs.Add('');
  Strs.Add('Dynamische Threads');

  for ValueInfo in ThreadInfo do
  begin
    Strs.Add(ValueInfo.Key + ': ' + ValueInfo.Value);
  end;


  StatisticsCriticalSection.Leave;
end;


procedure AddThreadTypeToStatistics(ThreadType: String);
var
  value: integer;
begin
  StatisticsCriticalSection.Enter;

  if not ThreadStatistics.ContainsKey(ThreadType) then
  begin
    ThreadStatistics.Add(ThreadType, 0);
  end;

  value := ThreadStatistics.Items[ThreadType];
  Inc(value);
  ThreadStatistics.Items[ThreadType] := value;

  StatisticsCriticalSection.Leave;
end;

procedure AddThreadInfoToStatistics(ThreadName, Value: String);
begin
  StatisticsCriticalSection.Enter;

  if not ThreadInfo.ContainsKey(ThreadName) then
  begin
    ThreadInfo.Add(ThreadName, '');
  end;

  ThreadInfo.Items[ThreadName] := value;

  StatisticsCriticalSection.Leave;
end;


initialization
  StatisticsCriticalSection := TCriticalSection.Create();
  ThreadStatistics := TDictionary<String, Integer>.Create;
  ThreadInfo := TDictionary<String, String>.Create;
finalization
  StatisticsCriticalSection.Free;
  ThreadStatistics.Free;
  ThreadInfo.Free;
end.
