unit DialogPanelHomeDaysDetail;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls, PlanTypes, PlanUtils,
  Vcl.ComCtrls, System.Generics.Collections, System.Generics.Defaults,
  Vcl.Grids, Vcl.ValEdit, Vcl.Menus, DialogPanelOptionArray, DialogForSingleGewichtungsOptions,
  DialogPanel, PlanDataObjects, DialogTeamPanel, Math;

type
  TDialogPanelHomeDaysDetail = class(TDialogTeamPanel)
    PanelMain: TPanel;
    PanelBottom: TPanel;
    Grid: TStringGrid;
    LabelAbstand: TLabel;
    LabelColSize: TLabel;
    PopupMenu: TPopupMenu;
    PopupBearbeiten: TMenuItem;
    PopupLoeschen: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListViewTeamsClick(Sender: TObject);
    procedure GridGetEditText(Sender: TObject; ACol, ARow: Integer;
      var Value: string);
    procedure GridContextPopup(Sender: TObject; MousePos: TPoint;
      var Handled: Boolean);
    procedure PopupLoeschenClick(Sender: TObject);
    procedure PopupBearbeitenClick(Sender: TObject);
    procedure PopupMenuPopup(Sender: TObject);
    procedure GridDblClick(Sender: TObject);
  private
    DefaultValues: THomeDayList;
    OldSelCol: integer;
    OldSelRow: integer;

    procedure FormToHomeDays(ResultList: TObjectList<THomeDay>);
    procedure HomeDaysToForm(HomeDays: TObjectList<THomeDay>);
    function GetTeamDateString(Date: TDateTime; HomeDays: TObjectList<THomeDay>): String;
    function FixCellStringValue(Value: String): String;
    procedure AutoSizeGrid;
  public
    procedure setPanel(PanelMain: TPanel); override;
    procedure DataToForm(); override;
    procedure FormToData(); override;
    procedure toDefault(); override;
    function  isDefault(): boolean; override;
    function  CheckData(): boolean; override;
  end;


implementation

uses
  DialogEditOneHomeDay;

{$R *.dfm}



{ TDialogPanelFreeDays }

function TDialogPanelHomeDaysDetail.CheckData: boolean;
var
  HomeDays: THomeDayList;
  HomeDay: THomeDay;
  HomeDay2: THomeDay;
  Found: boolean;
begin
  Result := True;
  HomeDays := THomeDayList.Create;
  try
    FormToHomeDays(HomeDays);

    for HomeDay in HomeDays do
    begin
      if HomeDay.DoubleDate then
      begin
        // Suche nach dem Nachbarn
        Found := False;
        for HomeDay2 in HomeDays do
        begin
          if HomeDay.DoubleDate then
          begin
            if 1 = Abs(Trunc(HomeDay.Date) - Trunc(HomeDay2.Date)) then
            begin
              Found := True;
            end;
          end;
        end;
        if not Found then
        begin
          ShowMessage('Am ' + MyFormatDate(HomeDay.Date) + ' wurde ein Doppelspieltag angegeben. Das ist nur zulssig, wenn am darauf folgenden Tag auch ein Doppelspieltag ist. Fr einen Doppelspieltag sind immer zwei Termine an zwei Tagen hintereinander ntig, welche beide als Doppelspieltag gekennzeichnet sind.');
          Result := False;
          Exit;
        end;
      end;
    end;
  finally
    HomeDays.Free;
  end;
end;

procedure TDialogPanelHomeDaysDetail.DataToForm;
var
  HomeDays: THomeDayList;
begin
  HomeDays := THomeDayList.Create;
  PlanData.GetHomeDays(TeamName, HomeDays);
  HomeDaysToForm(HomeDays);
  HomeDays.Free;
end;




function TDialogPanelHomeDaysDetail.GetTeamDateString(Date: TDateTime; HomeDays: TObjectList<THomeDay>): String;
var
  HomeDay: THomeDay;
begin
  Result := '';
  for HomeDay in HomeDays do
  begin
    if Date = Trunc(HomeDay.Date) then
    begin
      Result := HomeDay.AsShortString();
    end;
  end;
end;

procedure TDialogPanelHomeDaysDetail.AutoSizeGrid;
var
  row: integer;
  Col: Integer;
  Value: integer;
begin
  for Col := 1 to Grid.ColCount-1 do
  begin
    grid.ColWidths[Col] := LabelColSize.Width;
    for row:=1 to Grid.RowCount - 1 do
    begin
      Value := canvas.TextWidth(Grid.Cells[col,row]) + 20;
      grid.ColWidths[Col] := max(grid.ColWidths[Col], Value);
    end;
  end;
end;


procedure TDialogPanelHomeDaysDetail.PopupBearbeitenClick(Sender: TObject);
var
  i: integer;
  row: integer;
  currentDate, Date, EndDate: TDateTime;
  S: String;
  HomeDay: THomeDay;
  Dialog: TDialogEditOneHomeDay;
begin
  Date := MondayBefore(PlanData.root.GetAsDate(aFrom));
  EndDate := min(Date + 360, PlanData.root.GetAsDate(aUntil));

  row := 1;
  while Date <= Trunc(EndDate) do
  begin
    for i := 0 to 6 do
    begin
      currentDate := Date + i;

      if (row = grid.Row) and (i+1 = grid.Col) then
      begin
        S := Grid.Cells[i+1, row];
        HomeDay := THomeDay.Create;
        HomeDay.FromShortString(currentDate, S);

        Dialog := TDialogEditOneHomeDay.Create(Self);
        Dialog.SetValue(PlanData, TeamName, HomeDay, currentDate);
        if mrOk = Dialog.ShowModal then
        begin
          S := Dialog.GetValue().AsShortString;
          Grid.Cells[i+1, row] := S;
          FireOnChange();
        end;

        Dialog.Free;

        HomeDay.Free;
        Exit;
      end;
    end;

    Date := Trunc(Date + 7);
    Inc(Row);
  end;

  AutoSizeGrid();
end;


procedure TDialogPanelHomeDaysDetail.PopupLoeschenClick(Sender: TObject);
begin
  if (grid.Col > 0) and (grid.Row > 0) then
  begin
    grid.Cells[grid.Col, grid.Row] := '';
    FireOnChange();
  end;
end;

procedure TDialogPanelHomeDaysDetail.PopupMenuPopup(Sender: TObject);
begin
  if (grid.Col > 0) and (grid.Row > 0) then
  begin
    PopupLoeschen.Enabled := Trim(grid.Cells[grid.col, grid.row]) <> '';
  end
  else
  begin
    PopupLoeschen.Enabled := False;
  end;
end;

function TDialogPanelHomeDaysDetail.FixCellStringValue(Value: String): String;
var
  HomeDay: THomeDay;
begin
  HomeDay := THomeDay.Create;

  if HomeDay.FromShortString(Trunc(Now), Value) then
  begin
    Result := HomeDay.AsShortString();
  end
  else
  begin
    Result := '';
  end;

  HomeDay.Free;
end;




procedure TDialogPanelHomeDaysDetail.GridContextPopup(Sender: TObject;
  MousePos: TPoint; var Handled: Boolean);
var
  col, row: integer;
  Pos: TPoint;
begin
  grid.MouseToCell(MousePos.X, MousePos.Y, col, row);

  if (col > 0) and (row > 0) then
  begin
    grid.Col := col;
    grid.Row := row;

    Pos := grid.ClientToScreen(MousePos);

    PopupLoeschen.Enabled := Trim(grid.Cells[col, row]) <> '';

    PopupMenu.Popup(Pos.X, Pos.Y);
    Handled := true;
  end;
end;

procedure TDialogPanelHomeDaysDetail.GridDblClick(Sender: TObject);
var
  col, row: integer;
  Pos: TPoint;
begin
  GetCursorPos(Pos);

  Pos := grid.ScreenToClient(Pos);
  grid.MouseToCell(Pos.X, Pos.Y, col, row);

  if (col > 0) and (row > 0) then
  begin
    grid.Col := col;
    grid.Row := row;

    PopupBearbeitenClick(nil);
  end;
end;

procedure TDialogPanelHomeDaysDetail.GridGetEditText(Sender: TObject; ACol, ARow: Integer; var Value: string);
begin
  if (OldSelCol > 0) and (OldSelRow > 0)
      and ((OldSelCol <> ACol) or (OldSelRow <> ARow))
  then
  begin
    Grid.Cells[OldSelCol, OldSelRow] := FixCellStringValue(Grid.Cells[OldSelCol, OldSelRow]);
  end;

  Value := FixCellStringValue(Value);

  OldSelCol := ACol;
  OldSelRow := ARow;

  FireOnChange();
  AutoSizeGrid();
end;

type
  THackInplaceEdit = class(TInplaceEdit)
  public
    procedure SetPopupMenu(Value: TPopupMenu);
  end;


procedure TDialogPanelHomeDaysDetail.HomeDaysToForm(HomeDays: TObjectList<THomeDay>);
var
  Date, EndDate: TDateTime;
  S, S1, S2: String;
  row: integer;
  i: Integer;
  Edit: THackInplaceEdit;
begin
  OldSelCol := -1;
  OldSelRow := -1;

  Grid.Cells[1, 0] := 'Mo';
  Grid.Cells[2, 0] := 'Di';
  Grid.Cells[3, 0] := 'Mi';
  Grid.Cells[4, 0] := 'Do';
  Grid.Cells[5, 0] := 'Fr';
  Grid.Cells[6, 0] := 'Sa';
  Grid.Cells[7, 0] := 'So';

  Grid.ColWidths[0] := LabelAbstand.Width;

  for i := 1 to 7 do
  begin
    Grid.ColWidths[i] := LabelColSize.Width;
  end;

  Date := MondayBefore(PlanData.root.GetAsDate(aFrom));
  EndDate := min(Date + 360, PlanData.root.GetAsDate(aUntil));

  row := 1;
  while Date <= EndDate do
  begin
    Grid.RowCount := row + 1;

    DateTimeToString(S1, 'dd.mm', Date);
    DateTimeToString(S2, 'dd.mm.yyyy', Date + 6);

    S := S1 + ' - ' + S2;
    Grid.Cells[0, row] := S;

    for i := 0 to 6 do
    begin
      S := GetTeamDateString(Date + i, HomeDays);
      Grid.Cells[i+1, row] := S;
    end;


    Date := Trunc(Date + 7);
    Inc(Row);
  end;

  // Popup am InplaceEdit dranhngen
  for i := 0 to grid.ComponentCount-1 do
  begin
    if grid.Components[i] is TInplaceEdit then
    begin
      Edit := THackInplaceEdit(grid.Components[i]);
      Edit.SetPopupMenu(PopupMenu);
    end;
  end;

  AutoSizeGrid();

end;


procedure TDialogPanelHomeDaysDetail.FormCreate(Sender: TObject);
begin
  DefaultValues := THomeDayList.Create;
  FixControls(Self);
end;

procedure TDialogPanelHomeDaysDetail.FormDestroy(Sender: TObject);
begin
  DefaultValues.Free;
end;

procedure TDialogPanelHomeDaysDetail.FormToData;
var
  HomeDays: THomeDayList;
begin
  HomeDays := THomeDayList.Create;
  FormToHomeDays(HomeDays);
  PlanData.SetHomeDays(TeamName, HomeDays);
  HomeDays.Free;
end;


procedure TDialogPanelHomeDaysDetail.FormToHomeDays(ResultList: TObjectList<THomeDay>);
var
  i: integer;
  row: integer;
  currentDate, Date, EndDate: TDateTime;
  S: String;
  HomeDay: THomeDay;
begin
  ResultList.Clear;

  Date := MondayBefore(PlanData.root.GetAsDate(aFrom));
  EndDate := min(Date + 360, PlanData.root.GetAsDate(aUntil));

  row := 1;
  while Date <= Trunc(EndDate) do
  begin
    for i := 0 to 6 do
    begin
      currentDate := Date + i;
      S := Grid.Cells[i+1, row];

      HomeDay := THomeDay.Create;

      if HomeDay.FromShortString(currentDate, S) then
      begin
        ResultList.Add(HomeDay);
      end
      else
      begin
        HomeDay.Free;
      end;
    end;

    Date := Trunc(Date + 7);
    Inc(Row);
  end;
end;


function TDialogPanelHomeDaysDetail.isDefault: boolean;
var
  List1, List2: THomeDayList;
  Item1, Item2: THomeDay;
  Found: boolean;
begin
  List1 := THomeDayList.Create;
  List2 := THomeDayList.Create;

  if Assigned(PlanDataDefault) then
  begin
    PlanDataDefault.GetHomeDays(TeamName, List1);
  end;

  FormToData();

  PlanData.GetHomeDays(TeamName, List2);

  Result := List1.Count = List2.Count;

  for Item1 in List1 do
  begin
    Found := False;
    for Item2 in List2 do
    begin
      if Item1 <> Item2 then
      begin
        if Item1.IsTheSame(Item2) then
        begin
          Found := True;
        end;
      end;
    end;

    if not Found then
    begin
      Result := False;
      break;
    end;
  end;

  List1.Free;
  List2.Free;
end;

procedure TDialogPanelHomeDaysDetail.ListViewTeamsClick(Sender: TObject);
begin
  FireOnChange();
end;

procedure TDialogPanelHomeDaysDetail.setPanel(PanelMain: TPanel);
begin
  Self.PanelMain.Parent := PanelMain;
end;

procedure TDialogPanelHomeDaysDetail.toDefault;
var
  List: THomeDayList;
begin
  List := THomeDayList.Create;

  if Assigned(PlanDataDefault) then
  begin
    PlanDataDefault.GetHomeDays(TeamName, List);
  end;

  HomeDaysToForm(List);

  List.Free;
end;


{ THackInplaceEdit }

procedure THackInplaceEdit.SetPopupMenu(Value: TPopupMenu);
begin
  PopupMenu := Value;
end;

end.
