unit DialogPanelHomeCoupleCompactDetail;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls, PlanTypes, PlanUtils,
  Vcl.ComCtrls, System.Generics.Collections, System.Generics.Defaults,
  Vcl.Grids, Vcl.ValEdit, Vcl.Menus,
  DialogPanel, PlanDataObjects, System.UITypes, DialogTeamPanel;

type
  TKoppelWunschElem = class(TObject)
  public
    First: THomeDay;
    Second: THomeDay;
    constructor Create();
    destructor  Destroy(); override;
    procedure setDates(First: THomeDay; Second: THomeDay);
  end;

  TCalculateOptionsKoppelDate = class(TObject)
  public
    Date1: TDateTime;
    Date2: TDateTime;
    Option1: TWunschterminOption;
    Option2: TWunschterminOption;
    constructor Create();
    function isTheSame(Source: TCalculateOptionsKoppelDate): boolean;
  end;


  TDialogPanelHomeCoupleCompactDetail = class(TDialogTeamPanel)
    PanelMain: TPanel;
    ValueListHomeKoppel: TValueListEditor;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ValueListHomeKoppelStringsChange(Sender: TObject);
  private
    InLoad: boolean;
    workHomeKoppelList: TObjectList<TCalculateOptionsKoppelDate>;
    mapRowToHomeKoppel: TDictionary<integer, TCalculateOptionsKoppelDate>;

    procedure getKoppelWunschListForMannschaft(PlanData: TPlanData; Liste: TObjectList<TKoppelWunschElem>);
    function getComboIndexFromOptions(Option1,
      Option2: TWunschterminOption): integer;
    procedure getHomeKoppelOptionsByIndexAndTwoDates(var Option1,
      Option2: TWunschterminOption; Index: Integer);
    procedure addComboElemsForHeimKoppelAtOneDay(ResultList: TStrings;
      Prefix: String);
    procedure getComboElemsForHeimKoppelWuensche(ResultList: TStrings; Date1, Date2: TDateTime);
    procedure setHomeKoppelOptionsByIndex(Elem: TCalculateOptionsKoppelDate;
      Index: Integer);
    procedure LoadKoppelDates(PlanData: TPlanData; KoppelDates: TObjectList<TCalculateOptionsKoppelDate>);
    procedure DataObjectToForm(PlanData: TPlanData);
  public
    procedure setPanel(PanelMain: TPanel); override;
    procedure DataToForm(); override;
    procedure FormToData(); override;
    procedure toDefault(); override;
    function  isDefault(): boolean; override;
  end;


implementation

{$R *.dfm}


function KoppelValueFromOptions(Option: THomeDay): TWunschterminOption;
begin
  Result := woKoppelNothing;
  if Option.KoppelDate then
  begin
    if Option.KoppelPrio = 2 then
    begin
      Result := woKoppelHard;
    end;
    if Option.KoppelPrio = 0 then
    begin
      Result := woKoppelPossible;
    end;
    if Option.KoppelPrio = 1 then
    begin
      Result := woKoppelSoft;
    end;
  end;

  if Option.DoubleDate then
  begin
    if Option.KoppelPrio = 2 then
    begin
      Result := woKoppelDoubleDateHard;
    end;
    if Option.KoppelPrio = 0 then
    begin
      Result := woKoppelDoubleDatePossible;
    end;
    if Option.KoppelPrio = 1 then
    begin
      Result := woKoppelDoubleDateSoft;
    end;
  end;
end;



procedure TDialogPanelHomeCoupleCompactDetail.getKoppelWunschListForMannschaft(PlanData: TPlanData; Liste: TObjectList<TKoppelWunschElem>);
var
  HasBefore, HasNext: boolean;
  TempTermin: THomeDay;
  NextTermin: THomeDay;
  Elem: TKoppelWunschElem;
  HomeDays: THomeDayList;
  HomeDay: THomeDay;
begin
  Liste.Clear;

  HomeDays := THomeDayList.Create;

  PlanData.GetHomeDays(TeamName, HomeDays);

  for HomeDay in HomeDays do
  begin
    if HomeDay.IsHomeDay then
    begin
      HasBefore := False;
      HasNext := False;
      NextTermin := nil;

      for TempTermin in HomeDays do
      begin
        if TempTermin.IsHomeDay then
        begin
          if 1 = (Trunc(TempTermin.Date) - Trunc(HomeDay.Date)) then
          begin
            HasNext := True;
            NextTermin := TempTermin;
          end;
          if -1 = (Trunc(TempTermin.Date) - Trunc(HomeDay.Date)) then
          begin
            HasBefore := True;
          end;
        end;
      end;

      if HasNext then
      begin
        Elem := TKoppelWunschElem.Create;
        Elem.setDates(HomeDay, NextTermin);
        Liste.Add(Elem);
      end;

      if (not HasNext) and (not HasBefore) then
      begin
        if HomeDay.KoppelDate then
        begin
          Elem := TKoppelWunschElem.Create;
          Elem.setDates(HomeDay, nil);
          Liste.Add(Elem);
        end;
      end;
    end;
  end;

  HomeDays.Free;
end;


const
  MaxComboIndex = 12;


procedure TDialogPanelHomeCoupleCompactDetail.getHomeKoppelOptionsByIndexAndTwoDates(var Option1, Option2: TWunschterminOption; Index: Integer);
begin
  // Grausame Case-Latte fr das Kreuzprodukt der mglich Optionen
  Option1 := woKoppelNothing;
  Option2 := woKoppelNothing;
  // Bei zwei Spieltagen hintereinander gibt es die ganzen Mglichkeiten, die in getComboElemsForHeimKoppelWuensche
  // zusammengestllt werden. Hier muss ich das Ganze wieder auflsen

  case Index of
    0:
      Option1 := woKoppelNothing;
    1:
      begin
        Option1 := woKoppelDoubleDatePossible;
        Option2 := woKoppelDoubleDatePossible;
      end;
    2:
      begin
        Option1 := woKoppelDoubleDateSoft;
        Option2 := woKoppelDoubleDateSoft;
      end;
    3:
      begin
        Option1 := woKoppelDoubleDateHard;
        Option2 := woKoppelDoubleDateHard;
      end;
    4:
      Option1 := woKoppelPossible;
    5:
      Option1 := woKoppelSoft;
    6:
      Option1 := woKoppelHard;
    7:
      Option2 := woKoppelPossible;
    8:
      Option2 := woKoppelSoft;
    9:
      Option2 := woKoppelHard;
    10:
      begin
        Option2 := woKoppelPossible;
        Option1 := woKoppelPossible;
      end;
    11:
      begin
        Option2 := woKoppelSoft;
        Option1 := woKoppelSoft;
      end;
    12:
      begin
        Option2 := woKoppelHard;
        Option1 := woKoppelHard;
      end

  else
    raise Exception.Create('Unbekannter KoppeloptionIndex');
  end;
end;



function TDialogPanelHomeCoupleCompactDetail.getComboIndexFromOptions(Option1, Option2: TWunschterminOption): integer;
var
  TestOption1, TestOption2: TWunschterminOption;
  i: integer;
begin
  // Einfach die grausame Ermittlung durchlaufen lassen, bis die Werte gleich sind
  Result := 0;

  for i := 0 to MaxComboIndex do
  begin
    getHomeKoppelOptionsByIndexAndTwoDates(TestOption1, TestOption2, i);
    if (TestOption1 = Option1) and (TestOption2 = Option2) then
    begin
      Result := i;
    end;
  end;
end;


procedure TDialogPanelHomeCoupleCompactDetail.addComboElemsForHeimKoppelAtOneDay(ResultList: TStrings; Prefix: String);
begin
  ResultList.Add(Prefix + ' ' + cWunschterminOptionName[woKoppelPossible]);
  ResultList.Add(Prefix + ' ' + cWunschterminOptionName[woKoppelSoft]);
  ResultList.Add(Prefix + ' ' + cWunschterminOptionName[woKoppelHard]);
end;



procedure TDialogPanelHomeCoupleCompactDetail.getComboElemsForHeimKoppelWuensche(ResultList: TStrings; Date1, Date2: TDateTime);
begin
  ResultList.Clear;
  ResultList.Add(cWunschterminOptionName[woKoppelNothing]);
  ResultList.Add(cWunschterminOptionName[woKoppelDoubleDatePossible]);
  ResultList.Add(cWunschterminOptionName[woKoppelDoubleDateSoft]);
  ResultList.Add(cWunschterminOptionName[woKoppelDoubleDateHard]);

  addComboElemsForHeimKoppelAtOneDay(ResultList, '' + MyFormatDate(Date1) + ':');
  addComboElemsForHeimKoppelAtOneDay(ResultList, '' + MyFormatDate(Date2) + ':');
  addComboElemsForHeimKoppelAtOneDay(ResultList, '' + MyFormatDate(Date1) + ' und ' + MyFormatDate(Date2) + ':');
end;


procedure TDialogPanelHomeCoupleCompactDetail.LoadKoppelDates(PlanData: TPlanData; KoppelDates: TObjectList<TCalculateOptionsKoppelDate>);
var
  HeimKoppelWuensche: TObjectList<TKoppelWunschElem>;
  Elem: TKoppelWunschElem;
  S: String;
  WunschOptions, WunschOptions1, WunschOptions2: THomeDay;
  NewHomeElem: TCalculateOptionsKoppelDate;
begin
  KoppelDates.Clear;

  HeimKoppelWuensche := TObjectList<TKoppelWunschElem>.Create;
  getKoppelWunschListForMannschaft(PlanData, HeimKoppelWuensche);

  if HeimKoppelWuensche.Count > 0 then
  begin
    for Elem in HeimKoppelWuensche do
    begin
      if Elem.Second = nil then
      begin
        WunschOptions := Elem.First;

        if WunschOptions.KoppelPrio = 0 then
          S := cWunschterminOptionName[woKoppelPossible]
        else
        if WunschOptions.KoppelPrio = 2 then
          S := cWunschterminOptionName[woKoppelHard]
        else
        if WunschOptions.KoppelPrio = 1 then
          S := cWunschterminOptionName[woKoppelSoft];

        NewHomeElem := TCalculateOptionsKoppelDate.Create;
        NewHomeElem.Option1 := KoppelValueFromOptions(WunschOptions);
        NewHomeElem.Date1 := Trunc(Elem.First.Date);
        NewHomeElem.Date2 := 0;
        KoppelDates.Add(NewHomeElem);
      end
      else
      begin
        WunschOptions1 := Elem.First;
        WunschOptions2 := Elem.Second;

        NewHomeElem := TCalculateOptionsKoppelDate.Create;
        NewHomeElem.Option1 := KoppelValueFromOptions(WunschOptions1);
        NewHomeElem.Option2 := KoppelValueFromOptions(WunschOptions2);
        NewHomeElem.Date1 := Trunc(Elem.First.Date);
        NewHomeElem.Date2 := Trunc(Elem.Second.Date);
        KoppelDates.Add(NewHomeElem);
      end;
    end;
  end;
  HeimKoppelWuensche.Free;
end;



procedure TDialogPanelHomeCoupleCompactDetail.DataToForm;
begin
  DataObjectToForm(PlanData);
end;



procedure TDialogPanelHomeCoupleCompactDetail.DataObjectToForm(PlanData: TPlanData);
var
  ItemProp: TItemProp;
  Row: Integer;
  S: String;
  ComboIndex: integer;
  Elem: TCalculateOptionsKoppelDate;
begin
  InLoad := True;
  try

    ValueListHomeKoppel.Strings.Clear;

    LoadKoppelDates(PlanData, workHomeKoppelList);
    mapRowToHomeKoppel.Clear;

    for Elem in workHomeKoppelList do
    begin
      if Elem.Date2 = 0 then
      begin
        S := cWunschterminOptionName[Elem.Option1];

        Row := ValueListHomeKoppel.InsertRow(MyFormatDate(Elem.Date1), S, true);

        mapRowToHomeKoppel.Add(Row, Elem);

        ItemProp := ValueListHomeKoppel.ItemProps[Row - 1];

        ItemProp.ReadOnly := True;
        ItemProp.EditStyle := esPickList;
        ItemProp.PickList.Add(cWunschterminOptionName[woKoppelNothing]);
        ItemProp.PickList.Add(cWunschterminOptionName[woKoppelPossible]);
        ItemProp.PickList.Add(cWunschterminOptionName[woKoppelSoft]);
        ItemProp.PickList.Add(cWunschterminOptionName[woKoppelHard]);
      end
      else
      begin
        ComboIndex := getComboIndexFromOptions(Elem.Option1, Elem.Option2);

        Row := ValueListHomeKoppel.InsertRow(MyFormatDate(Elem.Date1) + ' und ' + MyFormatDate(Elem.Date2), '', true);

        mapRowToHomeKoppel.Add(Row, Elem);

        ItemProp := ValueListHomeKoppel.ItemProps[Row - 1];

        ItemProp.ReadOnly := True;
        ItemProp.EditStyle := esPickList;
        getComboElemsForHeimKoppelWuensche(ItemProp.PickList, Elem.Date1, Elem.Date2);

        S := ItemProp.PickList[ComboIndex];
        ValueListHomeKoppel.Cells[1, row] := S;

      end;
    end;

    if ValueListHomeKoppel.Strings.Count = 0 then
    begin
      Row := ValueListHomeKoppel.InsertRow('Koppelspiele sind in den Terminwnschen nicht enthalten', '', True);
      ItemProp := ValueListHomeKoppel.ItemProps[Row - 1];
      ItemProp.ReadOnly := True;
    end;
  finally
    InLoad := False;
  end;
end;


procedure TDialogPanelHomeCoupleCompactDetail.FormCreate(Sender: TObject);
begin
  ValueListHomeKoppel.DefaultRowHeight := MulDiv(ValueListHomeKoppel.DefaultRowHeight, Screen.PixelsPerInch, 96);
  workHomeKoppelList := TObjectList<TCalculateOptionsKoppelDate>.Create;
  mapRowToHomeKoppel := TDictionary<integer, TCalculateOptionsKoppelDate>.Create;

  FixControls(Self);
end;

procedure TDialogPanelHomeCoupleCompactDetail.FormDestroy(Sender: TObject);
begin
  workHomeKoppelList.Free;
  mapRowToHomeKoppel.Free;
end;

procedure TDialogPanelHomeCoupleCompactDetail.setHomeKoppelOptionsByIndex(Elem: TCalculateOptionsKoppelDate; Index: Integer);
begin
  // Grausame Case-Latte fr das Kreuzprodukt der mglich Optionen
  Elem.Option1 := woKoppelNothing;
  Elem.Option2 := woKoppelNothing;
  if Elem.Date2 = 0 then
  begin
    case Index of
      0:
        Elem.Option1 := woKoppelNothing;
      1:
        Elem.Option1 := woKoppelPossible;
      2:
        Elem.Option1 := woKoppelSoft;
      3:
        Elem.Option1 := woKoppelHard;
    else
      raise Exception.Create('Unbekannter KoppeloptionIndex');
    end;
  end
  else
  begin
    getHomeKoppelOptionsByIndexAndTwoDates(Elem.Option1, Elem.Option2, Index);
  end;
end;


procedure TDialogPanelHomeCoupleCompactDetail.FormToData;
var
  row: integer;
  ItemProp: TItemProp;
  Index: integer;
  HomeDays: THomeDayList;
  HomeDay: THomeDay;
  KoppelDate: TCalculateOptionsKoppelDate;
begin
  for row in mapRowToHomeKoppel.Keys do
  begin
    ItemProp := ValueListHomeKoppel.ItemProps[Row - 1];
    Index := ItemProp.PickList.IndexOf(ValueListHomeKoppel.Cells[1, row]);
    setHomeKoppelOptionsByIndex(mapRowToHomeKoppel[row], Index);
  end;

  HomeDays := THomeDayList.Create;

  PlanData.GetHomeDays(TeamName, HomeDays);


  for HomeDay in HomeDays do
  begin
    if HomeDay.IsHomeDay then
    begin
      HomeDay.KoppelDate := false;
      HomeDay.DoubleDate := false;
      HomeDay.KoppelPrio := 0;
    end;
  end;


  for KoppelDate in workHomeKoppelList do
  begin
    for HomeDay in HomeDays do
    begin
      if HomeDay.IsHomeDay then
      begin
        if Trunc(HomeDay.Date) = Trunc(KoppelDate.Date1) then
        begin
          if KoppelDate.Option1 in [woKoppelPossible, woKoppelSoft, woKoppelHard] then
          begin
            HomeDay.KoppelDate := true;
            if KoppelDate.Option1 = woKoppelSoft then
            begin
              HomeDay.KoppelPrio := 1;
            end;
            if KoppelDate.Option1 = woKoppelHard then
            begin
              HomeDay.KoppelPrio := 2;
            end;
          end;

          if KoppelDate.Option1 in [woKoppelDoubleDatePossible, woKoppelDoubleDateSoft, woKoppelDoubleDateHard] then
          begin
            // DoubleDate
            HomeDay.DoubleDate := true;
            if KoppelDate.Option1 = woKoppelDoubleDateSoft then
            begin
              HomeDay.KoppelPrio := 1;
            end;
            if KoppelDate.Option1 = woKoppelDoubleDateHard then
            begin
              HomeDay.KoppelPrio := 2;
            end;
          end;
        end;

        if Trunc(HomeDay.Date) = Trunc(KoppelDate.Date2) then
        begin
          if KoppelDate.Option2 in [woKoppelPossible, woKoppelSoft, woKoppelHard] then
          begin
            HomeDay.KoppelDate := true;
            if KoppelDate.Option2 = woKoppelSoft then
            begin
              HomeDay.KoppelPrio := 1;
            end;
            if KoppelDate.Option2 = woKoppelHard then
            begin
              HomeDay.KoppelPrio := 2;
            end;
          end;

          if KoppelDate.Option2 in [woKoppelDoubleDatePossible, woKoppelDoubleDateSoft, woKoppelDoubleDateHard] then
          begin
            // DoubleDate
            HomeDay.DoubleDate := true;
            if KoppelDate.Option2 = woKoppelDoubleDateSoft then
            begin
              HomeDay.KoppelPrio := 1;
            end;
            if KoppelDate.Option2 = woKoppelDoubleDateHard then
            begin
              HomeDay.KoppelPrio := 2;
            end;
          end;
        end;
      end;
    end;
  end;


  PlanData.SetHomeDays(TeamName, HomeDays);

  HomeDays.Free;
end;


function TDialogPanelHomeCoupleCompactDetail.isDefault: boolean;
var
  KoppelDatesDefault: TObjectList<TCalculateOptionsKoppelDate>;
  i: integer;
begin
  FormToData();
  KoppelDatesDefault := TObjectList<TCalculateOptionsKoppelDate>.Create;
  LoadKoppelDates(PlanDataDefault, KoppelDatesDefault);

  Result := workHomeKoppelList.Count = KoppelDatesDefault.Count;

  if Result then
  begin
    for i := 0 to workHomeKoppelList.Count-1 do
    begin
      if not workHomeKoppelList[i].isTheSame(KoppelDatesDefault[i]) then
      begin
        Result := False;
      end;
    end;
  end;
  KoppelDatesDefault.Free;
end;

procedure TDialogPanelHomeCoupleCompactDetail.setPanel(PanelMain: TPanel);
begin
  Self.PanelMain.Parent := PanelMain;
end;


procedure TDialogPanelHomeCoupleCompactDetail.toDefault;
begin
  DataObjectToForm(PlanDataDefault);
  FormToData;
  DataToForm;
  FireOnChange;
end;


procedure TDialogPanelHomeCoupleCompactDetail.ValueListHomeKoppelStringsChange(Sender: TObject);
begin
  if not InLoad then
  begin
    FireOnChange();
  end;
end;

{ TKoppelWunschElem }

constructor TKoppelWunschElem.Create;
begin
  inherited Create;
  First := nil;
  Second := nil;
end;

destructor TKoppelWunschElem.Destroy;
begin
  First.Free;
  Second.Free;
  inherited;
end;

procedure TKoppelWunschElem.setDates(First, Second: THomeDay);
begin
  FreeAndNil(Self.First);
  FreeAndNil(Self.Second);

  if Assigned(First) then
  begin
    Self.First := THomeDay.Create;
    Self.First.Assign(First);
  end;

  if Assigned(Second) then
  begin
    Self.Second := THomeDay.Create;
    Self.Second.Assign(Second);
  end;

end;

{ TCalculateOptionsKoppelDate }

constructor TCalculateOptionsKoppelDate.Create;
begin
  inherited Create;
  Date1 := 0;
  Date2 := 0;
  Option1 := woKoppelNothing;
  Option2 := woKoppelNothing;
end;

function TCalculateOptionsKoppelDate.isTheSame(Source: TCalculateOptionsKoppelDate): boolean;
begin
  Result := (Date1 = Source.Date1)
            and (Date2 = Source.Date2)
            and (Option1 = Source.Option1)
            and (Option2 = Source.Option2);
end;

end.
