unit DialogOptions;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls, PlanTypes, PlanUtils,
  Vcl.ComCtrls, System.Generics.Collections, System.Generics.Defaults,
  Vcl.Grids, Vcl.ValEdit, EditMandatoryDatesDialog, Vcl.Menus, DialogPanelOptionArray, DialogForSingleGewichtungsOptions;

type

  TOnSaveEvent = procedure(Sender: TObject; SaveOnlyOptions: boolean) of object;

  TFormOptions = class(TForm)
    ButtonOK: TButton;
    CheckBoxFreitagIs60km: TCheckBox;
    Bevel1: TBevel;
    PageControl: TPageControl;
    TabSheetGewichtung: TTabSheet;
    PanelBottom: TPanel;
    TabSheetDoppelrunde: TTabSheet;
    CheckBoxDoubleRound: TCheckBox;
    CheckBoxDateAutomatic: TCheckBox;
    Bevel2: TBevel;
    DateTimeMid1: TDateTimePicker;
    LabelFirstRoundDate: TLabel;
    DateTimeMid2: TDateTimePicker;
    LabelThirdRoundDate: TLabel;
    ButtonToStandard: TButton;
    Timer1: TTimer;
    Bevel3: TBevel;
    ComboBoxPlanRounds: TComboBox;
    GroupBox2: TGroupBox;
    GroupBox1: TGroupBox;
    ButtonGewichtungMannschaften: TButton;
    PanelMannschaftenGewichtung: TPanel;
    Panel1: TPanel;
    PanelPlanGewichtung: TPanel;
    Panel2: TPanel;
    Panel3: TPanel;
    Panel4: TPanel;
    Panel5: TPanel;
    procedure ButtonOKClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure ControlChanged(Sender: TObject);
    procedure ListViewFreeDatesChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure CheckBoxDateAutomaticClick(Sender: TObject);
    procedure ListViewAusweichtermineChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure PageControlChange(Sender: TObject);
    procedure ButtonToStandardClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure ComboBoxPlanRoundsChange(Sender: TObject);
    procedure ButtonGewichtungMannschaftenClick(Sender: TObject);
  private
    Plan: TPlan;
    { Private-Deklarationen }
    FOnAfterSave: TOnSaveEvent;
    InLoad: boolean;

    PanelMannschaftOptions: TFormPanelOptionArray;
    PanelPlanOptions: TFormPanelOptionArray;

    procedure Save();
    procedure OptionsToForm(Options: TCalculateOptions);
    procedure FormToOptions(Options: TCalculateOptions);
    procedure EnableButtons();
  public
    { Public-Deklarationen }
    procedure SetValues(Plan: TPlan);
    property OnAfterSave: TOnSaveEvent read FOnAfterSave write FOnAfterSave;
  end;


implementation

{$R *.dfm}

type
  TSubDialogController = class(TObject)
    Plan: TPlan;
  public
    constructor Create();
    destructor  Destroy(); override;
    function  Execute(): boolean;
    procedure DoButtonClick(Sender: TObject; Index: integer);
  end;

procedure TFormOptions.ButtonGewichtungMannschaftenClick(Sender: TObject);
var
  DialogController: TSubDialogController;
begin
  DialogController := TSubDialogController.Create;
  DialogController.Plan.Assign(Plan);
  if DialogController.Execute then
  begin
    Plan.getOptions().Assign(DialogController.Plan.getOptions());
    ControlChanged(nil);
  end;
  DialogController.Free;
end;

procedure TFormOptions.ButtonOKClick(Sender: TObject);
begin
  Visible := False;
end;

procedure TFormOptions.ButtonToStandardClick(Sender: TObject);
var
  KostenType: TMannschaftsKostenType;
  PlanKostenType: TPlanKostenType;
  values: TList<TCalculateOptionsValues>;
begin
  if PageControl.ActivePage = TabSheetGewichtung then
  begin
    CheckBoxFreitagIs60km.Checked := True;

    values := TList<TCalculateOptionsValues>.Create;

    for PlanKostenType := Low(TPlanKostenType) to High(TPlanKostenType) do
    begin
      values.Add(coNormal);
    end;
    PanelPlanOptions.SetValues(values);

    values.Free;

    values := TList<TCalculateOptionsValues>.Create;

    for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      values.Add(coNormal);
    end;
    PanelMannschaftOptions.SetValues(values);

    values.Free;

    Plan.getOptions().GewichtungMannschaften.Clear;


    ControlChanged(nil);
  end;

  if PageControl.ActivePage = TabSheetDoppelrunde then
  begin
    ComboBoxPlanRounds.ItemIndex := 0;
    CheckBoxDoubleRound.Checked := False;
    CheckBoxDateAutomatic.Checked := True;
    CheckBoxDateAutomaticClick(nil);
  end;

end;

procedure TFormOptions.CheckBoxDateAutomaticClick(Sender: TObject);
begin
  if CheckBoxDateAutomatic.Checked then
  begin
    DateTimeMid1.DateTime := Plan.getOptions().getAutomaticMidDate1(Plan);
    DateTimeMid2.DateTime := Plan.getOptions().getAutomaticMidDate2(Plan);
  end;

  ControlChanged(Sender);
end;

procedure TFormOptions.ComboBoxPlanRoundsChange(Sender: TObject);
begin
  if not InLoad then
  begin
    EnableButtons();
    Save();
  end;
end;

procedure TFormOptions.ControlChanged(Sender: TObject);
begin
  if not InLoad then
  begin
    EnableButtons();
    Save();
  end;
end;

procedure TFormOptions.EnableButtons;
var
  StandardEnabled: boolean;
  Value: TMannschaftsKostenType;
  values: TList<TCalculateOptionsValues>;
  optionValue: TCalculateOptionsValues;
  visibleState: TList<boolean>;
begin
  CheckBoxDateAutomatic.Enabled := CheckBoxDoubleRound.Checked;
  LabelFirstRoundDate.Enabled := CheckBoxDoubleRound.Checked and (not CheckBoxDateAutomatic.Checked);
  LabelThirdRoundDate.Enabled := LabelFirstRoundDate.Enabled;
  DateTimeMid1.Enabled := LabelFirstRoundDate.Enabled;
  DateTimeMid2.Enabled := LabelFirstRoundDate.Enabled;

  StandardEnabled := False;
  if PageControl.ActivePage = TabSheetGewichtung then
  begin
    StandardEnabled := StandardEnabled or (CheckBoxFreitagIs60km.Checked <> True);

    values := TList<TCalculateOptionsValues>.Create;
    PanelPlanOptions.GetValues(values);
    for optionValue in Values do
    begin
      StandardEnabled := StandardEnabled or (optionValue <> coNormal);
    end;
    values.Free;

    values := TList<TCalculateOptionsValues>.Create;
    PanelMannschaftOptions.GetValues(values);

    for optionValue in Values do
    begin
      StandardEnabled := StandardEnabled or (optionValue <> coNormal);
    end;

    values.Free;
  end;

  if PageControl.ActivePage = TabSheetDoppelrunde then
  begin
    StandardEnabled := StandardEnabled or (ComboBoxPlanRounds.ItemIndex <> 0);
    StandardEnabled := StandardEnabled or (CheckBoxDoubleRound.Checked <> False);
    StandardEnabled := StandardEnabled or (CheckBoxDateAutomatic.Checked <> True);
  end;

  ButtonToStandard.Enabled := StandardEnabled;

  visibleState := TList<boolean>.Create;
  for Value := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    if Plan.HasValuesForType(Value) then
    begin
      visibleState.Add(true)
    end
    else
    begin
      visibleState.Add(false)
    end;
  end;

  PanelMannschaftOptions.SetVisibleState(visibleState);

  visibleState.Free;

end;



procedure TFormOptions.FormCreate(Sender: TObject);
var
  kostenType: TMannschaftsKostenType;
  PlanKostenType: TPlanKostenType;
  titles: TStrings;
begin
  FixFormSize(Self, 848, 697);
  InLoad := False;

  PageControl.ActivePageIndex := 0;

  PanelPlanOptions := TFormPanelOptionArray.Create(Self);
  PanelPlanOptions.SetPanel(PanelPlanGewichtung);
  PanelPlanOptions.OnChanged := ControlChanged;

  titles:= TStringList.Create;
  for PlanKostenType := Low(TPlanKostenType) to High(TPlanKostenType) do
  begin
    titles.Add(cPlanKostenTypeNamen[PlanKostenType]);
  end;
  PanelPlanOptions.Init(titles);
  titles.Free;

  PanelMannschaftOptions := TFormPanelOptionArray.Create(Self);
  PanelMannschaftOptions.SetPanel(PanelMannschaftenGewichtung);
  PanelMannschaftOptions.OnChanged := ControlChanged;

  titles:= TStringList.Create;
  for kostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    titles.Add(cMannschaftsKostenTypeNamen[kostenType]);
  end;
  PanelMannschaftOptions.Init(titles);
  titles.Free;

  FixControls(Self);
end;

procedure TFormOptions.FormToOptions(Options: TCalculateOptions);
var
  i: TMannschaftsKostenType;
  PlanKostenType: TPlanKostenType;
  EmptyOptions: TCalculateOptions;
  values: TList<TCalculateOptionsValues>;
  PlanGewichtung: array[Low(TPlanKostenType) .. High(TPlanKostenType)] of TCalculateOptionsValues;
begin
  EmptyOptions := TCalculateOptions.Create;
  Options.FreitagIsAllowedBy60km := CheckBoxFreitagIs60km.Checked;
  Options.RoundPlaning := TRoundPlaning(ComboBoxPlanRounds.ItemIndex);
  Options.DoubleRound := CheckBoxDoubleRound.Checked;
  Options.AutomaticMidDate := CheckBoxDateAutomatic.Checked;
  Options.MidDate1 := DateTimeMid1.DateTime;
  Options.MidDate2 := DateTimeMid2.DateTime;

  values := TList<TCalculateOptionsValues>.Create;
  PanelPlanOptions.GetValues(values);
  for PlanKostenType := Low(TPlanKostenType) to High(PlanKostenType) do
  begin
    PlanGewichtung[PlanKostenType] := values[Ord(PlanKostenType)];
  end;
  values.Free;

  Options.SpieltagGewichtung := PlanGewichtung[pktGameDayLength];
  Options.SpieltagOverlapp := PlanGewichtung[pktGameDayOverlap];
  Options.LastSpieltagGewichtung := PlanGewichtung[pktLastGameDayLength];
  Options.LastSpieltagOverlapp := PlanGewichtung[pktLastGameDayOverlap];
  Options.SisterTeamsAmAnfangGewichtung := PlanGewichtung[pktSisterGamesAtBegin];

  values := TList<TCalculateOptionsValues>.Create;
  PanelMannschaftOptions.GetValues(values);
  for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    Options.GewichtungMainMannschaftsKosten[i] := values[Ord(i)];
  end;
  values.Free;

  EmptyOptions.Free;
  EnableButtons();
end;





procedure TFormOptions.ListViewAusweichtermineChange(Sender: TObject;
  Item: TListItem; Change: TItemChange);
begin
  ControlChanged(nil);
end;

procedure TFormOptions.ListViewFreeDatesChange(Sender: TObject; Item: TListItem;
  Change: TItemChange);
begin
  ControlChanged(nil);
end;

procedure TFormOptions.OptionsToForm(Options: TCalculateOptions);
var
  i: TMannschaftsKostenType;
  values: TList<TCalculateOptionsValues>;
begin
  InLoad := True;
  try
    CheckBoxFreitagIs60km.Checked := Options.FreitagIsAllowedBy60km;
    ComboBoxPlanRounds.ItemIndex := Ord(Options.RoundPlaning);
    CheckBoxDoubleRound.Checked := Options.DoubleRound;
    CheckBoxDateAutomatic.Checked := Options.AutomaticMidDate;
    DateTimeMid1.DateTime := Options.getMidDate1(Plan);
    DateTimeMid2.DateTime := Options.getMidDate2(Plan);

    values := TList<TCalculateOptionsValues>.Create;
    values.Add(Options.SpieltagOverlapp);
    values.Add(Options.SpieltagGewichtung);
    values.Add(Options.LastSpieltagOverlapp);
    values.Add(Options.LastSpieltagGewichtung);
    values.Add(Options.SisterTeamsAmAnfangGewichtung);
    PanelPlanOptions.SetValues(values);
    values.Free;


    values := TList<TCalculateOptionsValues>.Create;
    for i := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      values.Add(Options.GewichtungMainMannschaftsKosten[i]);
    end;

    PanelMannschaftOptions.SetValues(values);
    values.Free;

  finally
    InLoad := False;
  end;

  EnableButtons();
end;




procedure TFormOptions.PageControlChange(Sender: TObject);
begin
  EnableButtons();
end;


procedure TFormOptions.Save;
begin
  StartWaitCursor();
  FormToOptions(Plan.getOptions());

  if Assigned(FOnAfterSave) then
  begin
    OnAfterSave(Self, True);
  end;
end;


procedure TFormOptions.SetValues(Plan: TPlan);
begin
  Self.Plan := Plan;
  OptionsToForm(Self.Plan.getOptions());
end;


procedure TFormOptions.Timer1Timer(Sender: TObject);
begin
  EnableButtons();
end;


{ TSubDialogController }

constructor TSubDialogController.Create;
begin
  inherited;
  Plan := TPlan.Create;
end;

destructor TSubDialogController.Destroy;
begin
  Plan.Free;
  inherited;
end;

procedure TSubDialogController.DoButtonClick(Sender: TObject; Index: integer);
var
  Dlg: TDialogForSingleGewichtungsOptions;
  values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Mannschaft: TMannschaft;
  KostenType: TMannschaftsKostenType;
  visibleState: TList<boolean>;
begin
  Mannschaft := Plan.Mannschaften[Index];
  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;
  Dlg.SetTitle('Gewichtung fr: ' + Mannschaft.teamName);

  for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    titles.Add(cMannschaftsKostenTypeNamen[KostenType]);
    values.Add(Plan.getOptions.GewichtungMannschaften.getDetail(Mannschaft.teamId, KostenType));
  end;

  visibleState := TList<boolean>.Create;
  for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
  begin
    if Plan.HasValuesForType(KostenType) then
    begin
      visibleState.Add(true)
    end
    else
    begin
      visibleState.Add(false)
    end;
  end;
  Dlg.Init(titles);
  Dlg.SetValues(values);
  Dlg.SetVisibleState(visibleState);

  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(values);

    for KostenType := Low(TMannschaftsKostenType) to High(TMannschaftsKostenType) do
    begin
      Plan.getOptions.GewichtungMannschaften.addDetail(Mannschaft.teamId, KostenType, values[Ord(KostenType)]);
    end;
  end;
  values.Free;
  titles.Free;
  visibleState.Free;
  Dlg.Free();
end;

function TSubDialogController.Execute: boolean;
var
  Dlg: TDialogForSingleGewichtungsOptions;
  values: TList<TCalculateOptionsValues>;
  titles: TStrings;
  Mannschaft: TMannschaft;
  i: integer;
begin
  Result := false;
  Dlg := TDialogForSingleGewichtungsOptions.Create(nil);
  Dlg.ShowButtons := True;
  Dlg.OnButtonClick := DoButtonClick;
  Dlg.SetTitle('Gewichtung der Mannschaften');
  values := TList<TCalculateOptionsValues>.Create;
  titles := TStringList.Create;

  for Mannschaft in Plan.Mannschaften do
  begin
    titles.Add(Mannschaft.teamName);
    values.Add(Plan.getOptions.GewichtungMannschaften.getMain(Mannschaft.teamId));
  end;

  Dlg.Init(titles);
  Dlg.SetValues(values);
  if mrOK = Dlg.ShowModal() then
  begin
    Dlg.GetValues(values);

    for i := 0 to Plan.Mannschaften.Count-1 do
    begin
      Mannschaft := Plan.Mannschaften[i];
      Plan.getOptions.GewichtungMannschaften.addMain(Mannschaft.teamId, values[i]);
    end;

    Result := true;

  end;
  values.Free;
  titles.Free;
  Dlg.Free();
end;

end.
