unit DialogGetUpdates;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Variants, System.Classes, Vcl.Graphics,
  Vcl.Controls, Vcl.Forms, Vcl.Dialogs, Vcl.StdCtrls, Vcl.ExtCtrls, PlanTypes, PlanUtils, System.IOUtils,
  UrlMon, System.IniFiles, ShellApi;

type
  TUpdateSearchThread = class(TThread)
  public
    HasError: boolean;
    HasUpdates: boolean;
    Url: String;
    Version: String;
  public
    constructor Create();
    procedure Execute; override;
  end;



  TFormGetUpdates = class(TForm)
    ButtonOK: TButton;
    Label1: TLabel;
    TimerUpdate: TTimer;
    LinkLabel: TLabel;
    procedure ButtonOKClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure TimerUpdateTimer(Sender: TObject);
    procedure LinkLabelClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    SearchThread: TUpdateSearchThread;
  public
  end;


implementation

{$R *.dfm}

procedure TFormGetUpdates.ButtonOKClick(Sender: TObject);
begin
  ModalResult := mrOk;
end;

procedure TFormGetUpdates.FormCreate(Sender: TObject);
begin
  FixControls(Self);
  SearchThread := TUpdateSearchThread.Create;
  SearchThread.Start;
end;

procedure TFormGetUpdates.FormDestroy(Sender: TObject);
begin
  if SearchThread <> nil then
  begin
    if not SearchThread.Finished then
    begin
      TerminateThread(SearchThread.Handle, 0);
    end;
    SearchThread.Free;
  end;
end;

procedure TFormGetUpdates.LinkLabelClick(Sender: TObject);
begin
  Shellexecute( 0, nil, pChar(LinkLabel.Caption), nil, nil, SW_SHOW);
end;

procedure TFormGetUpdates.TimerUpdateTimer(Sender: TObject);
begin
  if SearchThread <> nil then
  begin
    if SearchThread.Finished then
    begin
      TimerUpdate.Enabled := False;

      if SearchThread.HasError then
      begin
        Label1.Caption := 'Updateinformation kann nicht abgerufen werden';
      end
      else
      begin
        if SearchThread.HasUpdates then
        begin
          Label1.Caption := 'Die neue Version ' + SearchThread.Version + ' des Andigenerators kann hier runtergeladen werden:';
          LinkLabel.Caption := SearchThread.Url;
          LinkLabel.Visible := True;
        end
        else
        begin
          Label1.Caption := 'Andigenerator ist aktuell';
        end;
      end;
    end;
  end;
end;

function GetAppVersionStr: string;
var
  Exe: string;
  Size, Handle: DWORD;
  Buffer: TBytes;
  FixedPtr: PVSFixedFileInfo;
begin
  Exe := Application.ExeName;
  Size := GetFileVersionInfoSize(PChar(Exe), Handle);
  if Size = 0 then
    RaiseLastOSError;
  SetLength(Buffer, Size);
  if not GetFileVersionInfo(PChar(Exe), Handle, Size, Buffer) then
    RaiseLastOSError;
  if not VerQueryValue(Buffer, '\', Pointer(FixedPtr), Size) then
    RaiseLastOSError;
  Result := Format('%d.%d.%d.%d',
    [LongRec(FixedPtr.dwFileVersionMS).Hi,  //major
     LongRec(FixedPtr.dwFileVersionMS).Lo,  //minor
     LongRec(FixedPtr.dwFileVersionLS).Hi,  //release
     LongRec(FixedPtr.dwFileVersionLS).Lo]) //build
end;


{ TUpdateSearchThread }

constructor TUpdateSearchThread.Create;
begin
  inherited Create(true);
end;


function IsNewerVersion(Ver1, Ver2: String): Integer;

  function IsInteger(const Source: String): Boolean;
  Begin
    Result := true;
    try
      StrToInt(Source);
    except
      result := False;
    end;
  end;


  function CheckBigger(str1, str2: String): Integer;
  var
    i1, i2: Integer;
  begin
    str1 := Trim(str1);
    str2 := Trim(str2);

    if IsInteger(str1) and IsInteger(str2) then
    begin
      i1 := StrToInt(str1);
      i2 := StrToInt(str2);

      if i1 > i2 then
        result := 1
      else if i1 < i2 then
        result := -1
      else
        result := 0;
    end
    else
    begin
      result := CompareStr(str1, str2);
    end;
  end;

var
  iPos1, iPos2: Integer;
  V1, V2: String;
begin
  result := 0;

  iPos1 := Pos('.', Ver1);
  iPos2 := Pos('.', Ver2);

  while (iPos1 > 0) and (iPos2 > 0) and (result = 0) do
  begin
    V1 := Copy(Ver1, 1, iPos1 - 1);
    V2 := Copy(Ver2, 1, iPos2 - 1);

    Ver1 := Copy(Ver1, iPos1 + 1, Length(Ver1));
    Ver2 := Copy(Ver2, iPos2 + 1, Length(Ver2));

    result := CheckBigger(V1, V2);

    iPos1 := Pos('.', Ver1);
    iPos2 := Pos('.', Ver2);
  end;

  if result = 0 then
  begin
    V1 := Ver1;
    V2 := Ver2;

    if iPos1 > 0 then
      V1 := Copy(Ver1, 1, iPos1 - 1);

    if iPos2 > 0 then
      V2 := Copy(Ver2, 1, iPos2 - 1);

    result := CheckBigger(V1, V2);
  end;
end;


procedure TUpdateSearchThread.Execute;
var
  TempFileName, AppVersion: String;
  IniFile: TIniFile;
begin
  HasError := False;
  HasUpdates := False;
  TempFileName := TPath.GetTempFileName();

  if S_OK = URLDownloadToFile(nil,
                  'https://sv-schwaig.de/download/generator/updateinfo.ini',
                      PChar(TempFileName),
                    0,
                    nil) then
  begin
    IniFile := TIniFile.Create(TempFileName);

    AppVersion := GetAppVersionStr;
    Version := IniFile.ReadString('info', 'version', '');
    Url := IniFile.ReadString('info', 'url', '');

    if IsNewerVersion(AppVersion, Version) < 0 then
    begin
      HasUpdates := True;
    end;
    IniFile.Free;
  end
  else
  begin
    HasError := true;
  end;

  TFile.Delete(TempFileName);
end;


end.
