unit AndiGeneratorMain;

(*
Copyright 2017 Andreas Hofmann
Lizenz:
Der Sourcecode steht unter der GPL V3
siehe <http://www.gnu.org/licenses/>.
*)


interface

uses
  System.UITypes, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, PlanTypes, PlanUtils, StdCtrls, math, ExtCtrls, PlanPanel, PlanOptimizer,
  DialogOptions, Vcl.ComCtrls, Vcl.ToolWin, Vcl.ImgList, PrintUtil, ShellApi, System.Generics.Collections, System.Generics.Defaults,
  System.IOUtils, Vcl.Buttons, DialogSavePlan, DialogProfile,
  Vcl.Menus, DialogQuestionUseUserOptions, DialogQuestionHardErrors, DialogForSinglePanel,
  DialogPanelRanking, PlanDataObjects, DialogForMultiplePanel, DialogPanelFreeDays, DialogPanelMandatoryDays, DialogPanelMainData,
  DialogPanelTeams, DialogPanel60km, DialogPanelHomeDays, DialogPanelPredefinedGames, DialogPanelSisterTeams,
  DialogPanelAuswaertskoppel, DialogPanelLocationsCompact, DialogPanelHomeCoupleCompact, DialogFirstStart, DialogPanelHomeRight,
  DialogGetUpdates, System.Win.Registry, System.ImageList;

const
  cModificationExtension = 'modifications';


type
  TFormAndiGeneratorMain = class(TForm)
    LabelDurchlauf: TLabel;
    PanelTop: TPanel;
    PanelPlan: TPanel;
    dlgOpen: TOpenDialog;
    TimerSyncPlans: TTimer;
    LabelPlanName: TLabel;
    ToolButtonLoad: TToolButton;
    ToolButtonStartPause: TToolButton;
    ToolButtonExport: TToolButton;
    Panel2: TPanel;
    ToolButtonOptions: TToolButton;
    ToolBarMain: TToolBar;
    ImageListToolbar: TImageList;
    SaveDialog: TSaveDialog;
    LabelPlanSelector: TLabel;
    ComboBoxPlanSelector: TComboBox;
    ToolBarViewPlans: TToolBar;
    ButtonRemove: TToolButton;
    ButtonSave: TToolButton;
    ButtonProfile: TButton;
    PanelSaveToolButtons: TPanel;
    ToolButtonDivers: TToolButton;
    PopupMenuDivers: TPopupMenu;
    MenuPrint: TMenuItem;
    Anleitung1: TMenuItem;
    SaveDialogXML: TSaveDialog;
    LabelHitTest: TLabel;
    ComboBoxZoom: TComboBox;
    ToolButtonData: TToolButton;
    PopupMenuOpen: TPopupMenu;
    MenuOpen: TMenuItem;
    MenuNew: TMenuItem;
    dlgNew: TSaveDialog;
    AufUpdatesprfen1: TMenuItem;
    TimerSearchUpdate: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure TimerSyncPlansTimer(Sender: TObject);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState;
      WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure ToolButtonOptionsClick(Sender: TObject);
    procedure ToolButtonExportClick(Sender: TObject);
    procedure ToolButtonStartPauseClick(Sender: TObject);
    procedure ComboBoxPlanSelectorChange(Sender: TObject);
    procedure ButtonSaveClick(Sender: TObject);
    procedure ButtonRemoveClick(Sender: TObject);
    procedure ButtonProfileClick(Sender: TObject);
    procedure Anleitung1Click(Sender: TObject);
    procedure MenuPrintClick(Sender: TObject);
    procedure ComboBoxZoomChange(Sender: TObject);
    procedure ToolButtonDataClick(Sender: TObject);
    procedure MenuOpenClick(Sender: TObject);
    procedure MenuNewClick(Sender: TObject);
    procedure AufUpdatesprfen1Click(Sender: TObject);
    procedure TimerSearchUpdateTimer(Sender: TObject);
  private
    PlanPanel: TFormPlanPanel;
    LastKosten: MyDouble;
    LastUpdateTick: DWORD;
    FormOptions: TFormOptions;
    LastTimerTick: DWORD;
    LastSucessTick: DWORD;
    LastDurchLaufe: Int64;
    PlanIsLoaded: boolean;
    currentLoadedFileName: String;
    currentLoadedFileDate: TDateTime;

    SearchUpdateThread: TUpdateSearchThread;
    procedure EnableButtons();
    procedure KillOptimizer;
    { Private-Deklarationen }
    procedure FillPlanSelector;
    function GetCurrentViewPlan: TPlan;
    function GetCurrentPlanDirectory(PlanData: TPlanData): String;
    procedure SyncSavedPlans(ReloadAll: boolean);
    function SyncSavedPlan(FileName: String): boolean;
    function LoadSavePlan(SavedPlan: TSavedPlan; FileName: String): Boolean;
    procedure OpenFile(FileName: String; WithLoadMessages: boolean);
    procedure DoOnAfterPlanChanged(Sender: TObject; SaveOnlyOptions: boolean);
    procedure SavePlan;
    procedure InitPlanMainData;
    procedure StartTestUpdateThread;
  public
    { Public-Deklarationen }
    PlanDataClickTTFile: TPlanData;
    PlanData: TPlanData;
    PlanLoaded: TPlan;
    PlanOptimized: TPlan;
    Optimizer: TPlanOptimizer;
    CurrentPlanDirectory: String;
    SavedPlans: TObjectList<TSavedPlan>;
  end;

var
  FormAndiGeneratorMain: TFormAndiGeneratorMain;


function getMainPlan(): TPlan;
procedure DoAfterMainPlanChanged(SaveOnlyOptions: boolean);
function getCurrentViewedPlan(): TPlan;

implementation

{$R *.dfm}

procedure TFormAndiGeneratorMain.EnableButtons;
begin
  ToolButtonStartPause.Enabled := PlanIsLoaded;
  ToolButtonExport.Enabled := PlanIsLoaded;
  MenuPrint.Enabled := PlanIsLoaded;
  LabelPlanSelector.Visible := PlanIsLoaded;
  ComboBoxPlanSelector.Visible := PlanIsLoaded;
  ToolButtonOptions.Enabled := PlanIsLoaded;
  ToolButtonData.Enabled := PlanIsLoaded;

  ButtonRemove.Visible := PlanIsLoaded;
  ButtonSave.Visible := PlanIsLoaded;
  ButtonRemove.Enabled := ComboBoxPlanSelector.ItemIndex > 1;
  ButtonSave.Enabled := ComboBoxPlanSelector.ItemIndex = 0;


end;

procedure TFormAndiGeneratorMain.FormCreate(Sender: TObject);
var
  Registry: TRegistryIniFile;
  LastUpdateSearch: TDateTime;
begin
{$ifdef PROFILE}
  ButtonProfile.Visible := True;
{$endif}
{$ifdef CACHE_HIT_TEST}
  LabelHitTest.Visible := True;
{$endif}
  SavedPlans := TObjectList<TSavedPlan>.Create(True);
  PlanData := TPlanData.Create;
  PlanLoaded := TPlan.Create;
  PlanOptimized := TPlan.Create;

  PlanPanel := TFormPlanPanel.Create(Self);
  PlanPanel.setPanel(PanelPlan);

  Caption := Caption + ' ' + GetFileVersionString(Application.ExeName);

  EnableButtons();

  FixControls(Self);

  Registry := TRegistryIniFile.Create('Software\Andreas Hofmann\Andi-Generator');

  LastUpdateSearch := Registry.ReadDate('update', 'lastSearch', 0);

  if LastUpdateSearch < now - 2 then
  begin
    StartTestUpdateThread();
    Registry.WriteDate('update', 'lastSearch', now);
  end;

  Registry.Free;

end;

procedure TFormAndiGeneratorMain.StartTestUpdateThread();
begin
  SearchUpdateThread := TUpdateSearchThread.Create;
  SearchUpdateThread.Start;
end;

procedure TFormAndiGeneratorMain.FormDestroy(Sender: TObject);
begin
  PlanLoaded.Free;
  PlanOptimized.Free;

  KillOptimizer();

  SavedPlans.free;
  PlanData.Free;
  PlanDataClickTTFile.Free;

  if(SearchUpdateThread <> nil) then
  begin
    if not SearchUpdateThread.Finished then
    begin
      TerminateThread(SearchUpdateThread.Handle, 0);
    end;
    SearchUpdateThread.Free;
  end;
end;


procedure TFormAndiGeneratorMain.FormMouseWheel(Sender: TObject;
  Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint;
  var Handled: Boolean);
var
   vsb: TControlScrollbar;
   ctrl: TControl;
begin
  ctrl := FindVCLWindow(MousePos);
   //ctrl := ControlAtPos(ScreenToClient(MousePos), false, true, true);
   while (ctrl <> nil) and not (ctrl is TScrollBox) do begin
     ctrl := ctrl.Parent;
   end;
   if ctrl is TScrollBox then begin
     vsb := TScrollBox(ctrl).VertScrollBar;
     vsb.Position := vsb.Position - Sign(WheelDelta)*vsb.Increment*8;
     Handled := true;
   end;
end;

procedure TFormAndiGeneratorMain.Anleitung1Click(Sender: TObject);
var
  S: String;
begin
  S := ExtractFilePath(Application.ExeName) + '\Andigenerator.pdf';
  if not FileExists(S) then
	begin
    ShowMessage('Die Datei ' + S + ' wurde nicht gefunden');
	end
  else
	if 32 > ShellExecute(0, nil, pChar(S), nil, nil, SW_SHOW) then
	begin
    ShowMessage('Die Anleitung kann nicht angezeigt werden, evtl. ist der Acrobat Reader nicht installiert!');
	end;
end;

procedure TFormAndiGeneratorMain.AufUpdatesprfen1Click(Sender: TObject);
var
  Dlg: TFormGetUpdates;
begin
  Dlg := TFormGetUpdates.Create(Self);
  Dlg.ShowModal();
  Dlg.free;
end;

procedure TFormAndiGeneratorMain.ButtonProfileClick(Sender: TObject);
var
  Plan: TPlan;
  WasPaused: boolean;
begin
  WasPaused := False;
  if Assigned(Optimizer) then
  begin
    WasPaused := Optimizer.Paused;
    Optimizer.Paused := True;
  end;

  Plan := TPlan.Create;
  Plan.Assign(GetCurrentViewPlan());
  DialogProfile.ShowDialogProfile(Plan);
  Plan.Free;


  if Assigned(Optimizer) then
  begin
    if WasPaused then
      Optimizer.Paused := True
    else
      Optimizer.Paused := False;
  end;

end;

procedure TFormAndiGeneratorMain.ButtonRemoveClick(Sender: TObject);
var
  i: Integer;
  WasPaused: boolean;
  CurrentPlan: TPlan;
begin
  WasPaused := False;
  if Assigned(Optimizer) then
  begin
    WasPaused := Optimizer.Paused;
    Optimizer.Paused := True;
  end;

  try
    if mrYes = MessageDlg('"' + ComboBoxPlanSelector.Items[ComboBoxPlanSelector.ItemIndex] + '" lschen?', mtConfirmation, [mbYes, mbNo], 0) then
    begin
      CurrentPlan := GetCurrentViewPlan();
      for i := SavedPlans.Count-1 downto 0 do
      begin
        if CurrentPlan = SavedPlans[i].Plan then
        begin
          DeleteFile(SavedPlans[i].FileName);
        end;
      end;

      SyncSavedPlans(False);
    end;
  finally
    if Assigned(Optimizer) then
    begin
      if WasPaused then
        Optimizer.Paused := True
      else
        Optimizer.Paused := False;
    end;
  end;
end;

procedure TFormAndiGeneratorMain.ButtonSaveClick(Sender: TObject);
var
  S: String;
  FileName: String;
  WasPaused: boolean;
begin
  WasPaused := False;
  if Assigned(Optimizer) then
  begin
    WasPaused := Optimizer.Paused;
    Optimizer.Paused := True;
  end;

  try
    while true do
    begin
      if CurrentPlanDirectory = '' then
      begin
        ShowMessage('Speicherverzeichnis Appdata kann nicht ermittlt werden.');
        Exit;
      end;

      if not ShowDialogSavePlan(CurrentPlanDirectory, SavedPlans, S) then
      begin
        Exit;
      end;

      FileName := CurrentPlanDirectory + '\' + EncodeFileName(S) + '.xml';

      PlanOptimized.SaveScheduleToXML(FileName);

      SyncSavedPlans(False);


      break;
    end;
  finally
    if Assigned(Optimizer) then
    begin
      if WasPaused then
        Optimizer.Paused := True
      else
        Optimizer.Paused := False;
    end;
  end;
end;


procedure TFormAndiGeneratorMain.ComboBoxPlanSelectorChange(Sender: TObject);
begin
  PlanPanel.SetPlan(GetCurrentViewPlan);
  EnableButtons;
end;

procedure TFormAndiGeneratorMain.ComboBoxZoomChange(Sender: TObject);
var
  ZoomValue: MyDouble;
  S: String;
begin
  S := ComboBoxZoom.Items[ComboBoxZoom.ItemIndex];
  S := Copy(S, 1, Length(S)-1);
  ZoomValue := StrToInt(S) / 100.0;
  PlanPanel.SetZoom(ZoomValue);
end;


procedure TFormAndiGeneratorMain.SavePlan();
var
  DiffData: TPlanData;
  ModificationsFileName: String;
begin
  if not Assigned(PlanDataClickTTFile)  then
  begin
    // Wir haben einen komplett selbst gemachten Plan, kein Diffspeichern ntig
    PlanData.SaveToXML(currentLoadedFileName);
  end
  else
  begin
    DiffData := TPlanData.Create;
    ModificationsFileName := TPath.ChangeExtension(currentLoadedFileName, cModificationExtension);

    DiffData.Diff(PlanData, PlanDataClickTTFile);

    DiffData.SaveToXML(ModificationsFileName);
    DiffData.Free;

  end;
end;

procedure TFormAndiGeneratorMain.DoOnAfterPlanChanged(Sender: TObject; SaveOnlyOptions: boolean);
var
  I: Integer;
begin
  StartWaitCursor();

  PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');

  if not SaveOnlyOptions then
  begin
    SavePlan();
  end;

  PlanLoaded.Load(PlanData);
  PlanOptimized.AssignPlanData(PlanLoaded);
  PlanOptimized.RemoveNotValidDates();

  InitPlanMainData();
  PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');


  for I := 0 to SavedPlans.Count-1 do
  begin
    if SavedPlans[i].Valid then
    begin
      SavedPlans[i].Plan.AssignPlanData(PlanLoaded);
    end;
  end;

  if not SaveOnlyOptions then
  begin
    SyncSavedPlans(True);
  end;

  if Assigned(Optimizer) then
  begin
    Optimizer.setNewPlanData(PlanOptimized);
  end;
  PlanPanel.SetPlan(GetCurrentViewPlan);
  LastKosten := PlanOptimized.CalculateKosten(-1);
  if Assigned(FormOptions) then
  begin
    FormOptions.SetValues(PlanLoaded);
  end;
end;


procedure TFormAndiGeneratorMain.MenuNewClick(Sender: TObject);
var
  TempPlanData: TPlanData;
begin
  if dlgNew.Execute then
  begin
    // Neue Datei erzeugen
    TempPlanData := TPlanData.Create;
    TempPlanData.SaveToXML(dlgNew.FileName);
    TempPlanData.Free;

    OpenFile(dlgNew.FileName, False);

    ToolButtonDataClick(nil);

  end;
end;

procedure TFormAndiGeneratorMain.MenuOpenClick(Sender: TObject);
begin
  if dlgOpen.Execute then
  begin
    OpenFile(dlgOpen.FileName, True);
  end;
end;

procedure TFormAndiGeneratorMain.MenuPrintClick(Sender: TObject);
var
  WasPaused: boolean;
begin
  WasPaused := False;
  if Assigned(Optimizer) then
  begin
    WasPaused := Optimizer.Paused;
    Optimizer.Paused := True;
  end;

  PlanPanel.Print();

  if Assigned(Optimizer) then
  begin
    if WasPaused then
      Optimizer.Paused := True
    else
      Optimizer.Paused := False;
  end;

end;

procedure TFormAndiGeneratorMain.KillOptimizer();
begin
  if Assigned(Optimizer) then
  begin
    if Optimizer.Suspended then
    begin
      Optimizer.Suspended := False;
    end;
    Optimizer.Terminate;

    while not Optimizer.Finished do
    begin
      Sleep(2);
    end;

    FreeAndNil(Optimizer);

    ToolButtonStartPause.Caption := 'Generierung starten';
    ToolButtonStartPause.ImageIndex := 7;
  end;
end;



procedure TFormAndiGeneratorMain.TimerSearchUpdateTimer(Sender: TObject);
var
  Dlg: TFormGetUpdates;
begin
  if SearchUpdateThread <> nil then
  begin
    if SearchUpdateThread.Finished and (Application.ModalLevel = 0) then
    begin
      if SearchUpdateThread.HasUpdates then
      begin
        Dlg := TFormGetUpdates.Create(Self);
        Dlg.ShowModal();
        Dlg.free;
      end;
      FreeAndNil(SearchUpdateThread);
      TimerSearchUpdate.Enabled := False;
    end;
  end
  else
  begin
    TimerSearchUpdate.Enabled := False;
  end;
end;

procedure TFormAndiGeneratorMain.TimerSyncPlansTimer(Sender: TObject);
var
  ActDurchlauf: Int64;
  ActTick: DWORD;
  Plans, NewKosten: MyDouble;
begin
{$ifdef CACHE_HIT_TEST}
  LabelHitTest.Caption := 'Hit: ' + MyFormatInt(CacheHit) + ' Miss: ' + MyFormatInt(CacheMiss);
{$endif}
  TimerSyncPlans.Enabled := False;

  try
    SyncSavedPlans(False);
    if Assigned(Optimizer) then
    begin
      if not Optimizer.Paused then
      begin
        Optimizer.copyPlanDates(PlanOptimized);

        ActDurchlauf := Optimizer.Durchlaufe;
        ActTick := GetTickCount();

        if (LastTimerTick > 0) and (ActTick - LastTimerTick > 0) then
        begin

          //Plans := PlanPerSec(ActDurchlauf - LastDurchLaufe, ActTick - LastTimerTick);
          Plans := Optimizer.PlanPerSec();
          if Plans > 0 then
          begin
            LabelDurchlauf.Caption := MyFormatInt(ActDurchlauf) + ' Plne wurden berechnet (' + MyFormatInt(Trunc(Plans)) + ' Plne/s)';

            if (LastSucessTick > 0) and ((GetTickCount() - LastSucessTick) > 10000) then
            begin
              LabelDurchlauf.Caption := LabelDurchlauf.Caption + ', letzte Verbesserung vor: ' + MyFormatClickDiff(GetTickCount() - LastSucessTick);
            end;

  {$ifdef TRACE}
            LabelDurchlauf.Caption := LabelDurchlauf.Caption + ' Thread: ' + GetCurrentViewPlan.LastOptimizedThread;
  {$endif}
          end
          else
          begin
            LabelDurchlauf.Caption := '';
          end;

          if (LastKosten <> PlanOptimized.CalculateKosten(-1)) or (GetTickCount - LastUpdateTick > 10000 ) then
          begin
            NewKosten := PlanOptimized.CalculateKosten(-1);

            if NewKosten <> LastKosten then
            begin
              LastSucessTick := GetTickCount();
            end;

            LastKosten := NewKosten;

            PlanPanel.SetPlan(GetCurrentViewPlan);
            LastUpdateTick := GetTickCount();
          end;
        end;

        LastTimerTick := ActTick;
        LastDurchLaufe := ActDurchlauf;
      end;
    end;
  finally
    TimerSyncPlans.Enabled := True;
  end;
end;

function TFormAndiGeneratorMain.GetCurrentViewPlan: TPlan;
begin
  if ComboBoxPlanSelector.ItemIndex < 0 then
  begin
    Result := nil;
  end
  else
  if ComboBoxPlanSelector.ItemIndex = 0 then
  begin
    Result := PlanOptimized;
  end
  else
  if ComboBoxPlanSelector.ItemIndex = 1 then
  begin
    Result := PlanLoaded;
  end
  else
  begin
    result := (ComboBoxPlanSelector.Items.Objects[ComboBoxPlanSelector.ItemIndex] as TPlan);
  end;
end;

procedure TFormAndiGeneratorMain.ToolButtonDataClick(Sender: TObject);
var
  Form: TDialogForMultiplePanel;
  WasPaused: boolean;
begin
  WasPaused := False;
  if Assigned(Optimizer) then
  begin
    WasPaused := Optimizer.Paused;
    Optimizer.Paused := True;
  end;

  Form := TDialogForMultiplePanel.Create(Self);
  Form.SetValues(PlanData, PlanDataClickTTFile, [TDialogPanelMainData, TDialogPanelTeams,
                                                  TDialogPanelRanking, TDialogPanelLocationsCompact,
                                                  TDialogPanelHomeCoupleCompact, TDialogPanelAuswaertskoppel,
                                                  TDialogPanelHomeDays, TDialogPanelSisterTeams,
                                                  TDialogPanel60km, TDialogPanelFreeDays,
                                                  TDialogPanelMandatoryDays, TDialogPanelPredefinedGames,
                                                  TDialogPanelHomeRight]);


  (*
  Form.SetValues(PlanData, PlanDataClickTTFile, [TDialogPanelMainData, TDialogPanelTeams,
                                                  TDialogPanelHomeDays, TDialogPanel60km,
                                                  TDialogPanelSisterTeams, TDialogPanelAuswaertskoppel,
                                                  TDialogPanelLocationsCompact, TDialogPanelHomeCoupleCompact,
                                                  TDialogPanelPredefinedGames, TDialogPanelFreeDays,
                                                  TDialogPanelMandatoryDays, TDialogPanelRanking]);
  *)

  if mrOk = Form.ShowModal then
  begin
    PlanData.Assign(Form.getPlanData());
    DoOnAfterPlanChanged(nil, false);
  end;
  Form.Free;

  if Assigned(Optimizer) then
  begin
    if WasPaused then
      Optimizer.Paused := True
    else
      Optimizer.Paused := False;
  end;

end;

procedure TFormAndiGeneratorMain.ToolButtonExportClick(Sender: TObject);
var
  TempPlan: TPlan;
  Messages: TStrings;
begin
  if GetCurrentViewPlan <> nil then
  begin
    TempPlan := TPlan.Create;
    Messages := TStringList.Create;
    try
      TempPlan.Assign(GetCurrentViewPlan);

      TempPlan.getHardErrorMessages(Messages);

      if Messages.Count > 0 then
      begin
        if not DialogHardErrors(Messages) then
        begin
          Exit;
        end;
      end;

      SaveDialog.FileName := MakeValidFileName(TempPlan.PlanName) + '.csv';
      if SaveDialog.Execute() then
      begin
        TempPlan.SaveScheduleToCsv(SaveDialog.FileName);
      end;
    finally
      TempPlan.Free;
      Messages.Free;
    end;
  end;
end;


procedure TFormAndiGeneratorMain.OpenFile(FileName: String; WithLoadMessages: boolean);
var
  Messages: TStrings;
  DiffData: TPlanData;
  ModificationsFileName: String;
  FirstStart: boolean;
  FirstStartDlg: TDialogFirstStart;
  Year, Month, Day: Word;
begin
  currentLoadedFileName := FileName;
  FileAge(currentLoadedFileName, currentLoadedFileDate);
  StartWaitCursor();
  KillOptimizer();
  SavedPlans.Clear;
  FirstStart := False;

  FreeAndNil(PlanDataClickTTFile);
  if IsClickTTFile(FileName) then
  begin
    PlanDataClickTTFile := TPlanData.Create;
    PlanDataClickTTFile.LoadFromClickTTFile(FileName);
    PlanData.Assign(PlanDataClickTTFile);

    ModificationsFileName := TPath.ChangeExtension(currentLoadedFileName, cModificationExtension);
    if FileExists(ModificationsFileName) then
    begin
      DiffData := TPlanData.Create;
      DiffData.LoadFromXML(ModificationsFileName);
      PlanData.Merge(DiffData);
      DiffData.Free;
    end
    else
    begin
      FirstStart := True;
    end;
  end
  else
  begin
    PlanData.LoadFromXML(FileName);
  end;

{$ifopt D+}
  PlanData.SaveToXML('D:\temp\andigenerator.xml');
{$endif}

  PlanLoaded.Load(PlanData);
  CurrentPlanDirectory := GetCurrentPlanDirectory(PlanData);

  PlanLoaded.getOptions().Load(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');

  Messages := TStringList.Create;
  PlanLoaded.getOptions().getDiffToDefault(PlanLoaded, Messages);
  if WithLoadMessages then
  begin
    if Messages.Count > 0 then
    begin
      if not DialogUseOptionsQuestion(Messages) then
      begin
        PlanLoaded.getOptions().Clear;
        PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');
      end;
    end;
  end;

  if WithLoadMessages then
  begin

    if 6 * 30 > Abs(PlanLoaded.DateBegin - PlanLoaded.DateEnd) then
    begin
      if PlanLoaded.getOptions().RoundPlaning <> rpHalfRound then
      begin
        // Weniger als 6 Monate ist die gesamte Runde -> ist wahrscheinlich eine Halbrunde
        if idYes = MessageDlg('Die Runde ist krzer als 6 Monate. Soll nur eine Halbrunde generiert werden?', mtConfirmation, [mbYes, mbNo], 0) then
        begin
          PlanLoaded.getOptions().RoundPlaning := rpHalfRound;
          PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');
        end;
      end;
    end
    else
    // Wahrscheinlich soll nur die Rckrunde generiert werden
    if 60 > Abs(now - PlanLoaded.DateBeginRueckrundeInXML) then
    begin
      DecodeDate(PlanLoaded.DateBegin, year, month, day);
      if (not (PlanLoaded.getOptions.RoundPlaning in [rpCorona, rpSecondOnly])) and (year = 2020)  then
      begin
        if idYes = MessageDlg('Die Vorrunde liegt in der Vergangenheit. Soll nur die Rckrunde mit den Coronaspezialitten generiert werden?', mtConfirmation, [mbYes, mbNo], 0) then
        begin
          PlanLoaded.getOptions().RoundPlaning := rpCorona;
          PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');
        end;
      end;

      if (not (PlanLoaded.getOptions.RoundPlaning in[rpCorona, rpSecondOnly])) then
      begin
        if idYes = MessageDlg('Die Vorrunde liegt in der Vergangenheit. Soll nur die Rckrunde generiert werden?', mtConfirmation, [mbYes, mbNo], 0) then
        begin
          PlanLoaded.getOptions().RoundPlaning := rpSecondOnly;
          PlanLoaded.getOptions().Save(PlanLoaded, CurrentPlanDirectory + '\AndiGenerator.options');
        end;
      end;
    end;
  end;


  if WithLoadMessages and FirstStart then
  begin
    FirstStartDlg := TDialogFirstStart.Create(Self);
    FirstStartDlg.SetValue(PlanData, PlanDataClickTTFile, PlanLoaded);
    FirstStartDlg.ShowModal;
    FirstStartDlg.Free;
  end;



  Messages.Free;

  PlanLoaded.AssignOptions(PlanLoaded.getOptions());

  PlanOptimized.Assign(PlanLoaded);
  PlanOptimized.clearAllDates;
  PlanOptimized.RemoveNotValidDates();

  PlanIsLoaded := true;

  InitPlanMainData();

  PlanPanel.pgcMain.ActivePageIndex := 0;

  ComboBoxPlanSelector.ItemIndex := 1;
  ComboBoxPlanSelectorChange(nil);

end;

procedure TFormAndiGeneratorMain.InitPlanMainData();
var
  i: integer;
  OldIndex: integer;
begin

  CurrentPlanDirectory := GetCurrentPlanDirectory(PlanData);

  LabelPlanName.Caption := PlanLoaded.PlanName;
  LabelPlanName.Visible := True;

  OldIndex := ComboBoxPlanSelector.ItemIndex;

  FillPlanSelector();

  SyncSavedPlans(False);

  if OldIndex < ComboBoxPlanSelector.Items.Count then
  begin
    ComboBoxPlanSelector.ItemIndex := OldIndex;
  end
  else
  begin
    ComboBoxPlanSelector.ItemIndex := 0;
  end;
  ComboBoxPlanSelectorChange(nil);

  if Assigned(FormOptions) then
  begin
    FormOptions.SetValues(PlanLoaded);
  end;

  for i := 0 to SavedPlans.Count-1 do
  begin
    if SavedPlans[i].Valid then
    begin
      SavedPlans[i].Plan.AssignOptions(PlanOptimized.getOptions());
    end;
  end;

  EnableButtons();
end;



function TFormAndiGeneratorMain.GetCurrentPlanDirectory(PlanData: TPlanData): String;
var
  year, month, day: Word;
  DateBegin: TDateTime;
  PlanName: String;
begin
  try
    DateBegin := PlanData.root.GetAsDate(aFrom);
    PlanName := PlanData.root.GetAsString(aName);
    DecodeDate(DateBegin, year, month, day);
    Result := getAppLocalPath() + '\Andi-Generator\' + EncodeFileName(PlanName + ' ' + IntToStr(Year));
    ForceDirectories(Result);
  except
    Result := '';
  end;
end;

function TFormAndiGeneratorMain.LoadSavePlan(SavedPlan: TSavedPlan; FileName: String): Boolean;
begin
  Result := False;
  SavedPlan.FileName := FileName;
  SavedPlan.Name := DecodeFileName(TPath.GetFileNameWithoutExtension(FileName));
  FileAge(FileName, SavedPlan.FileAge);
  try
    SavedPlan.Plan.Assign(PlanLoaded);
    SavedPlan.Plan.LoadScheduleFromXml(FileName);
    SavedPlan.Valid := True;
    Result := True;
  Except
    SavedPlan.Valid := False;
  end;
end;


function TFormAndiGeneratorMain.SyncSavedPlan(FileName: String): Boolean;
var
  I: Integer;
  SavedPlan: TSavedPlan;
  FileDate: TDateTime;
begin
  Result := False;
  for I := 0 to SavedPlans.Count-1 do
  begin
    if FileName = SavedPlans[i].FileName then
    begin
      FileAge(FileName, FileDate);
      if FileDate <> SavedPlans[i].FileAge then
      begin
        Result := LoadSavePlan(SavedPlans[i], FileName);
        Exit;
      end
      else
      begin
        Exit;
      end;
    end;
  end;

  SavedPlan := TSavedPlan.Create;
  Result := LoadSavePlan(SavedPlan, FileName);
  SavedPlans.Add(SavedPlan);
end;

type
  TSavedPlanComparer = class(TComparer<TSavedPlan>)
    function Compare(const Left, Right: TSavedPlan): Integer; override;
  end;

function TSavedPlanComparer.Compare(const Left, Right: TSavedPlan): Integer;
begin
  Result := -CompareValue(Left.FileAge, right.FileAge);
end;


procedure TFormAndiGeneratorMain.SyncSavedPlans(ReloadAll: boolean);
var
  FileNames: TStrings;
  FileName: String;
  i: Integer;
  SomeThingChanged: boolean;
  Comparer: TSavedPlanComparer;
begin
  if ReloadAll then
  begin
    for i := 0 to SavedPlans.Count-1 do
    begin
      SavedPlans[i].FileAge := 0;
    end;
  end;

  SomeThingChanged := False;
  if CurrentPlanDirectory <> '' then
  begin
    FileNames := TStringList.Create;
    FilesFromDir(FileNames, CurrentPlanDirectory + '\*.xml');

    // Falls etwas gelscht wurde rausnehmen
    for i := SavedPlans.Count-1 downto 0 do
    begin
      if 0 > FileNames.IndexOf(ExtractFileName(SavedPlans[i].FileName)) then
      begin
        SomeThingChanged := True;
        SavedPlans.Delete(i);
      end;
    end;


    for i := 0 to FileNames.Count-1 do
    begin
      FileName := CurrentPlanDirectory + '\' + FileNames[i];
      if SyncSavedPlan(FileName) then
      begin
        SomeThingChanged := True;
      end;
    end;
    FileNames.Free;
  end;

  if SomeThingChanged then
  begin
    Comparer := TSavedPlanComparer.Create;
    SavedPlans.Sort(Comparer);
    Comparer.Free;
    FillPlanSelector();
    PlanPanel.SetPlan(GetCurrentViewPlan);
  end;
end;

procedure TFormAndiGeneratorMain.FillPlanSelector();
var
  i: Integer;
  OldIndex: Integer;
  OldSelection: String;
begin
  OldIndex := ComboBoxPlanSelector.ItemIndex;
  if ComboBoxPlanSelector.ItemIndex >= 0 then
  begin
    OldSelection := ComboBoxPlanSelector.Items[ComboBoxPlanSelector.ItemIndex];
  end
  else
  begin
    OldSelection := '';
  end;

  ComboBoxPlanSelector.Items.Clear;
  ComboBoxPlanSelector.Items.Add('laufende Generierung');
  ComboBoxPlanSelector.Items.Add('in Click-TT vorhandener Plan');

  for i := 0 to SavedPlans.Count-1 do
  begin
    if SavedPlans[i].Valid then
    begin
      ComboBoxPlanSelector.Items.AddObject(SavedPlans[i].Name, SavedPlans[i].Plan);
    end;
  end;

  if OldIndex >= 0 then
  begin
    if ComboBoxPlanSelector.Items.IndexOf(OldSelection) >= 0 then
    begin
      ComboBoxPlanSelector.ItemIndex := ComboBoxPlanSelector.Items.IndexOf(OldSelection);
    end
    else
    begin
      ComboBoxPlanSelector.ItemIndex := min(OldIndex, ComboBoxPlanSelector.Items.Count-1);
    end;

    if ComboBoxPlanSelector.ItemIndex <> OldIndex then
    begin
      PlanPanel.SetPlan(GetCurrentViewPlan);
      EnableButtons;
    end;
  end;
end;

procedure TFormAndiGeneratorMain.ToolButtonOptionsClick(Sender: TObject);
begin
  if not Assigned(FormOptions) then
  begin
    FormOptions := TFormOptions.Create(Self);
    FormOptions.OnAfterSave := DoOnAfterPlanChanged;
    FormOptions.SetValues(PlanLoaded);
  end;
  FormOptions.Visible := True;
end;

procedure TFormAndiGeneratorMain.ToolButtonStartPauseClick(Sender: TObject);
begin
{$ifdef CACHE_HIT_TEST}
  CacheHit := 0;
  CacheMiss := 0;
{$endif}
{$ifdef TRACE}
  TraceString('Beginn StartClick');
{$endif}
  StartWaitCursor();
  if not Assigned(Optimizer) then
  begin
    ToolButtonStartPause.Caption := 'Generierung pausieren';
    ToolButtonStartPause.ImageIndex := 6;

    Optimizer := TPlanOptimizer.Create();
    Optimizer.Plan.Assign(PlanOptimized);
    Optimizer.Plan.RemoveNotValidDates();

    Optimizer.Start;
    LabelDurchlauf.Caption := '';
    LabelDurchlauf.Visible := True;
    LastKosten := -1;
    LastTimerTick := 0;
    LastSucessTick := 0;
    PlanPanel.pgcMain.ActivePageIndex := 2;

    ComboBoxPlanSelector.ItemIndex := 0;
    ComboBoxPlanSelectorChange(nil);

  end
  else
  begin
    Optimizer.Paused := not Optimizer.Paused;
    if Optimizer.Paused then
    begin
      ToolButtonStartPause.Caption := 'Generierung fortsetzen';
      ToolButtonStartPause.ImageIndex := 7;
    end
    else
    begin
      ToolButtonStartPause.Caption := 'Generierung pausieren';
      ToolButtonStartPause.ImageIndex := 6;
    end;
  end;

  EnableButtons();
{$ifdef TRACE}
  TraceString('Ende StartClick');
{$endif}

end;

function getMainPlan(): TPlan;
begin
  Result := FormAndiGeneratorMain.PlanLoaded;
end;

procedure DoAfterMainPlanChanged(SaveOnlyOptions: boolean);
begin
  FormAndiGeneratorMain.DoOnAfterPlanChanged(nil, SaveOnlyOptions);
end;

function getCurrentViewedPlan(): TPlan;
begin
  Result := FormAndiGeneratorMain.GetCurrentViewPlan;
end;


initialization
end.
